/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System.Collections.Generic;
using System;
using TPClib.Common;

namespace TPClib.Image
{
	/// <summary>
	/// DICOM specific header information
	/// </summary>
	public class DicomHeader : ImageHeader, IComparable<DicomHeader>
	{
		/// <summary>
		/// Default orientation for Dicom files
		/// </summary>
		public static readonly Orientation DefaultDicomOrientation = Orientation.Radiological;

		/// <summary>
		/// Patient orientation, tag(0018,5100). Default == Feet First Supine
		/// </summary>
		public string dicomOrientation = "FFS";

		/// <summary>
		/// The center positions of upper left hand corner voxel of slices, tag(0020,0032). 
		/// The length of the array is the same as number of slices in the image.
		/// <seealso href="ftp://medical.nema.org/medical/dicom/final/cp212_ft.pdf"/>
		/// </summary>
		public Point[] imagepositions = new Point[] { new Point() };

		/// <summary>
		/// Image orientation, tag(0020,0037). Six values represent direction cosines for direction 
		/// angles of the first row, the first column of slice with respect to the patient orientation.
		/// {cosines for x, cosines for y, cosines for z}.
		/// 
		/// The direction of the axes is defined fully by the patient's orientation. 
		/// The x-axis is increasing to the left hand side of the patient. 
		/// The y-axis is increasing to the posterior side of the patient. 
		/// The z-axis is increasing toward the head of the patient.
		/// </summary>
		public float[] ImageOrientations
		{
			set
			{
				if (value.Length != 6) throw new TPCInvalidArgumentsException("Must have exactly 6 orientation values.");
				imageorientations = value;
			}
			get
			{
				return imageorientations;
			}
		}

		/// <summary>
		/// Image orientation, tag(0020,0037). First three define the x-unit vector, last three the y-unit vector
		/// </summary>
		private float[] imageorientations = new float[] { 1.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f };

		private const string defaultStudyUID = "1.2.246.0.0.0";

		/// <summary>
		/// Image instance number
		/// </summary>
		public int instance_nr = 0;

		/// <summary>
		/// Study instance UID in tag (0020,000D). Default = instance UID starting with finland (1.2.246)
		/// </summary>
		public string StudyInstanceUID
		{
			get
			{
				if (this.Contains("StudyInstanceUID"))
				{
					return this["StudyInstanceUID"].ToString();
				}
				else return defaultStudyUID;
			}
			set
			{
				this["StudyInstanceUID"] = value.ToString();
			}
		}

		/// <summary>
		/// Series instance UID in tag (0020,000E). Use the Instance Creator UID as base.
		/// </summary>
		public string SeriesInstanceUID
		{
			get
			{
				if (this.Contains("SeriesInstanceUID"))
				{
					return this["SeriesInstanceUID"].ToString();
				}
				else return this.StudyInstanceUID + '.' + this.series_nr;
			}
			set
			{
				this["SeriesInstanceUID"] = value.ToString();
			}
		}

		/// <summary>
		/// Series instance UID in tag (0020,000E). Use the Instance Creator UID as base.
		/// </summary>
		public string SOPInstanceUID
		{
			get
			{
				if (this.Contains("SOPInstanceUID"))
				{
					return this["SOPInstanceUID"].ToString();
				}
				else return this.SeriesInstanceUID + '.' + this.instance_nr;
			}
			set
			{
				this["SOPInstanceUID"] = value.ToString();
			}
		}

		/// <summary>
		/// Slice start time in milliseconds. Used for frame sorting.
		/// </summary>
		public int slice_start = 0;

		/// <summary>
		/// Slice location in z-axis (planes). Used for z-orientation determination when reading data. 
		/// This read-only field is the same as location position.
		/// <see cref="imagepositions"/>
		/// </summary>
		public float SliceLocation
		{
			get
			{
				return (float)imagepositions[0].Z;
			}
		}

		/// <summary>
		/// Gate number (1-based). Used fot gate sorting.
		/// </summary>
		public int gate_number = 0;

		/// <summary>
		/// Series number. Used to group one image set.
		/// </summary>
		public int series_nr = 0;

		/// <summary>
		/// Scale intercept in field (0x0028, 0x1052). This field is used in read and write to scale the intensity 
		/// values to true values. (default == 0.0)
		/// </summary>
		public float scaleintercept = 0.0f;

		/// <summary>
		/// Scale factor in field (0x0028, 0x1053). This field is used in read and write to scale the intensity 
		/// values to true values. (default == 0.0)
		/// </summary>
		public float scalefactor = 1.0f;

		/// <summary>
		/// Signed/unsigned tacs
		/// </summary>
		public bool signed = false;

		/// <summary>
		/// Bytes of tacs in file per single pixel value
		/// </summary>
		public int bytesperpixel = 0;

		/// <summary>
		/// Number of data samples are used to define a pixel value. 
		/// Default == 1. This value can be 3 for RBB values
		/// </summary>
		public int samplesperpixel = 1;

		/// <summary>
		/// Frame start time.
		/// </summary>
		public double frame_start_time = 0.0;

		/// <summary>
		/// Frame duration.
		/// </summary>
		public double frame_duration = 0.0;

		/// <summary>
		/// Stream position for tacs tag
		/// </summary>
		public long stream_data_position = 0;

		/// <summary>
		/// Byte ordering for the pixel data
		/// </summary>
		public TPCDicom.Encoding.TransferSyntax transfersyntax = TPCDicom.Encoding.TransferSyntax.Default;

		/// <summary>
		/// Default constructor
		/// </summary>
		public DicomHeader()
		{
			Orientation = DefaultDicomOrientation;
		}

		/// <summary>
		/// Copy constructor. This is used because C# does not clearly state whether Clone in 
		/// ICloneable intercace should perform deep or shallow copy.
		/// </summary>
		/// <param name="h">header that has copied data</param>
		public DicomHeader(ImageHeader h)
			: base(h)
		{
			if (h is DicomHeader)
			{
				DicomHeader hdr = h as DicomHeader;
				this.bytesperpixel = hdr.bytesperpixel;
				this.frame_duration = hdr.frame_duration;
				this.frame_start_time = hdr.frame_start_time;
				this.gate_number = hdr.gate_number;
				this.imageorientations = hdr.imageorientations;
				this.imagepositions = new Point[hdr.imagepositions.Length];
				for (int i = 0; i < imagepositions.Length; i++)
				{
					imagepositions[i] = new Point(hdr.imagepositions[i]);
				}
				this.instance_nr = hdr.instance_nr;
				this.dicomOrientation = hdr.dicomOrientation;
				this.scalefactor = hdr.scalefactor;
				this.scaleintercept = hdr.scaleintercept;
				this.series_nr = hdr.series_nr;
				this.signed = hdr.signed;
				this.slice_start = hdr.slice_start;
				this.stream_data_position = hdr.stream_data_position;
				this.transfersyntax = hdr.transfersyntax;
				this.SeriesInstanceUID = hdr.SeriesInstanceUID;
				this.StudyInstanceUID = hdr.StudyInstanceUID;
			}
			else
			{
				this.imagepositions = new Point[h.Planes];
				double zPos = -(h.Siz.SizeZ / 2.0);
				for (int i = 0; i < imagepositions.Length; i++)
				{
					imagepositions[i] = new Point(-h.SizeX / 2.0f, -h.SizeX / 2.0f, zPos);
					zPos += h.SizeZ;
				}
			}
		}

		/// <summary>
		/// Returns a String that represents the current Object.
		/// </summary>
		/// <returns>A String that represents the current Object.</returns>
		public override string ToString()
		{
			return "DicomHeader[instance=" + instance_nr + " scale=" + scalefactor + " z_location=" + SliceLocation + " width=" + Width + " height=" + Height + " planes=" + Planes + " frames=" + Frames + " pos=" + stream_data_position + "]";
		}

		/// <summary>
		/// Comparison operator
		/// </summary>
		/// <param name="other">Header to compare with</param>
		/// <returns>-1, if this header precedes the other, 0 if equal, 1 if the other precedes this</returns>
		public int CompareTo(DicomHeader other)
		{
			//sort null objects as 'smaller' than existing ones
			if (this == null)
			{
				if (other == null) return 0;
				else return 1;
			}
			else if (other == null) return -1;
			//group with series number
			if (this.series_nr < other.series_nr) return -1;
			if (this.series_nr > other.series_nr) return 1;
			//use gate numbers if available
			if (this.gate_number < other.gate_number) return -1;
			if (this.gate_number > other.gate_number) return 1;

			// if dicomheader defines data as dynamic, sort according to frame times
			// static images must not be sorted by frame time
			if (this.IsDynamic && other.IsDynamic)
			{
				if (this.frame_start_time < other.frame_start_time) return -1;
				if (this.frame_start_time > other.frame_start_time) return 1;
				if (this.slice_start < other.slice_start) return -1;
				if (this.slice_start > other.slice_start) return 1;
			}

			if (this.SliceLocation < other.SliceLocation) return -1;
			if (this.SliceLocation > other.SliceLocation) return 1;

			if (this.instance_nr < other.instance_nr) return -1;
			if (this.instance_nr > other.instance_nr) return 1;

			//couldn't find any sorting difference
			return 0;
		}
	}
}
