/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Image
{
	public class Ecat7Header : ImageHeader
	{
		/// <summary>
		/// Default orientation.
		/// </summary>
		public static readonly Orientation DefaultOrientation = Orientation.Neurological;

		/// <summary>
		/// Default constructor.
		/// </summary>
		public Ecat7Header()
			: base()
		{
			Init();
		}

		/// <summary>
		/// Create an Ecat7 header from an image header
		/// </summary>
		/// <param name="hdr">Image header</param>
		public Ecat7Header(ImageHeader hdr)
			: base(hdr)
		{
			if (hdr is Ecat7Header)
			{
				Ecat7Header h = hdr as Ecat7Header;
				this.acquisition_mode = h.acquisition_mode;
				this.acquisition_type = h.acquisition_type;
				this.angular_compression = h.angular_compression;
				this.axial_samp_mode = h.axial_samp_mode;
				this.bed_elevation = h.bed_elevation;
				this.bed_position = h.bed_position;
				this.bin_size = h.bin_size;
				this.branching_fraction = h.branching_fraction;
				this.calibration_units = h.calibration_units;
				this.calibration_units_label = h.calibration_units_label;
				this.coin_samp_mode = h.coin_samp_mode;
				this.compression_code = h.compression_code;
				this.data_units = h.data_units;
				this.distance_scanned = h.distance_scanned;
				this.dosage = h.dosage;
				this.dose_start_time = h.dose_start_time;
				this.ecat_calibration_factor = h.ecat_calibration_factor;
				this.facility_name = h.facility_name;
				this.file_type = h.file_type;
				this.fill = h.fill;
				this.gantry_rotation = h.gantry_rotation;
				this.gantry_tilt = h.gantry_tilt;
				this.init_bed_position = h.init_bed_position;
				this.intrinsic_tilt = h.intrinsic_tilt;
				this.isotope_halflife = h.isotope_halflife;
				this.isotope_name = h.isotope_name;
				this.lwr_sctr_thres = h.lwr_sctr_thres;
				this.lwr_true_thres = h.lwr_true_thres;
				this.magic_number = h.magic_number;
				this.num_bed_pos = h.num_bed_pos;
				this.num_frames = h.num_frames;
				this.num_gates = h.num_gates;
				this.num_planes = h.num_planes;
				this.operator_name = h.operator_name;
				this.original_file_name = h.original_file_name;
				this.patient_age = h.patient_age;
				this.patient_birth_date = h.patient_birth_date;
				this.patient_dexterity = h.patient_dexterity;
				this.patient_height = h.patient_height;
				this.patient_id = h.patient_id;
				this.patient_name = h.patient_name;
				this.patient_orientation = h.patient_orientation;
				this.patient_sex = h.patient_sex;
				this.patient_weight = h.patient_weight;
				this.physician_name = h.physician_name;
				this.plane_separation = h.plane_separation;
				this.radiopharmaceutical = h.radiopharmaceutical;
				this.scan_start_time = h.scan_start_time;
				this.septa_state = h.septa_state;
				this.serial_number = h.serial_number;
				this.study_description = h.study_description;
				this.study_type = h.study_type;
				this.sw_version = h.sw_version;
				this.system_type = h.system_type;
				this.transaxial_fov = h.transaxial_fov;
				this.transm_source_type = h.transm_source_type;
				this.upr_true_thres = h.upr_true_thres;
				this.user_process_code = h.user_process_code;
				this.well_counter_corr_factor = h.well_counter_corr_factor;
				this.wobble_speed = h.wobble_speed;
			}
			else Init();
		}

		private void Init()
		{
			"MATRIX70v".CopyTo(0, magic_number, 0, 9);
			"Unknown".CopyTo(0, study_type, 0, 7);
			file_type = datatype_e.ecat7_UNKNOWN;
			sw_version = 70;
			ecat_calibration_factor = 1.0f;
			calibration_units = 2;
			"<no original filename>".CopyTo(0, original_file_name, 0, 22);
			Orientation = DefaultOrientation;
		}

		/// <summary>
		/// File dataunit
		/// </summary>
		public override DataUnit Dataunit
		{
			get
			{
				return DataUnit.Parse(new string(data_units).Trim(new char[] { ' ', '\0' }));
			}
			set
			{
				string unit = value.ToString();
				data_units = unit.ToCharArray(0, 32 < unit.Length ? 32 : unit.Length);
				this["data_units"] = data_units;
			}
		}

		/// <summary>
		/// Study description
		/// </summary>
		public override string Description
		{
			get
			{
				return RemoveControlCharacters(new string(study_description));
			}
			set
			{
				study_description = value.ToCharArray(0, 32 < value.Length ? 32 : value.Length);
				this["study_description"] = study_description;
			}
		}

		/// <summary>
		/// Injection time
		/// </summary>
		public override DateTime DoseStartTime
		{
			get
			{
				return Ecat7File.EcatToDateTime(dose_start_time);
			}
			set
			{
				dose_start_time = Ecat7File.DateTimeToEcatSeconds(value);
				this["dose_start_time"] = dose_start_time;
			}
		}

		/// <summary>
		/// Injected dose
		/// </summary>
		public override float InjectedDose
		{
			get
			{
				return dosage;
			}
			set
			{
				dosage = value;
				this["dosage"] = dosage;
			}
		}

		/// <summary>
		/// Dynamic header data
		/// </summary>
		public override bool IsDynamic
		{
			get
			{
				return acquisition_type == Acquisition_type_e.Dynamic_emission;
			}
			set
			{
				acquisition_type = Acquisition_type_e.Dynamic_emission;
				this["acquisition_type"] = acquisition_type;
			}
		}

		/// <summary>
		/// Isotope
		/// </summary>
		public override Isotope_enumerator Isotope
		{
			get
			{
				return TPClib.Isotope.CreateIsotope(new string(isotope_name).Trim(new char[] { ' ', '\0' }));
			}
			set
			{
				isotope_name = value.ToString().ToCharArray();
				isotope_halflife = (float)TPClib.Isotope.GetHalflife(value);
				this["isotope_name"] = isotope_name;
				this["isotope_halflife"] = isotope_halflife;
			}
		}

		/// <summary>
		/// Image modality
		/// </summary>
		public override ImageModality Modality
		{
			get
			{
				return ImageModality.Unknown;
			}
			set { }
		}

		/// <summary>
		/// Patient id
		/// </summary>
		public override string PatientID
		{
			get
			{
				return new string(patient_id).Trim(new char[] { ' ', '\0' });
			}
			set
			{
				patient_id = value.ToCharArray(0, 16 < value.Length ? 16 : value.Length);
				this["patient_id"] = patient_id;
			}
		}

		/// <summary>
		/// Patient name
		/// </summary>
		public override PersonName PatientName
		{
			get
			{
				string name = new string(patient_name).Trim(new char[] { ' ', '\0' });
				string[] names = name.Split(new char[] { ',', ' ' }, 2, StringSplitOptions.RemoveEmptyEntries);
				return new PersonName(names);
			}
			set
			{
				string name = value.LastName + ", " + value.FirstNames;
				patient_name = name.ToCharArray(0, 32 < name.Length ? 32 : name.Length);
				this["patient_name"] = patient_name;
			}
		}

		/// <summary>
		/// Radiopharma used
		/// </summary>
		public override string Radiopharma
		{
			get
			{
				return RemoveControlCharacters(new string(radiopharmaceutical));
			}
			set
			{
				radiopharmaceutical = value.ToCharArray(0, 32 < value.Length ? 32 : value.Length);
				this["radiopharmaceutical"] = radiopharmaceutical;
			}
		}

		/// <summary>
		/// Unix file type indentification number
		/// </summary>
		public char[] magic_number = new char[14];
		/// <summary>
		/// Scan file's creation number
		/// </summary>
		public char[] original_file_name = new char[32];
		/// <summary>
		/// Software version number 
		/// </summary>
		public short sw_version;
		/// <summary>
		/// Scanner model 
		/// </summary>
		public short system_type;
		/// <summary>
		/// Matrix file type (short)
		/// </summary>
		public datatype_e file_type;
		/// <summary>
		/// Serial number of the gantry 
		/// </summary>
		public char[] serial_number = new char[10];
		/// <summary>
		/// Date and time when acquisition was started (sec from base time) 
		/// </summary>
		public int scan_start_time;
		/// <summary>
		/// String representation of the isotope 
		/// </summary>
		public char[] isotope_name = new char[8];
		/// <summary>
		/// Half-life of isotope (sec) 
		/// </summary>
		public float isotope_halflife;
		/// <summary>
		/// String representation of the tracer name 
		/// </summary>
		public char[] radiopharmaceutical = new char[32];
		/// <summary>
		/// Angle (degrees) 
		/// </summary>
		public float gantry_tilt;
		/// <summary>
		/// Angle (degrees) 
		/// </summary>
		public float gantry_rotation;
		/// <summary>
		/// Bed height from lowest point (cm) 
		/// </summary>
		public float bed_elevation;
		/// <summary>
		/// Angle that the first detector of Bucket 0 is offset from top center (in degrees) 
		/// </summary>
		public float intrinsic_tilt;
		/// <summary>
		/// Revolutions/minute (0 if not wobbled) 
		/// </summary>
		public short wobble_speed;
		/// <summary>
		/// Enumerated type (SRC_NONE, _RRING, _RING,_ROD, _RROD) (short)
		/// </summary>
		public Transm_source_type_e transm_source_type;
		/// <summary>
		/// Total distance scanned (cm) 
		/// </summary>
		public float distance_scanned;
		/// <summary>
		/// Diameter of transaxial view (cm) 
		/// </summary>
		public float transaxial_fov;
		/// <summary>
		/// 0=no mash, 1=mash of 2, 2=mash of 4 
		/// </summary>
		public short angular_compression;
		/// <summary>
		/// 0=Net trues, 1=Prompts and Delayed, 3=Prompts, Delayed, and Multiples 
		/// </summary>
		public short coin_samp_mode;
		/// <summary>
		/// 0=Normal, 1=2X, 2=3X 
		/// </summary>
		public short axial_samp_mode;
		/// <summary>
		/// Ecat calibration factor. Used to convet ECAT counts to activity counts.
		/// </summary>
		public float ecat_calibration_factor;
		/// <summary>
		/// 0=Uncalibrated; 1=Calibrated; 2=Processed 
		/// </summary>
		public short calibration_units;
		/// <summary>
		/// Whether data_units[] is filled or not? 
		/// </summary>
		public short calibration_units_label;
		/// <summary>
		/// Enumerated type (COMP_NONE, (This is the only value)) (short)
		/// </summary>
		public Compression_code_e compression_code;
		/// <summary>
		/// Study descriptor 
		/// </summary>
		public char[] study_type = new char[12];
		/// <summary>
		/// Patient identification descriptor 
		/// </summary>
		public char[] patient_id = new char[16];
		/// <summary>
		/// Patient name (free format ASCII) 
		/// </summary>
		public char[] patient_name = new char[32];
		/// <summary>
		/// Enumerated type (SEX_MALE, _FEMALE,_UNKNOWN) (char)
		/// </summary>
		public ImageHeader.Patient_sex_e patient_sex;
		/// <summary>
		/// Enumerated type (DEXT_RT, _LF, _UNKNOWN) (char)
		/// </summary>
		public char patient_dexterity;
		/// <summary>
		/// Patient age (years) 
		/// </summary>
		public float patient_age;
		/// <summary>
		/// Patient height (cm) 
		/// </summary>
		public float patient_height;
		/// <summary>
		/// Patient weight (kg) 
		/// </summary>
		public float patient_weight;
		/// <summary>
		/// YYYYMMDD (sec from time zero) 
		/// </summary>
		public int patient_birth_date;
		/// <summary>
		/// Attending Physician name (free format) 
		/// </summary>
		public char[] physician_name = new char[32];
		/// <summary>
		/// Operator name (free format) 
		/// </summary>
		public char[] operator_name = new char[32];
		/// <summary>
		/// Free format ASCII 
		/// </summary>
		public char[] study_description = new char[32];
		/// <summary>
		/// Enumerated type (0=Undefined, 1=Blank, 2=Transmission, 3=Static emission, 4=Dynamic
		/// emission, 5=Gated emission, 6=Transmission rectilinear, 7=Emission rectilinear) 
		/// </summary>
		public Acquisition_type_e acquisition_type;
		/// <summary>
		/// Enumerated Type (Bit 0 clear - Feet first, Bit 0 set - Head first, Bit 1-2 00 - Prone, 
		///  Bit 1-2 01 - Supine, Bit 1-2 10 - Decubitus Right, Bit 1-2 11 - Decubitus Left) (short)
		///  </summary>
		public Patient_orientation patient_orientation;
		/// <summary>
		/// Free format ASCII 
		/// </summary>
		public char[] facility_name = new char[20];
		/// <summary>
		/// Number of planes of tacs collected 
		/// </summary>
		public short num_planes;
		/// <summary>
		/// Number of frames of tacs collected OR Highest frame number (in partially reconstructed files) 
		/// </summary>
		public short num_frames;
		/// <summary>
		/// Number of gates of tacs collected 
		/// </summary>
		public short num_gates;
		/// <summary>
		/// Number of bed positions of tacs collected 
		/// </summary>
		public short num_bed_pos;
		/// <summary>
		/// Absolute location of initial bed position (in cm.) 
		/// </summary>
		public float init_bed_position;
		/// <summary>
		/// Absolute bed location (in cm.) 
		/// </summary>
		public float[] bed_position = new float[15];
		/// <summary>
		/// Physical distance between adjacent planes (cm) 
		/// </summary>
		public float plane_separation;
		/// <summary>
		/// Lowest threshold setting for scatter (in KeV) 
		/// </summary>
		public short lwr_sctr_thres;
		/// <summary>
		/// Lower threshold setting for trues in (in KeV) 
		/// </summary>
		public short lwr_true_thres;
		/// <summary>
		/// Upper threshold setting for trues (in KeV) 
		/// </summary>
		public short upr_true_thres;
		/// <summary>
		/// Data processing code (defined by user) 
		/// </summary>
		public char[] user_process_code = new char[10];
		/// <summary>
		/// Enumerated Type (0=Normal, 1=Windowed, 2=Windowed and Nonwindowed, 3=Dual energy,
		/// 4=Upper energy, 5=Emission and Transmission) (short)
		/// </summary>
		public Acquisition_mode_e acquisition_mode;
		/// <summary>
		/// Width of view sample (cm) 
		/// </summary>
		public float bin_size;
		/// <summary>
		/// Fraction of decay by positron emission 
		/// </summary>
		public float branching_fraction;
		/// <summary>
		/// Actual time radiopharmaceutical was injected or flow was started (in sec since base time 1.1.1970 00:00:00) 
		/// </summary>
		public int dose_start_time;
		/// <summary>
		/// Radiopharmaceutical dosage at time of injection (Bq/cc) 
		/// </summary>
		public float dosage;
		/// <summary>
		/// TBD 
		/// </summary>
		public float well_counter_corr_factor;
		/// <summary>
		/// Free text field; fixed strings: "ECAT counts/sec", "Bq/cc" 
		/// </summary>
		public char[] data_units = new char[32];
		/// <summary>
		/// Septa position during scan (0=septa extended, 1=septa retracted) 
		/// </summary>
		public short septa_state;
		/// <summary>
		/// CTI Reserved space (12 bytes)
		/// </summary>
		public short[] fill = new short[6];

		/// <summary>
		/// Resolves minimum and maximum for dataype
		/// </summary>
		/// <returns>array containing {minimum, maximum}</returns>
		public int[] resolveMinMax()
		{
			switch (file_type)
			{
				case datatype_e.ecat7_IMAGE8:
				case datatype_e.ecat7_3DSCAN8:
					return new int[] { -128, 127 };
				case datatype_e.ecat7_VOLUME8:
					return new int[] { 0, 256 };
				case datatype_e.ecat7_IMAGE16:
				case datatype_e.ecat7_PROJ16:
					return new int[] { -32768, 32767 };
				case datatype_e.ecat7_VOLUME16:
					return new int[] { 0, 32767 };
				case datatype_e.ecat7_POLARMAP:
				case datatype_e.ecat7_2DSCAN:
				case datatype_e.ecat7_3DSCANFIT:
				case datatype_e.ecat7_ATTEN:
				case datatype_e.ecat7_2DNORM:
				case datatype_e.ecat7_PROJ:
				case datatype_e.ecat7_3DSCAN:
				case datatype_e.ecat7_3DNORM:
				case datatype_e.ecat7_UNKNOWN:
				default:
					return new int[] { 0, 0 };
			}
		}
		/// <summary>
		/// Resolves imagetype from mainheader
		/// </summary>
		/// <returns>imagetype</returns>
		public ecat7_imagetype resolveImageType()
		{
			switch (file_type)
			{
				case datatype_e.ecat7_IMAGE8:
				case datatype_e.ecat7_VOLUME8:
				case datatype_e.ecat7_IMAGE16:
				case datatype_e.ecat7_VOLUME16:
					return ecat7_imagetype.IMAGE;
				case datatype_e.ecat7_POLARMAP:
					return ecat7_imagetype.POLARMAP;
				case datatype_e.ecat7_2DSCAN:
				case datatype_e.ecat7_3DSCAN8:
				case datatype_e.ecat7_3DSCANFIT:
					return ecat7_imagetype.RAW;
				case datatype_e.ecat7_ATTEN:
				case datatype_e.ecat7_2DNORM:
				case datatype_e.ecat7_PROJ:
				case datatype_e.ecat7_PROJ16:
				case datatype_e.ecat7_3DSCAN:
				case datatype_e.ecat7_3DNORM:
				case datatype_e.ecat7_UNKNOWN:
				default:
					return ecat7_imagetype.IMAGE;
			}
		}
		/// <summary>
		/// Resolves file type from datatype.
		/// </summary>
		/// <param name="datatype">datatype</param>
		/// <returns>filetype enumeration value</returns>
		public datatype_e resolveFileType(ImageFile.DataType datatype)
		{
			switch (datatype)
			{
				case ImageFile.DataType.BIT1:
					return datatype_e.ecat7_UNKNOWN;
				case ImageFile.DataType.BIT8_S:
				case ImageFile.DataType.BIT8_U:
					return datatype_e.ecat7_VOLUME8;
				case ImageFile.DataType.BIT16_S:
				case ImageFile.DataType.BIT16_U:
					return datatype_e.ecat7_VOLUME16;
				case ImageFile.DataType.BIT32_S:
				case ImageFile.DataType.BIT32_U:
				case ImageFile.DataType.BIT64_S:
				case ImageFile.DataType.BIT64_U:
				case ImageFile.DataType.FLT32:
				case ImageFile.DataType.FLT64:
				case ImageFile.DataType.ASCII:
				case ImageFile.DataType.VAXI16:
				case ImageFile.DataType.VAXI32:
				case ImageFile.DataType.VAXFL32:
				case ImageFile.DataType.COLRGB:
				case ImageFile.DataType.SUNI2:
				case ImageFile.DataType.SUNI4:
					return datatype_e.ecat7_UNKNOWN;
			}
			return datatype_e.ecat7_UNKNOWN;
		}
		/// <summary>
		/// Resolves general datatype from mainheader
		/// </summary>
		/// <returns>datatype</returns>
		public ImageFile.DataType resolveDataType()
		{
			switch (file_type)
			{
				case datatype_e.ecat7_VOLUME8:
				case datatype_e.ecat7_IMAGE8:
				case datatype_e.ecat7_3DSCAN8:
					return ImageFile.DataType.BIT8_S;
				case datatype_e.ecat7_UNKNOWN:
				case datatype_e.ecat7_2DSCAN:
				case datatype_e.ecat7_IMAGE16:
				case datatype_e.ecat7_ATTEN:
				case datatype_e.ecat7_2DNORM:
				case datatype_e.ecat7_POLARMAP:
				case datatype_e.ecat7_VOLUME16:
				case datatype_e.ecat7_PROJ:
				case datatype_e.ecat7_PROJ16:
				case datatype_e.ecat7_3DSCAN:
				case datatype_e.ecat7_3DNORM:
				case datatype_e.ecat7_3DSCANFIT:
					return ImageFile.DataType.BIT16_S;
			}
			return ImageFile.DataType.BIT16_S;
		}
		/// <summary>
		/// Ecat7 datatype
		/// </summary>
		public enum datatype_e
		{
			/// <summary>
			/// Unknown type
			/// </summary>
			ecat7_UNKNOWN = 0,
			/// <summary>
			/// 2D scan
			/// </summary>
			ecat7_2DSCAN = 1,
			/// <summary>
			/// 16-bit image
			/// </summary>
			ecat7_IMAGE16 = 2,
			/// <summary>
			/// Attenuation image
			/// </summary>
			ecat7_ATTEN = 3,
			/// <summary>
			/// 2D normalization
			/// </summary>
			ecat7_2DNORM = 4,
			/// <summary>
			/// Polarmap
			/// </summary>
			ecat7_POLARMAP = 5,
			/// <summary>
			/// 8-bit volume
			/// </summary>
			ecat7_VOLUME8 = 6,
			/// <summary>
			/// 16-bit volume
			/// </summary>
			ecat7_VOLUME16 = 7,
			/// <summary>
			/// Projection
			/// </summary>
			ecat7_PROJ = 8,
			/// <summary>
			/// 16-bit Projection
			/// </summary>
			ecat7_PROJ16 = 9,
			/// <summary>
			/// 8-bit image
			/// </summary>
			ecat7_IMAGE8 = 10,
			/// <summary>
			/// 3D scan
			/// </summary>
			ecat7_3DSCAN = 11,
			/// <summary>
			/// 8-bit 3D scan
			/// </summary>
			ecat7_3DSCAN8 = 12,
			/// <summary>
			/// 3D normalization
			/// </summary>
			ecat7_3DNORM = 13,
			/// <summary>
			/// 3D scan (fit)
			/// </summary>
			ecat7_3DSCANFIT = 14
		}
		/// <summary>
		/// Transmission source type
		/// </summary>
		public enum Transm_source_type_e
		{
			/// <summary>
			/// No source
			/// </summary>
			SRC_NONE,
			/// <summary>
			/// R-ring
			/// </summary>
			RRING,
			/// <summary>
			/// ring
			/// </summary>
			_RING,
			/// <summary>
			/// rod
			/// </summary>
			_ROD,
			/// <summary>
			/// prod
			/// </summary>
			_RROD
		}
		/// <summary>
		/// Compression code type (allways = COMP_NONE)
		/// </summary>
		public enum Compression_code_e
		{
			/// <summary>
			/// no compression
			/// </summary>
			COMP_NONE
		}
		/// <summary>
		/// Acquisition type
		/// </summary>
		public enum Acquisition_type_e
		{
			/// <summary>
			/// Unknown
			/// </summary>
			Undefined = 0,
			/// <summary>
			/// Blank
			/// </summary>
			Blank,
			/// <summary>
			/// Transmission scan
			/// </summary>
			Transmission,
			/// <summary>
			/// Static emission
			/// </summary>
			Static_emission,
			/// <summary>
			/// Dynamic emission
			/// </summary>
			Dynamic_emission,
			/// <summary>
			/// Gated acquisition
			/// </summary>
			Gated_emission,
			/// <summary>
			/// Rectilinear transmission
			/// </summary>
			Transmission_rectilinear,
			/// <summary>
			/// Rectilinear emission
			/// </summary>
			Emission_rectilinear
		}

		/// <summary>
		/// Acquisition mode
		/// </summary>
		public enum Acquisition_mode_e
		{
			/// <summary>
			/// Normal mode
			/// </summary>
			Normal = 0,
			/// <summary>
			/// Windowed mode
			/// </summary>
			Windowed,
			/// <summary>
			/// Windowed or non-windowed mode
			/// </summary>
			Windowed_and_Nonwindowed,
			/// <summary>
			/// Dual energy mode
			/// </summary>
			Dual_energy,
			/// <summary>
			/// Upper energy mode
			/// </summary>
			Upper_energy,
			/// <summary>
			/// Emission or transmission mode
			/// </summary>
			Emission_and_Transmission
		}

		/// <summary>
		/// Ecat7 image type (raw, image or polarmap)
		/// </summary>
		public enum ecat7_imagetype
		{
			/// <summary>
			/// Raw tacs image
			/// </summary>
			RAW,
			/// <summary>
			/// Intensity image
			/// </summary>
			IMAGE,
			/// <summary>
			/// Polar map
			/// </summary>
			POLARMAP
		}

		/// <summary>
		/// Patient orientation
		/// </summary>
		public enum Patient_orientation : ushort
		{
			/// <summary>
			/// Feet first prone (0000b)
			/// </summary>
			Feet_first_Prone = 0x0,
			/// <summary>
			/// Feet first supine (0010b)
			/// </summary>
			Feet_first_Supine = 0x2,
			/// <summary>
			/// Feet first decubitus right (0100b)
			/// </summary>
			Feet_first_Decubitus_Right = 0x4,
			/// <summary>
			/// Feet first decubitus left (0110b)
			/// </summary>
			Feet_first_Decubitus_Left = 0x6,
			/// <summary>
			/// Head first prone (0001b)
			/// </summary>
			Head_first_Prone = 0x1,
			/// <summary>
			/// Head first supine (0011b)
			/// </summary>
			Head_first_Supine = 0x3,
			/// <summary>
			/// Head first decubitus right (0101b)
			/// </summary>
			Head_first_Decubitus_Right = 0x5,
			/// <summary>
			/// Head first decubitus left (0111b)
			/// </summary>
			Head_first_Decubitus_Left = 0x7,
			/// <summary>
			/// Unknown orientation
			/// </summary>
			Unknown = 0x8
		}
	}
}
