/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Image
{
    /// <summary>
    /// General header class containing key-data item pairs. Only one value 
    /// can exist for each field name, case-insensitive.
    /// </summary>
    public class GeneralHeader : List<KeyValuePair<string, Object>>
    {
        /// <summary>
        /// Default constructor
        /// </summary>
        public GeneralHeader()
        {
        }
        /// <summary>
        /// Field name that is used in next search for matching pairs from list.
        /// </summary>
        private static string searchedFieldName;
        /// <summary>
        /// Search operator fo field name.
        /// </summary>
        /// <param name="p">key pair that is evaluated</param>
        /// <returns></returns>
        private static bool NameMatch(KeyValuePair<string, Object> p)
        {
            //case-insensitive search
            return (string.Compare(p.Key, searchedFieldName, false) == 0);
        }
        /// <summary>
        /// Gets value defined by field.
        /// </summary>
        /// <param name="fieldName">field name</param>
        /// <returns>field data object</returns>
        /// <exception cref="TPCInvalidArgumentsException">if field is not found from header</exception>
        public Object GetValue(string fieldName)
        {
            searchedFieldName = fieldName;
            int i = base.FindIndex(NameMatch);
            if (i == -1)
                throw new TPCInvalidArgumentsException("Field [" + fieldName + "] was not found from header.");
            return this[i].Value;
        }
        /// <summary>
        /// Gets or sets value defined by field.
        /// </summary>
        /// <param name="fieldName">field name</param>
        /// <returns>field data object</returns>
        public object this[string fieldName]
        {
            get
            {
                searchedFieldName = fieldName;
                int i = base.FindIndex(NameMatch);
                if (i == -1)
                    throw new TPCInvalidArgumentsException("Field [" + fieldName + "] was not found from header.");
                return this[i].Value;
            }
            set
            {
                searchedFieldName = fieldName;
                int i = base.FindIndex(NameMatch);
                if (i != -1)
                {
                    this[i] = new KeyValuePair<string, object>(fieldName, value);
                }
                else
                {
                    this.Add(fieldName, value);
                }
            }
        }
        /// <summary>
        /// Sets value of a field.
        /// </summary>
        /// <param name="fieldName">field name</param>
        /// <param name="value">new field value</param>
        public void SetValue(string fieldName, Object value)
        {
            searchedFieldName = fieldName;
            int i = base.FindIndex(NameMatch);
            this[i] = new KeyValuePair<string, object>(fieldName, value);
        }
        /// <summary>
        /// Adds new item into list.
        /// </summary>
        /// <param name="fieldname">field name</param>
        /// <param name="o">field data</param>
        /// <exception cref="TPCException">if the same name already exists in list</exception>
        public void Add(string fieldname, object o)
        {
            base.Add(new KeyValuePair<string, object>(fieldname, o));
        }
        /// <summary>
        /// Adds new item into list.
        /// </summary>
        /// <param name="p">new item to add</param>
        /// <exception cref="TPCException">if the same name already exists in list</exception>
        public new void Add(KeyValuePair<string, object> p)
        {
            searchedFieldName = p.Key;
            int i = this.FindIndex(NameMatch);
            if (i != -1)
                throw new TPCGeneralHeaderException("Name [" + p.Key + "] already exists in the list.", this[i]);
            this.Add(p);
        }
        /// <summary>
        /// Removes the occurrence with specific field name from the list.
        /// </summary>
        /// <param name="fieldname">name of the field</param>
        public void Remove(string fieldname)
        {
            searchedFieldName = fieldname;
            int i = base.FindIndex(NameMatch);
            if (i != -1) base.Remove(base[i]);
        }
        /// <summary>
        /// Returns all field names. Names are in the order they 
        /// appear in list.
        /// </summary>
        /// <returns>field names in list</returns>
        public string[] GetFieldnames()
        {
            string[] r = new string[this.Count];
            for (int i = 0; i < this.Count; i++)
            {
                r[i] = this[i].Key;
            }
            return r;
        }
        /// <summary>
        /// Returns all values. Values are in the order they 
        /// appear in list.
        /// </summary>
        /// <returns>field names in list</returns>
        public object[] GetValues()
        {
            object[] r = new object[this.Count];
            for (int i = 0; i < this.Count; i++)
            {
                r[i] = this[i].Value;
            }
            return r;
        }
        /// <summary>
        /// Comparator for sorting fields by their field names.
        /// </summary>
        /// <param name="x">list item</param>
        /// <param name="y">list item</param>
        /// <returns>-1 (y greater),0 (equal) or 1 (x greater) depending on order</returns>
        private static int CompareFieldsByName(KeyValuePair<string, Object> x, KeyValuePair<string, Object> y)
        {
            return string.Compare(x.Key, y.Key);
        }
        /// <summary>
        /// Sorts name-value pairs by names in ascending alphapetical order.
        /// </summary>
        public void SortByNames()
        {
            this.Sort(CompareFieldsByName);
        }

        /// <summary>
        /// Reads all entries in this header and converts all value to strings.
        /// </summary>
        /// <returns></returns>
        public SortedList<string, string> StringList()
        {
            SortedList<string, string> stringlist = new SortedList<string, string>();

            string valuestring, name;

            foreach (KeyValuePair<string, object> entry in this)
            {
                name = entry.Key;
                try
                {
                    valuestring = ElementToString(entry.Value);
                }
                catch
                {
                    valuestring = "<Non-readable>";
                }
                try
                {
                    stringlist.Add(name, valuestring);
                }
                catch
                {
                    // Ignore duplicate values
                }
            }
            return stringlist;
        }

        /// <summary>
        /// Converts objects to string representations.
        /// Handles arrays and Dicom values
        /// </summary>
        /// <param name="obj">Object to convert</param>
        /// <returns>String representation of the object</returns>
        public static string ElementToString(object obj)
        {
            Type valuetype = obj.GetType();
            string valuestring = "";

            // Dicom values
            if (valuetype == typeof(openDicom.DataStructure.Value))
            {
                openDicom.DataStructure.Value value = (obj as openDicom.DataStructure.Value);

                if (value.IsMultiValue)
                {
                    valuestring = ElementToString(value[0]);
                    for (int i = 1; i < value.Count; i++)
                    {
                        valuestring += ";" + ElementToString(value[i]);
                    }
                }
                // Sequences and nested data sets are only shown by name
                else if (!value.IsSequence && !value.IsNestedDataSet && !value.IsPixelData)
                {
                    valuestring = ElementToString(value[0]);
                }
            }
            // Arrays
            else if (valuetype.IsArray)
            {
                Array arr = obj as Array;

                // Separator between array elements
                string arrayseparator = ";";
                // If array is byte[] or char[], assume it is text -> no separators
                if (valuetype == typeof(byte[]) || valuetype == typeof(char[])) { arrayseparator = string.Empty; }

                foreach (object o in arr)
                {
                    valuestring += arrayseparator + ElementToString(o);
                }
                valuestring += arrayseparator;
            }
            // Other
            else
            {
                // Byte is interpreted as character
                if (valuetype == typeof(byte))
                {
                    valuestring = ((char)(byte)obj).ToString();
                }
                else
                {
                    valuestring = obj.ToString();
                }
            }

            return valuestring;
        }
    }
}
