/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections;
using System.Collections.Generic;

namespace TPClib.Image
{
	/// <summary>
	/// Class for Image header values.
	/// </summary>
	public class HeaderValue : IConvertible, IEnumerable, IEnumerable<IConvertible>
	{
		#region Private

		private IConvertible[] value;

		private static readonly IFormatProvider format = System.Globalization.NumberFormatInfo.InvariantInfo;

		#endregion

		#region Constructors

		/// <summary>
		/// Default constructor.
		/// </summary>
		/// <param name="n">Number of values in the value array</param>
		private HeaderValue(int n = 0)
		{
			value = new IConvertible[n];
		}

		/// <summary>
		/// Constructor for single value.
		/// </summary>
		/// <param name="ic">Value</param>
		public HeaderValue(IConvertible ic)
		{
			value = new IConvertible[] { ic };
		}

		/// <summary>
		/// Constructor for multiple values.
		/// </summary>
		/// <param name="icArr">Array of values</param>
		public HeaderValue(IConvertible[] icArr)
		{
			value = icArr;
		}

		/// <summary>
		/// Constructor for multiple values. Exception thrown, if the values are not IConvertible.
		/// </summary>
		/// <param name="arr">Array of values</param>
		public HeaderValue(Array arr)
			: this(arr.Length)
		{
			Array.Copy(arr, value, arr.Length);
		}

		#endregion

		#region Public

		/// <summary>
		/// 
		/// </summary>
		public enum HeaderField : uint
		{
			// Image information
			/// <summary></summary>
			WIDTH,
			/// <summary></summary>
			HEIGHT,
			/// <summary></summary>
			PLANES,
			/// <summary></summary>
			FRAMES,
			/// <summary></summary>
			GATES,
			/// <summary></summary>
			BEDPOSITIONS,
			/// <summary></summary>
			DATATYPE,
			/// <summary></summary>
			XPOSITION,
			/// <summary></summary>
			YPOSITION,
			/// <summary></summary>
			ZPOSITION,
			/// <summary></summary>
			VOXELWIDTH,
			/// <summary></summary>
			VOXELHEIGHT,
			/// <summary></summary>
			VOXELTHICKNESS,
			/// <summary></summary>
			PLANEDISTANCE,
			/// <summary></summary>
			FIELDOFVIEW,
			/// <summary></summary>
			DATAUNIT,
			/// <summary></summary>
			DYNAMIC,
			/// <summary></summary>
			FRAMESTARTTIMES,
			/// <summary></summary>
			FRAMEDURATIONS,
			// Radiological information
			/// <summary></summary>
			DOSAGE,
			/// <summary></summary>
			HALFLIFE,
			/// <summary></summary>
			INJECTIONTIME,
			/// <summary></summary>
			ISOTOPE,
			/// <summary></summary>
			RADIOPHARMA,
			// Patient information
			/// <summary></summary>
			PATIENTID,
			/// <summary></summary>
			PATIENTNAME,
			/// <summary></summary>
			PATIENTWEIGHT,
			/// <summary></summary>
			PATIENTHEIGHT,
			/// <summary></summary>
			PATIENTSEX,
			/// <summary></summary>
			PATIENTBIRTHDATE,
			// System information
			/// <summary></summary>
			DATABASENAME,
			/// <summary></summary>
			SYSTEMTYPE,
			/// <summary></summary>
			GANTRYSERIAL,
			/// <summary></summary>
			GANTRYTILT,
			/// <summary></summary>
			GANTRYROTATION,
			/// <summary></summary>
			INITIALBEDPOSITION,
			/// <summary></summary>
			BEDELEVATION,
			/// <summary></summary>
			DISTANCE,
			/// <summary></summary>
			PATIENTORIENTATION,
			// Study information
			/// <summary></summary>
			STUDYSTARTTIME,
			/// <summary></summary>
			STUDYDESCRIPTION,
			/// <summary></summary>
			MODALITY,
			/// <summary></summary>
			PHYSICIAN,
			/// <summary></summary>
			OPERATOR,
			/// <summary></summary>
			FACILITY
		}

		/// <summary>
		/// Empty value
		/// </summary>
		public static readonly HeaderValue EmptyValue = new HeaderValue();

		/// <summary>
		/// Value is an array
		/// </summary>
		public bool IsArray { get { return value.Length > 1; } }

		/// <summary>
		/// Value is the empty value
		/// </summary>
		public bool IsEmpty { get { return value == null; } }

		/// <summary>
		/// Equality operator
		/// </summary>
		/// <param name="obj">Comparison object</param>
		/// <returns>True, if this is equal to the compared object</returns>
		public override bool Equals(object obj)
		{
			return value[0].Equals(obj);
		}

		/// <summary>
		/// Hash code for this object.
		/// </summary>
		/// <returns>Hash code</returns>
		public override int GetHashCode()
		{
			return value.GetHashCode();
		}

		/// <summary>
		/// String representation of this object.
		/// </summary>
		/// <returns>String representation</returns>
		public override string ToString() { return value.ToString(); }

		/// <summary>
		/// Convert this object to an array. If this object has only a single value, returns an array of length 1.
		/// </summary>
		/// <returns>Array of IConvertible objects</returns>
		public IConvertible[] ToArray() { return value; }

		/// <summary>
		/// Enumerator for this object. Enumerates through all the component values in this object.
		/// </summary>
		/// <returns>Enumerator</returns>
		public IEnumerator GetEnumerator()
		{
			return value.GetEnumerator();
		}

		/// <summary>
		/// Enumerator for this object. Enumerates through all the component values in this object.
		/// </summary>
		/// <returns>Enumerator</returns>
		IEnumerator<IConvertible> IEnumerable<IConvertible>.GetEnumerator()
		{
			return (IEnumerator<IConvertible>)value.GetEnumerator();
		}

		#endregion

		#region IConvertible interface

		/// <summary>
		/// Conversion to boolean.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public bool ToBoolean(IFormatProvider provider)
		{
			return value[0].ToBoolean(provider);
		}

		/// <summary>
		/// Conversion to byte.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public byte ToByte(IFormatProvider provider)
		{
			return value[0].ToByte(provider);
		}

		/// <summary>
		/// Conversion to char.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public char ToChar(IFormatProvider provider)
		{
			return value[0].ToChar(provider);
		}

		/// <summary>
		/// Conversion to date time.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public DateTime ToDateTime(IFormatProvider provider)
		{
			return value[0].ToDateTime(provider);
		}

		/// <summary>
		/// Conversion to decimal.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public decimal ToDecimal(IFormatProvider provider)
		{
			return value[0].ToDecimal(provider);
		}

		/// <summary>
		/// Conversion to double.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public double ToDouble(IFormatProvider provider)
		{
			return value[0].ToDouble(provider);
		}

		/// <summary>
		/// Conversion to short.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public short ToInt16(IFormatProvider provider)
		{
			return value[0].ToInt16(provider);
		}

		/// <summary>
		/// Conversion to int.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public int ToInt32(IFormatProvider provider)
		{
			return value[0].ToInt32(provider);
		}

		/// <summary>
		/// Conversion to long.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public long ToInt64(IFormatProvider provider)
		{
			return value[0].ToInt64(provider);
		}

		/// <summary>
		/// Conversion to signed byte.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public sbyte ToSByte(IFormatProvider provider)
		{
			return value[0].ToSByte(provider);
		}

		/// <summary>
		/// Conversion to floating point.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public float ToSingle(IFormatProvider provider)
		{
			return value[0].ToSingle(provider);
		}

		/// <summary>
		/// Conversion to string.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public string ToString(IFormatProvider provider)
		{
			return value[0].ToString(provider);
		}

		/// <summary>
		/// Conversion to any IConvertible type.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <param name="conversionType">Result type</param>
		/// <returns>Converted value</returns>
		public object ToType(Type conversionType, IFormatProvider provider)
		{
			return value[0].ToType(conversionType, provider);
		}

		/// <summary>
		/// Conversion to unsigned short.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public ushort ToUInt16(IFormatProvider provider)
		{
			return value[0].ToUInt16(provider);
		}

		/// <summary>
		/// Conversion to unsigned long.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public uint ToUInt32(IFormatProvider provider)
		{
			return value[0].ToUInt32(provider);
		}

		/// <summary>
		/// Conversion to unsigned long.
		/// </summary>
		/// <param name="provider">Formatting</param>
		/// <returns>Converted value</returns>
		public ulong ToUInt64(IFormatProvider provider)
		{
			return value[0].ToUInt64(provider);
		}

		/// <summary>
		/// Type code.
		/// </summary>
		/// <returns>Type code for this object</returns>
		public TypeCode GetTypeCode()
		{
			return value[0].GetTypeCode();
		}

		#endregion

		#region Conversion operators

		/// <summary>
		/// Conversion operator to double. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Double(HeaderValue hv)
		{
			return hv.value[0].ToDouble(format);
		}

		/// <summary>
		/// Conversion operator to float. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Single(HeaderValue hv)
		{
			return hv.value[0].ToSingle(format);
		}

		/// <summary>
		/// Conversion operator to short. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Int16(HeaderValue hv)
		{
			return hv.value[0].ToInt16(format);
		}

		/// <summary>
		/// Conversion operator to int. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Int32(HeaderValue hv)
		{
			return hv.value[0].ToInt32(format);
		}

		/// <summary>
		/// Conversion operator to long. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Int64(HeaderValue hv)
		{
			return hv.value[0].ToInt64(format);
		}

		/// <summary>
		/// Conversion operator to unsigned short. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator UInt16(HeaderValue hv)
		{
			return hv.value[0].ToUInt16(format);
		}

		/// <summary>
		/// Conversion operator to unsigned int. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator UInt32(HeaderValue hv)
		{
			return hv.value[0].ToUInt32(format);
		}

		/// <summary>
		/// Conversion operator to unsigned long. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator UInt64(HeaderValue hv)
		{
			return hv.value[0].ToUInt64(format);
		}

		/// <summary>
		/// Conversion operator to byte. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Byte(HeaderValue hv)
		{
			return hv.value[0].ToByte(format);
		}

		/// <summary>
		/// Conversion operator to signed byte. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator SByte(HeaderValue hv)
		{
			return hv.value[0].ToSByte(format);
		}

		/// <summary>
		/// Conversion operator to char. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Char(HeaderValue hv)
		{
			return hv.value[0].ToChar(format);
		}

		/// <summary>
		/// Conversion operator to string. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator String(HeaderValue hv)
		{
			return hv.value[0].ToString(format);
		}

		/// <summary>
		/// Conversion operator to boolean. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Boolean(HeaderValue hv)
		{
			return hv.value[0].ToBoolean(format);
		}

		/// <summary>
		/// Conversion operator to decimal. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator Decimal(HeaderValue hv)
		{
			return hv.value[0].ToDecimal(format);
		}

		/// <summary>
		/// Conversion operator to date time. If this object has more than one value, returns conversion of the first value.
		/// </summary>
		/// <param name="hv">Header value to convert</param>
		/// <returns>Converted value</returns>
		public static implicit operator DateTime(HeaderValue hv)
		{
			return hv.value[0].ToDateTime(format);
		}

		/// <summary>
		/// Conversion operator from double.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Double d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from float.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Single d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from short.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Int16 d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from int.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Int32 d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from long.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Int64 d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from unsigned short.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(UInt16 d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from unsigned int.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(UInt32 d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from unsigned long.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(UInt64 d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from byte.
		/// </summary>
		/// <param name="b">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Byte b)
		{
			return new HeaderValue(b);
		}

		/// <summary>
		/// Conversion operator from signed byte.
		/// </summary>
		/// <param name="s">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(SByte s)
		{
			return new HeaderValue(s);
		}

		/// <summary>
		/// Conversion operator from string.
		/// </summary>
		/// <param name="s">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(String s)
		{
			return new HeaderValue(s);
		}

		/// <summary>
		/// Conversion operator from char.
		/// </summary>
		/// <param name="c">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Char c)
		{
			return new HeaderValue(c);
		}

		/// <summary>
		/// Conversion operator from boolean.
		/// </summary>
		/// <param name="b">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Boolean b)
		{
			return new HeaderValue(b);
		}

		/// <summary>
		/// Conversion operator from decimal.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Decimal d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from date time.
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(DateTime d)
		{
			return new HeaderValue(d);
		}

		/// <summary>
		/// Conversion operator from an array. Exception thrown, if the values in the array cannot be converted.
		/// </summary>
		/// <param name="a">Array to convert</param>
		/// <returns>Converted header value</returns>
		public static implicit operator HeaderValue(Array a)
		{
			return new HeaderValue(a);
		}

		#endregion
	}

	/// <summary>
	/// 
	/// </summary>
	public interface IImageheader
	{
		/// <summary>
		/// 
		/// </summary>
		/// <param name="hf"></param>
		/// <returns></returns>
		HeaderValue this[HeaderValue.HeaderField hf] { get; }
	}
}
