/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections;
using TPClib.Common;

namespace TPClib.Image
{
	/// <summary>
	/// Represents an image in N-dimensional space.
	/// </summary>
	public class Image : System.ICloneable, System.Collections.IEnumerable
	{
		#region Private/protected

		/// <summary>
		/// Data of image in one array. The numbers represents
		/// actual true values (already scaled if it was necessary).
		/// </summary>
		private float[][] data;

		/// <summary>
		/// Frame start tacs in milliseconds
		/// </summary>
		private float[] frameStartTimes;

		/// <summary>
		/// Frame durations in milliseconds
		/// </summary>
		private float[] frameDurations;

		/// <summary>
		/// Number of voxels per plane
		/// </summary>
		private int planeLength;

		/// <summary>
		/// Number of voxels per frame
		/// </summary>
		private int frameLength;

		/// <summary>
		/// Image contains dynamic data i.e. the fourth dimension represents time
		/// </summary>
		private bool dynamic = false;

		/// <summary>
		/// Number of voxels in the image
		/// </summary>
		private int dataLength;

		/// <summary>
		/// Maximum value
		/// </summary>
		private float dataMax = Single.MinValue;

		/// <summary>
		/// Minimum value
		/// </summary>
		private float dataMin = Single.MaxValue;

		/// <summary>
		/// Image mean value
		/// </summary>
		private float dataMean = 0;

		/// <summary>
		/// True, if dataMax, dataMin and dataMean have not been updated since the last change to this image
		/// </summary>
		private bool updateNeeded = true;

		/// <summary>
		/// Image dimensions
		/// </summary>
		protected readonly IntLimits dims = new IntLimits();

		#endregion

		#region Constructors

		/// <summary>
		/// Constructs image with zero dimensions.
		/// </summary>
		/// <param name="dynamic">Create image with dynamic data</param>
		public Image(bool dynamic = false)
		{
			this.frameStartTimes = new float[0];
			this.frameDurations = new float[0];
			this.data = new float[0][];
		}

		/// <summary>
		/// Constructor. Create an image based on header data.
		/// </summary>
		/// <param name="hdr">Image header</param>
		public Image(ImageHeader hdr) : this(hdr.Dim, hdr.IsDynamic)
		{
			this.SetFrameStartTimes(hdr.FrameStartTimes);
			this.SetFrameDurations(hdr.FrameDurations);
		}

		/// <summary>
		/// Constructs image with desired dimensions. Image
		/// has zero at all voxels.
		/// </summary>
		/// <param name="dimx">x-dimension</param>
		/// <param name="dimy">y-dimension</param>
		/// <param name="dimz">z-dimension</param>
		/// <param name="dynamic">Create image with dynamic data</param>
		public Image(int dimx, int dimy, int dimz, bool dynamic = false)
			: this(new int[] { dimx, dimy, dimz }, dynamic) { }

		/// <summary>
		/// Constructs image with desired dimensions. Image
		/// has zero at all voxels.
		/// </summary>
		/// <param name="dimx">x-dimension</param>
		/// <param name="dimy">y-dimension</param>
		/// <param name="dimz">z-dimension</param>
		/// <param name="dimt">t-dimension (time points)</param>
		/// <param name="dynamic">Create image with dynamic data</param>
		public Image(int dimx, int dimy, int dimz, int dimt, bool dynamic = true)
			: this(new int[] { dimx, dimy, dimz, dimt }, dynamic) { }

		/// <summary>
		/// Constructs image with desired dimensions. Image
		/// has zero at all voxels.
		/// </summary>
		/// <param name="dimension">dimension object, dimension values will be converted to integer values</param>
		/// <param name="dynamic">Create image with dynamic data</param>
		public Image(IntLimits dimension, bool dynamic = false)
			: this(dimension.ToArray(), dynamic) { }

		/// <summary>
		/// Constructs image with desired dimensions. Image
		/// has zero at all voxels.
		/// </summary>
		/// <param name="dimension">dimension values</param>
		/// <param name="dynamic">Create image with dynamic data</param>
		public Image(int[] dimension, bool dynamic = false)
		{
			if (dimension.Length > 6)
				throw new TPCInvalidArgumentsException("Only maximum of 6-dimensional images are supported by this class.");

			this.dims = new IntLimits(dimension);

			dataLength = Dim.GetProduct();
			planeLength = Dim.Height * Dim.Width;
			frameLength = planeLength * Dim.Planes;

			int totalFrames = dataLength / frameLength;

			frameStartTimes = new float[Dim.Frames];
			frameDurations = new float[Dim.Frames];

			data = new float[totalFrames][];

			for (int i = 0; i < data.Length; i++)
			{
				data[i] = new float[frameLength];
			}

			this.dynamic = dynamic;
		}

		/// <summary>
		/// Copy constructor
		/// </summary>
		/// <param name="image">Copy of this image</param>
		public Image(Image image)
			: this(image.Dim, image.IsDynamic)
		{
			for(int i = 0; i < image.data.Length; i++)
			{
				Array.Copy(image.data[i], this.data[i], image.data[i].Length);
			}
			Array.Copy(image.frameStartTimes, this.frameStartTimes, image.frameStartTimes.Length);
			Array.Copy(image.frameDurations, this.frameDurations, image.frameDurations.Length);
			this.dataMax = image.dataMax;
			this.dataMin = image.dataMin;
		}

		#endregion

		#region Public

		/// <summary>
		/// Image contains dynamic data i.e. the fourth dimension represents time
		/// </summary>
		public virtual bool IsDynamic
		{
			get { return dynamic; }
			set { dynamic = value; }
		}

		/// <summary>
		/// Total number of intensity values in image.
		/// </summary>
		public virtual int DataLength { get { return dataLength; } }

		/// <summary>
		/// Length of one image plane in voxels.
		/// </summary>
		public virtual int Planelength { get { return planeLength; } }

		/// <summary>
		/// Length of one image frame in voxels. 
		/// </summary>
		public virtual int Framelength { get { return frameLength; } }

		/// <summary>
		/// short-cut for image width
		/// </summary>
		public int DimX { get { return this.Width; } }

		/// <summary>
		/// short-cut for image width
		/// </summary>
		public int Width { get { return this.Dim.Width; } }

		/// <summary>
		/// short-cut for image height
		/// </summary>
		public int DimY { get { return this.Height; } }

		/// <summary>
		/// short-cut for image height
		/// </summary>
		public int Height { get { return this.Dim.Height; } }

		/// <summary>
		/// short-cut for image number of planes
		/// </summary>
		public int DimZ { get { return this.Planes; } }

		/// <summary>
		/// short-cut for image number of planes
		/// </summary>
		public int Planes { get { return this.Dim.Planes; } }

		/// <summary>
		/// short-cut for image time dimensions (frames). Will allways return 1 even for static data that has 
		/// no additional frame information.
		/// </summary>
		public int DimT { get { return this.Frames; } }

		/// <summary>
		/// Short-cut for number of frames.
		/// </summary>
		public int Frames { get { return this.Dim.Frames; } }

		/// <summary>
		/// Short-cut for number of gates.
		/// </summary>
		public int Gates { get { return this.Dim.Gates; } }

		/// <summary>
		/// Short-cut for number of bed positions.
		/// </summary>
		public int Beds { get { return this.Dim.Beds; } }

		/// <summary>
		/// Short-cut for detectorvectors.
		/// </summary>
		public int DetectorVector { get { return this.Dim.DetectorVectors; } }

		/// <summary>
		/// Short-cut for energywindows.
		/// </summary>
		public int EnergyWindows { get { return this.Dim.EnergyWindows; } }

		/// <summary>
		/// Copy of the image dimensions
		/// </summary>
		public IntLimits Dim { get { return new IntLimits(dims); } }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="index"></param>
		/// <param name="dimension"></param>
		/// <returns></returns>
		public IntLimits GetDimensionLimits(int index, int dimension)
		{
			return Dim.GetDimensionLimits(index, dimension);
		}

		/// <summary>
		/// Gets all image float tacs as 1-dimensional array.
		/// </summary>
		/// <returns>image tacs as an array</returns>
		public float[] GetArray()
		{
			float[] r = new float[DataLength];
			int offset = 0;
			foreach (float[] f in this.data)
			{
				Array.Copy(f, 0, r, offset, f.Length);
				offset += f.Length;
			}
			return r;
		}

		/// <summary>
		/// Returns an System.Collections.IEnumerator for the System.Array.
		/// </summary>
		/// <returns>enumerator over all image tacs</returns>
		public IEnumerator GetEnumerator()
		{
			foreach (float[] frame in this.data)
			{
				foreach (float f in frame)
				{
					yield return f;
				}
			}
		}

		/// <summary>
		/// Gets value at given index in image.
		/// </summary>
		/// <param name="index">index where value is</param>
		/// <exception cref="TPCInvalidArgumentsException">index is outside of array bounds</exception>
		/// <returns>value at index</returns>
		public float GetValue(int index)
		{
			return this[index];
		}

		/// <summary>
		/// Sets value at given index in image.
		/// </summary>
		/// <param name="index">index where value is set</param>
		/// <param name="value">new value</param>
		/// <exception cref="TPCInvalidArgumentsException">index is outside of array bounds</exception>
		public void SetValue(int index, float value)
		{
			this[index] = value;
		}

		/// <summary>
		/// Voxels of image as an array.
		/// </summary>
		/// <param name="location">index number (0-based)</param>
		/// <returns>value at index</returns>
		public virtual float this[int location]
		{
			get
			{
				int frameloc = location % frameLength;
				int frame = location / frameLength;
				return data[frame][frameloc];
			}
			set
			{
				int frameloc = location % frameLength;
				int frame = location / frameLength;
				data[frame][frameloc] = value;
				this.updateNeeded = true;
			}
		}

		/// <summary>
		/// Voxels of image as an multidimensional array. For multiframe images returns value from the first frame.
		/// </summary>
		/// <remarks>Implementation does not check image bounds for performance reasons</remarks>
		/// <param name="x">x-coordinate (0-based)</param>
		/// <param name="y">y-coordinate (0-based)</param>
		/// <param name="z">z-coordinate (0-based)</param>
		/// <returns>value at location</returns>
		public float this[int x, int y, int z]
		{
			get { return this[x, y, z, 0]; }
			set { this[x, y, z, 0] = value; }
		}

		/// <summary>
		/// Voxels of image as an multidimensional array.
		/// </summary>
		/// <remarks>Implementation does not check image bounds for performance reasons</remarks>
		/// <param name="x">x-coordinate (0-based)</param>
		/// <param name="y">y-coordinate (0-based)</param>
		/// <param name="z">z-coordinate (0-based)</param>
		/// <param name="t">t-coordinate (0-based) (e. g. frames)</param>
		/// <returns>value at location</returns>
		public float this[int x, int y, int z, int t]
		{
			get
			{
				return this[t * this.frameLength + z * this.planeLength + y * Dim.DimX + x];
			}
			set
			{
				this[t * this.frameLength + z * this.planeLength + y * Dim.DimX + x] = value;
			}
		}

		/// <summary>
		/// Voxels of image located by point.
		/// </summary>
		/// <param name="p">location in image</param>
		/// <returns>image voxel value at location</returns>
		public float this[Point p]
		{
			get
			{
				return this[(int)p.X, (int)p.Y, (int)p.Z];
			}
			set
			{
				this[(int)p.X, (int)p.Y, (int)p.Z] = value;
			}
		}

		/// <summary>
		/// Voxels of image located by point.
		/// </summary>
		/// <param name="p">location in image</param>
		/// <returns>image voxel value at location</returns>
		public float this[IntPoint p]
		{
			get
			{
				return this[p.X, p.Y, p.Z];
			}
			set
			{
				this[p.X, p.Y, p.Z] = value;
			}
		}

		/// <summary>
		/// Returns value at given point in image.
		/// </summary>
		/// <param name="point">point where value is retrieved (0-based coordinates)</param>
		/// <returns>tacs value</returns>
		/// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
		public float GetValue(IntPoint point)
		{
			return this[point];
		}

		/// <summary>
		/// Returns value at given location in image.
		/// </summary>
		/// <param name="x">x-dimension index</param>
		/// <param name="y">y-dimension index</param>
		/// <param name="z">z-dimension index</param>
		/// <param name="t">t-dimension index</param>
		/// <returns>tacs value</returns>
		/// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
		public float GetValue(int x, int y, int z, int t = 0)
		{
			return this[x, y, z, t];
		}

		/// <summary>
		/// Returns value at given location in image.
		/// </summary>
		/// <param name="location">location where value is retrieved (0-based coordinates), representing dimensions [x, y, z, ..]</param>
		/// <returns>tacs value</returns>
		/// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
		public float GetValue(int[] location)
		{
			int i = 0;
			int j = 0;
			int arrayindex = 0;
			int prod = 0;
			for (i = 0; i < location.Length; i++)
			{
				prod = 1;
				for (j = i - 1; j >= 0; j--)
					prod *= Dim.GetDimension(j);
				arrayindex += location[i] * prod;
			}
			return this[arrayindex];
		}

		/// <summary>
		/// Sets value at given point in image.
		/// </summary>
		/// <param name="point">point where value is set (0-based coordinates)</param>
		/// <param name="val">new value</param>
		/// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
		public void SetValue(IntPoint point, float val)
		{
			this[point] = val;
		}

		/// <summary>
		/// Sets value at given location in image.
		/// </summary>
		/// <param name="value">new value</param>
		/// <param name="location">location where value is set (0-based coordinates)</param>
		/// <exception cref="TPCInvalidArgumentsException">point outside of valid bounds was accessed</exception>
		public void SetValue(float value, int[] location)
		{
			int arrayindex = 0;
			int prod = 0;
			for (int i = 0; i < location.Length; i++)
			{
				prod = 1;
				for (int j = i - 1; j >= 0; j--)
				{
					prod *= Dim.GetDimension(j);
				}
				arrayindex += location[i] * prod;
			}
			this[arrayindex] = value;
		}

		/// <summary>
		/// Creates new subimage from current image.
		/// </summary>
		/// <param name="sub">subregion</param>
		/// <returns>image that has dimensions as subregion and contains corresponding voxel values</returns>
		public Image GetSubImage(IntLimits sub)
		{
			Image img = new Image(sub, this.dynamic);

			int firstBed = sub.GetLimit(Limits.BEDS, Limits.Limit.LOW);
			int lastBed = sub.GetLimit(Limits.BEDS, Limits.Limit.HIGH);
			int firstGate = sub.GetLimit(Limits.GATES, Limits.Limit.LOW);
			int lastGate = sub.GetLimit(Limits.GATES, Limits.Limit.HIGH);
			int firstFrame = sub.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
			int lastFrame = sub.GetLimit(Limits.FRAMES, Limits.Limit.HIGH);
			int firstPlaneStart = sub.GetLimit(Limits.PLANES, Limits.Limit.LOW) * this.planeLength;
			int lastPlaneEnd = sub.GetLimit(Limits.PLANES, Limits.Limit.HIGH) * this.planeLength;
			int firstRowStart = sub.GetLimit(Limits.HEIGHT, Limits.Limit.LOW) * this.Width;
			int lastRowEnd = sub.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) * this.Width;
			int firstVoxelStart = sub.GetLimit(Limits.WIDTH, Limits.Limit.LOW);
			int rowLength = sub.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) - firstVoxelStart;

			int index = 0;
			int start = 0;
			int subframe = 0;
			for (int b = firstBed; b < lastBed; b++)
			{
				for (int g = firstGate; g < lastGate; g++)
				{
					for (int f = firstFrame; f < lastFrame; f++)
					{
						index = 0;
						for (int p = firstPlaneStart; p < lastPlaneEnd; p += Planelength)
						{
							for (int r = firstRowStart; r < lastRowEnd; r += Width)
							{
								start = p + r + firstVoxelStart;
								Array.Copy(this.data[f], start, img.data[subframe], index, rowLength);
								index += rowLength;
							}
						}
						// Copy the correct frame times
						img.SetFrameStartTime(subframe, this.GetFrameStartTime(f));
						img.SetFrameDuration(subframe, this.GetFrameDuration(f));
						subframe++;
					}
				}
			}
			return img;
		}

		/// <summary>
		/// Sets value to subimage of current image.
		/// </summary>
		/// <param name="sub">subregion</param>
		/// <param name="img">values that are put into image</param>
		public void SetSubImage(IntLimits sub, ref Image img)
		{
			int firstBed = sub.GetLimit(Limits.BEDS, Limits.Limit.LOW);
			int lastBed = sub.GetLimit(Limits.BEDS, Limits.Limit.HIGH);
			int firstGate = sub.GetLimit(Limits.GATES, Limits.Limit.LOW);
			int lastGate = sub.GetLimit(Limits.GATES, Limits.Limit.HIGH);
			int firstFrame = sub.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
			int lastFrame = sub.GetLimit(Limits.FRAMES, Limits.Limit.HIGH);
			int firstPlaneStart = sub.GetLimit(Limits.PLANES, Limits.Limit.LOW) * this.planeLength;
			int lastPlaneEnd = sub.GetLimit(Limits.PLANES, Limits.Limit.HIGH) * this.planeLength;
			int firstRowStart = sub.GetLimit(Limits.HEIGHT, Limits.Limit.LOW) * this.Width;
			int lastRowEnd = sub.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) * this.Width;
			int firstVoxelStart = sub.GetLimit(Limits.WIDTH, Limits.Limit.LOW);
			int rowLength = sub.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) - firstVoxelStart;

			int index = 0;
			int start = 0;
			int subframe = 0;
			for (int b = firstBed; b < lastBed; b++)
			{
				for (int g = firstGate; g < lastGate; g++)
				{
					for (int f = firstFrame; f < lastFrame; f++)
					{
						index = 0;
						for (int p = firstPlaneStart; p < lastPlaneEnd; p += Planelength)
						{
							for (int r = firstRowStart; r < lastRowEnd; r += Width)
							{
								start = p + r + firstVoxelStart;
								Array.Copy(img.data[subframe], index, this.data[f], start, rowLength);
								index += rowLength;
							}
						}
						subframe++;
					}
				}
			}
			this.updateNeeded = true;
		}

		/// <summary>
		/// Multiplies all intensity with value
		/// </summary>
		/// <param name="value">multiplication value</param>
		public virtual void Multiply(float value)
		{
			for (int i = 0; i < DataLength; i++)
			{
				this[i] *= value;
			}
		}

		/// <summary>
		/// Returns minimum intensity value
		/// </summary>
		/// <returns>minimum value</returns>
		public virtual float GetMin()
		{
			if (updateNeeded) UpdateStatistics();
			return this.dataMin;
		}

		/// <summary>
		/// Returns maximum intensity value
		/// </summary>
		/// <returns>maximum value in image</returns>
		public virtual float GetMax()
		{
			if (updateNeeded) UpdateStatistics();
			return this.dataMax;
		}

		/// <summary>
		/// Returns minimum and maximum intensity value
		/// </summary>
		/// <returns>array containing [minimum, maximum] value</returns>
		public virtual float[] GetMinMax()
		{
			if (updateNeeded) UpdateStatistics();
			return new float[] { this.dataMin, this.dataMax };
		}

		/// <summary>
		/// Returns minimum and maximum intensity value
		/// </summary>
		/// <param name="sub">subregion for calculation</param>
		/// <returns>array containing [minimum, maximum] value</returns>
		public virtual float[] GetMinMax(IntLimits sub)
		{
			int firstBed = sub.GetLimit(Limits.BEDS, Limits.Limit.LOW);
			int lastBed = sub.GetLimit(Limits.BEDS, Limits.Limit.HIGH);
			int firstGate = sub.GetLimit(Limits.GATES, Limits.Limit.LOW);
			int lastGate = sub.GetLimit(Limits.GATES, Limits.Limit.HIGH);
			int firstFrame = sub.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
			int lastFrame = sub.GetLimit(Limits.FRAMES, Limits.Limit.HIGH);
			int firstPlaneStart = sub.GetLimit(Limits.PLANES, Limits.Limit.LOW) * this.planeLength;
			int lastPlaneEnd = sub.GetLimit(Limits.PLANES, Limits.Limit.HIGH) * this.planeLength;
			int firstRowStart = sub.GetLimit(Limits.HEIGHT, Limits.Limit.LOW) * this.Width;
			int lastRowEnd = sub.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) * this.Width;
			int firstVoxelStart = sub.GetLimit(Limits.WIDTH, Limits.Limit.LOW);
			int rowLength = sub.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) - firstVoxelStart;

			int start = 0;
			int subframe = 0;
			float min = Single.MaxValue;
			float max = Single.MinValue;
			for (int b = firstBed; b < lastBed; b++)
			{
				for (int g = firstGate; g < lastGate; g++)
				{
					for (int f = firstFrame; f < lastFrame; f++)
					{
						for (int p = firstPlaneStart; p < lastPlaneEnd; p += Planelength)
						{
							for (int r = firstRowStart; r < lastRowEnd; r += Width)
							{
								start = p + r + firstVoxelStart;
								for (int i = start; i < start + rowLength; i++)
								{
									if (data[f][i] < min) min = data[f][i];
									else if (data[f][i] > max) max = data[f][i];
								}
							}
						}
						subframe++;
					}
				}
			}
			return new float[2] { min, max };
		}

		/// <summary>
		/// Returns the mean intensity value
		/// </summary>
		/// <returns>mean value</returns>
		public virtual float GetMean()
		{
			if (updateNeeded) UpdateStatistics();
			return dataMean;
		}

		/// <summary>
		/// Returns the mean intensity value
		/// </summary>
		/// <param name="subregion">subregion for calculation</param>
		/// <returns>mean value</returns>
		public virtual float GetMean(IntLimits subregion)
		{
			if (subregion.GetLimit(Limits.PLANES, Limits.Limit.LOW) < 0 ||
				subregion.GetLimit(Limits.HEIGHT, Limits.Limit.LOW) < 0 ||
				subregion.GetLimit(Limits.WIDTH, Limits.Limit.LOW) < 0 ||
				subregion.GetLimit(Limits.PLANES, Limits.Limit.HIGH) > DimZ ||
				subregion.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH) > DimY ||
				subregion.GetLimit(Limits.WIDTH, Limits.Limit.HIGH) > DimX)
				throw new TPCInvalidArgumentsException("Subregion [" + subregion + "] is not inside image bounds [" + Dim + "]");
			if (DataLength == 0) return float.NaN;
			double mean = 0.0f;
			int i = 0;
			int frames_end = this.Frames;
			int frames_start = 0;
			if (subregion.Length > IntLimits.FRAMES)
			{
				frames_start = subregion.GetLimit(Limits.FRAMES, Limits.Limit.LOW);
				frames_end = subregion.GetLimit(Limits.FRAMES, Limits.Limit.HIGH);
			}
			int planes = (Dim.Length > IntLimits.PLANES ? Dim.Planes : 1);
			int planes_end = (Dim.Length > IntLimits.PLANES ? subregion.GetLimit(Limits.PLANES, Limits.Limit.HIGH) : 1);
			int planes_start = (Dim.Length > IntLimits.PLANES ? subregion.GetLimit(Limits.PLANES, Limits.Limit.LOW) : 0);
			for (int t = frames_start; t < frames_end; t++)
			{
				for (int z = planes_start; z < planes_end; z++)
				{
					for (int y = subregion.GetLimit(Limits.HEIGHT, Limits.Limit.LOW); y < subregion.GetLimit(Limits.HEIGHT, Limits.Limit.HIGH); y++)
					{
						for (int x = subregion.GetLimit(Limits.WIDTH, Limits.Limit.LOW); x < subregion.GetLimit(Limits.WIDTH, Limits.Limit.HIGH); x++)
						{
							i = t * planes * DimY * DimX + z * DimX * DimY + y * DimX + x;
							mean += (double)this[i];
						}
					}
				}
			}
			return (float)mean / (float)subregion.GetProduct();
		}

		/// <summary>
		/// Converts array index into coordinate location using image
		/// as reference_times image. Higher dimensions are ignored.
		/// </summary>
		/// <param name="arrayindex">index value in 1-dimensional array</param>
		/// <returns>3D-coordinate value that corresponds the index value</returns>
		public IntPoint To3DCoordinate(int arrayindex)
		{
			int x, y, z;
			z = arrayindex / (Width * Height);
			arrayindex -= z * Width * Height;
			y = arrayindex / Width;
			arrayindex -= y * Width;
			x = arrayindex;
			return new IntPoint(x, y, z);
		}

		/// <summary>
		/// Shortcut for getting limits for image's plane subregion.
		/// </summary>
		/// <param name="plane">plane index (0-based)</param>
		public IntLimits GetPlaneLimits(int plane)
		{
			return GetDimensionLimits(plane, IntLimits.PLANES);
		}

		/// <summary>
		/// Returns one plane from image.
		/// </summary>
		/// <param name="plane">plane index (0-based)</param>
		/// <returns>new image containing plane intensity values</returns>
		public Image GetPlane(int plane)
		{
			return GetSubImage(GetPlaneLimits(plane));
		}

		/// <summary>
		/// Sets frame start tacs.
		/// </summary>
		/// <param name="start_times">array of non-negative ascending values with same length as number of frames</param>
		/// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
		public void SetFrameStartTimes(double[] start_times)
		{
			float[] fvalues = new float[start_times.Length];
			for (int i = 0; i < fvalues.Length; i++)
				fvalues[i] = (float)start_times[i];
			SetFrameStartTimes(fvalues);
		}

		/// <summary>
		/// Sets frame start tacs.
		/// </summary>
		/// <param name="start_times">array of non-negative ascending values with same length as number of frames</param>
		/// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
		public void SetFrameStartTimes(float[] start_times)
		{
			if (start_times.Length != frameStartTimes.Length) throw new TPCInvalidArgumentsException("Tried to change number of frame start tacs.");
			if (start_times[0] < 0) throw new TPCInvalidArgumentsException("First start time is negative:" + start_times[0]);
			for (int i = 1; i < start_times.Length; i++)
				if (start_times[i - 1] > start_times[i]) throw new TPCInvalidArgumentsException("START tacs of frames are not in ascending order.");

			frameStartTimes = start_times;
		}

		/// <summary>
		/// Sets a frame start time.
		/// </summary>
		/// <param name="index">time index (0-based)</param>
		/// <param name="start_time">non-negative time value (milliseconds)</param>
		/// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
		public void SetFrameStartTime(int index, double start_time)
		{
			if (index < 0 || index >= frameStartTimes.Length)
				throw new TPCInvalidArgumentsException("Index " + index + " out of bounds [0.." + (frameStartTimes.Length - 1) + "]");
			if (index > 0 && start_time < frameStartTimes[index - 1])
				throw new TPCInvalidArgumentsException("Frame start time " + start_time + " is before previous index's value " + frameStartTimes[index - 1]);
			if (index < (frameStartTimes.Length - 1) && frameStartTimes[index + 1] > 0 && start_time > frameStartTimes[index + 1])
				throw new TPCInvalidArgumentsException("Frame start time " + start_time + " is after following index's value " + frameStartTimes[index + 1]);
			frameStartTimes[index] = (float)start_time;
		}

		/// <summary>
		/// Sets a frame end time.
		/// </summary>
		/// <param name="index">time index (0-based)</param>
		/// <param name="end_time">non-negative time value (milliseconds)</param>
		/// <exception cref="TPCInvalidArgumentsException">if input is invalid</exception>
		public void SetFrameEndTime(int index, double end_time)
		{
			if (index < 0 || index >= frameStartTimes.Length)
				throw new TPCInvalidArgumentsException("Index " + index + " out of bounds [0.." + (frameStartTimes.Length - 1) + "]");
			if (index > 0 && end_time < frameStartTimes[index - 1])
				throw new TPCInvalidArgumentsException("Frame end time " + end_time + " is before previous index's value " + frameStartTimes[index - 1]);
			if (index < (frameStartTimes.Length - 1) && end_time > frameStartTimes[index + 1])
				throw new TPCInvalidArgumentsException("Frame end time " + end_time + " is after following index's value " + frameStartTimes[index + 1]);
			if (end_time <= frameStartTimes[index])
				throw new TPCInvalidArgumentsException("Frame end time " + end_time + " is not after frame's start time " + frameStartTimes[index]);
			frameDurations[index] = (float)(end_time - frameStartTimes[index]);
		}

		/// <summary>
		/// Sets frame duration tacs.
		/// </summary>
		/// <param name="durations">array of non-negative values with same length as number of frames</param>
		/// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
		public void SetFrameDurations(float[] durations)
		{
			if (durations.Length != frameDurations.Length) throw new TPCInvalidArgumentsException("Tried to change number of frame start tacs.");
			for (int i = 0; i < durations.Length; i++)
				SetFrameDuration(i, durations[i]);
		}

		/// <summary>
		/// Sets frame duration tacs.
		/// </summary>
		/// <param name="durations">array of non-negative values with same length as number of frames</param>
		/// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
		public void SetFrameDurations(double[] durations)
		{
			if (durations.Length != frameDurations.Length) throw new TPCInvalidArgumentsException("Tried to change number of frame start tacs.");
			for (int i = 0; i < durations.Length; i++)
				SetFrameDuration(i, durations[i]);
		}

		/// <summary>
		/// Sets frame duration time.
		/// </summary>
		/// <param name="index">time index</param>
		/// <param name="duration">a non-negative value</param>
		/// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
		public void SetFrameDuration(int index, double duration)
		{
			if (index < 0 || index >= frameStartTimes.Length)
				throw new TPCInvalidArgumentsException("Index " + index + " out of bounds [0.." + (frameDurations.Length - 1) + "]");
			if (duration < 0) throw new TPCInvalidArgumentsException("Tried to set duration " + index + " a negative value:" + duration);
			frameDurations[index] = (float)duration;
		}

		/// <summary>
		/// Returns frame start time in milliseconds.
		/// </summary>
		/// <param name="i">time index (0-based)</param>
		/// <returns>time in milliseconds</returns>
		public float GetFrameStartTime(int i)
		{
			if (i < 0 || i >= frameStartTimes.Length)
				throw new TPCInvalidArgumentsException("Index " + i + " out of bounds [0.." + (frameStartTimes.Length - 1) + "]");
			return frameStartTimes[i];
		}

		/// <summary>
		/// Returns frame start tacs in milliseconds.
		/// </summary>
		/// <returns>array of tacs in milliseconds</returns>
		public float[] GetFrameStartTimes()
		{
			return frameStartTimes;
		}

		/// <summary>
		/// Returns frame duration in milliseconds.
		/// </summary>
		/// <param name="i">time index (0-based)</param>
		/// <returns>time in milliseconds</returns>
		public float GetFrameDuration(int i)
		{
			if (i < 0 || i >= frameStartTimes.Length)
				throw new TPCInvalidArgumentsException("Index " + i + " out of bounds [0.." + (frameDurations.Length - 1) + "]");
			return frameDurations[i];
		}

		/// <summary>
		/// Returns frame durations in milliseconds.
		/// </summary>
		/// <returns>array of tacs in milliseconds</returns>
		public float[] GetFrameDurations()
		{
			return frameDurations;
		}

		/// <summary>
		/// Rerturns frame end tacs.
		/// </summary>
		/// <returns>array of tacs in milliseconds</returns>
		public float[] GetFrameEndTimes()
		{
			float[] r = new float[frameStartTimes.Length];
			for (int i = 0; i < frameStartTimes.Length; i++)
				r[i] = frameStartTimes[i] + frameDurations[i];
			return r;
		}

		/// <summary>
		/// Returns frame end time.
		/// </summary>
		/// <param name="i">time index (0-based)</param>
		/// <returns>time in milliseconds</returns>
		public float GetFrameEndTime(int i)
		{
			if (i < 0 || i >= frameStartTimes.Length)
				throw new TPCInvalidArgumentsException("Index " + i + " out of bounds [0.." + (frameStartTimes.Length - 1) + "]");
			return frameStartTimes[i] + frameDurations[i];
		}

		/// <summary>
		/// Rerturns frame mid tacs.
		/// </summary>
		/// <returns>array of tacs in milliseconds</returns>
		public float[] GetFrameMidTimes()
		{
			float[] r = new float[frameStartTimes.Length];
			for (int i = 0; i < frameStartTimes.Length; i++)
				r[i] = frameStartTimes[i] + frameDurations[i] / 2.0f;
			return r;
		}

		/// <summary>
		/// Returns curve at point in dynamic file.
		/// </summary>
		/// <param name="x">x-coordinate location in image</param>
		/// <param name="y">y-coordinate location in image</param>
		/// <param name="z">z-coordinate location in image</param>
		/// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
		public float[] GetCurve(int x, int y, int z)
		{
			if (x < 0 || x >= Width)
				throw new TPCInvalidArgumentsException("x-coordinate " + x + " is out of image plane array bounds [0.." + (Width - 1) + "].");
			if (y < 0 || y >= Height)
				throw new TPCInvalidArgumentsException("y-coordinate " + y + " is out of image plane array bounds [0.." + (Height - 1) + "].");
			if (z < 0 || z >= Planes)
				throw new TPCInvalidArgumentsException("z-coordinate " + z + " is out of image plane array bounds [0.." + (Planes - 1) + "].");

			int no_high_dimensions = DataLength / this.frameLength;
			float[] r = new float[no_high_dimensions];
			int frameLength = this.frameLength;
			int location_i = z * DimX * DimY + y * DimX + x;

			//go through all higher dimensions in order
			for (int i = 0; i < no_high_dimensions; i++)
			{
				r[i] = data[i][location_i];
			}
			return r;
		}

		/// <summary>
		/// Gets curve in 3D-location.
		/// </summary>
		/// <param name="location">location in image 3D space( = x * y * z)</param>
		/// <returns>All tacs in location (all frames, gates etc.)</returns>
		/// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
		public float[] GetCurve(int location)
		{
			if (location < 0 || location >= this.frameLength)
				throw new TPCInvalidArgumentsException("Index " + location + " is out of image plane array bounds [0.." + (this.frameLength - 1) + "].");
			int no_high_dimensions = DataLength / this.frameLength;
			int frameLength = this.frameLength;
			float[] r = new float[no_high_dimensions];
			//go through all higher dimensions in order
			for (int i = 0; i < no_high_dimensions; i++)
			{
				r[i] = data[i][location];
			}
			return r;
		}

		/// <summary>
		/// Returns curve at point in dynamic file.
		/// </summary>
		/// <param name="location">3D-coordinate location in image</param>
		/// <returns>All tacs in location (all frames, gates etc.)</returns>
		/// <exception cref="TPCInvalidArgumentsException">if invalid input</exception>
		public float[] GetCurve(IntPoint location)
		{
			if (location.X < 0 || location.X >= Width)
				throw new TPCInvalidArgumentsException("x-coordinate " + location.X + " is out of image plane array bounds [0.." + (Width - 1) + "].");
			if (location.Y < 0 || location.Y >= Height)
				throw new TPCInvalidArgumentsException("y-coordinate " + location.Y + " is out of image plane array bounds [0.." + (Height - 1) + "].");
			if (location.Z < 0 || location.Z >= Planes)
				throw new TPCInvalidArgumentsException("z-coordinate " + location.Z + " is out of image plane array bounds [0.." + (Planes - 1) + "].");

			int no_high_dimensions = DataLength / this.frameLength;
			float[] r = new float[no_high_dimensions];
			int frameLength = this.frameLength;
			int location_i = location.Z * DimX * DimY + location.Y * DimX + location.X;

			//go through all higher dimensions in order
			for (int i = 0; i < no_high_dimensions; i++)
			{
				r[i] = data[i][location_i];
			}
			return r;
		}

		/// <summary>
		/// Shortcut for getting limits for image's frame subregion.
		/// </summary>
		/// <param name="frame">frame index (0-based)</param>
		/// <returns>limits object that represents frame dimensions</returns>
		public IntLimits GetFrameLimits(int frame)
		{
			return GetDimensionLimits(frame, IntLimits.FRAMES);
		}

		/// <summary>
		/// Shortcut for getting limits for image's gate subregion.
		/// </summary>
		/// <param name="gate">gate index (0-based)</param>
		/// <returns>limits object that represents gate dimensions</returns>
		public IntLimits GetGateLimits(int gate)
		{
			return GetDimensionLimits(gate, IntLimits.GATES);
		}

		/// <summary>
		/// Returns tacs of one frame from image.
		/// </summary>
		/// <param name="frame">frame index (0-based)</param>
		/// <returns>new image containing frame tacs</returns>
		public Image GetFrame(int frame)
		{
			Image r = GetSubImage(GetFrameLimits(frame));
			//r.dim.RemoveSingletons();
			return r;
		}

		/// <summary>
		/// Returns tacs of one frame from image.
		/// </summary>
		/// <param name="frame">frame index (0-based)</param>
		/// <param name="image"></param>
		/// <returns>new image containing frame tacs</returns>
		public void SetFrame(int frame, ref Image image)
		{
			SetSubImage(GetFrameLimits(frame), ref image);
		}

		/// <summary>
		/// Prints image information to stdout.
		/// </summary>
		public void PrintImage()
		{
			Console.WriteLine("Image:");
			Console.WriteLine(" dim:" + Dim);
			Console.WriteLine(" min:" + GetMin() + " max:" + GetMax());
		}

		/// <summary>
		/// Returns a String that represents the current Object.
		/// </summary>
		/// <returns>A String that represents the current Object.</returns>
		public override string ToString()
		{
			return "Image[" + Dim + ",min=" + GetMin() + ",max=" + GetMax() + "]";
		}

		/// <summary>
		/// Evaluates if images equals. Currently only image dimensions and tacs is evaluated.
		/// </summary>
		/// <param name="obj">evaluated object</param>
		/// <returns>false if files differ</returns>
		public override bool Equals(object obj)
		{
			if (obj is Image)
			{
				Image img = (obj as Image);
				if (Dim != img.Dim) return false;
				for (int i = 0; i < DataLength; i++)
					if (this[i] != img[i]) return false;
				return true;
			}
			else
			{
				return false;
			}
		}

		/// <summary>
		/// Override that just calls base class method.
		/// </summary>
		/// <returns>hash code of this object</returns>
		public override int GetHashCode()
		{
			return base.GetHashCode();
		}

		/// <summary>
		/// Cloning
		/// </summary>
		/// <remarks>implementation of System.IClonable interface</remarks>
		/// <returns>a new instance of a class with the same value as an existing instance</returns>
		public object Clone()
		{
			Image img = new Image(this.Dim);
			Array.Copy(data, img.data, DataLength);
			return img;
		}

		/// <summary>
		/// Flip image rows.
		/// </summary>
		/// <param name="img">Image to flip</param>
		public static void FlipX(Image img)
		{
			int width = img.Width;
			float temp;
			foreach (float[] f in img.data)
			{
				for (int rowStart = 0; rowStart < f.Length; rowStart += width)
				{
					for (int start = rowStart, end = rowStart + width - 1; start < end; start++, end--)
					{
						temp = f[start];
						f[start] = f[end];
						f[end] = temp;
					}
				}
			}
		}

		/// <summary>
		/// Flip image row order.
		/// </summary>
		/// <param name="img">Image to flip</param>
		public static void FlipY(Image img)
		{
			int planeSize = img.Planelength;
			int width = img.Width;
			float[] temp = new float[width];

			foreach (float[] f in img.data)
			{
				for (int planeStart = 0; planeStart < f.Length; planeStart += planeSize)
				{
					for (int start = planeStart, end = planeStart + planeSize - width; start < end; start += width, end -= width)
					{
						Array.Copy(f, start, temp, 0, width);
						Array.Copy(f, end, f, start, width);
						Array.Copy(temp, 0, f, end, width);
					}
				}
			}
		}

		/// <summary>
		/// Flip image plane order.
		/// </summary>
		/// <param name="img">Image to flip</param>
		public static void FlipZ(Image img)
		{
			int frameSize = img.Framelength;
			int planeSize = img.Planelength;
			float[] temp = new float[planeSize];

			foreach (float[] f in img.data)
			{
				for (int frameStart = 0; frameStart < f.Length; frameStart += frameSize)
				{
					for (int start = frameStart, end = frameStart + frameSize - planeSize; start < end; start += planeSize, end -= planeSize)
					{
						Array.Copy(f, start, temp, 0, planeSize);
						Array.Copy(f, end, f, start, planeSize);
						Array.Copy(temp, 0, f, end, planeSize);
					}
				}
			}
		}

		/// <summary>
		/// Flip to 
		/// </summary>
		/// <param name="o"></param>
		public static void Flip(Orientation oldOrientation, Orientation newOrientation, Image img)
		{
			if (oldOrientation.RightToLeft ^ newOrientation.RightToLeft) FlipX(img);
			if (oldOrientation.AnteriorToPosterior ^ newOrientation.AnteriorToPosterior) FlipY(img);
			if (oldOrientation.SuperiorToInferior ^ newOrientation.SuperiorToInferior) FlipZ(img);
		}

		#endregion

		#region Operators/conversions

		/// <summary>
		/// Multiply operator of image.
		/// </summary>
		/// <param name="value">multiplicator</param>
		/// <param name="image">target image</param>
		/// <returns>multiplied image</returns>
		public static Image operator *(float value, Image image)
		{
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] *= value;
			return r;
		}

		/// <summary>
		/// Multiply operator of image.
		/// </summary>
		/// <param name="image">target image</param>
		/// <param name="value">multiplicator</param>
		/// <returns>multiplied image</returns>
		public static Image operator *(Image image, float value)
		{
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] *= value;
			return r;
		}

		/// <summary>
		/// Multiply operator of image voxel-by-voxel.
		/// </summary>
		/// <param name="image">target image</param>
		/// <param name="value">multiplicator image</param>
		/// <returns>multiplied image</returns>
		public static Image operator *(Image image, Image value)
		{
			if (image.Dim != value.Dim)
				throw new TPCException("Cannot apply arithmetic - operation because image sizes are not the same");
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] *= value[i];
			return r;
		}

		/// <summary>
		/// Divide operator of image.
		/// </summary>
		/// <param name="image">target image</param>
		/// <param name="value">divisor</param>
		/// <returns>resulting image</returns>
		public static Image operator /(Image image, float value)
		{
			if (value == 0) throw new TPCException("Tried to divide image by zero.");
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] /= value;
			return r;
		}

		/// <summary>
		/// Addition operator for image.
		/// </summary>
		/// <param name="value">addition</param>
		/// <param name="image">target image.</param>
		/// <returns>resulting image</returns>
		public static Image operator +(float value, Image image)
		{
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] += value;
			return r;
		}

		/// <summary>
		/// Addition operator for image.
		/// </summary>
		/// <param name="image">target image.</param>
		/// <param name="value">addition</param>
		/// <returns>resulting image</returns>
		public static Image operator +(Image image, float value)
		{
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] += value;
			return r;
		}

		/// <summary>
		/// Decrement operator of image.
		/// </summary>
		/// <param name="image">target image</param>
		/// <param name="value">decrement</param>
		/// <returns>resulting image</returns>
		public static Image operator -(Image image, float value)
		{
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] -= value;
			return r;
		}

		/// <summary>
		/// Decrement operator for images.
		/// </summary>
		/// <param name="image">target image</param>
		/// <param name="value">decrement</param>
		/// <returns>resulting image</returns>
		public static Image operator -(Image image, Image value)
		{
			if (image.DataLength != value.DataLength)
				throw new TPCException("Cannot apply arithmetic - operation because image tacs lengths [" + image.DataLength + "] and [" + value.DataLength + "] are not the same");
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] -= value[i];
			return r;
		}

		/// <summary>
		/// Addition operator for images.
		/// </summary>
		/// <param name="image">target image</param>
		/// <param name="value">increment</param>
		/// <returns>resulting image</returns>
		public static Image operator +(Image image, Image value)
		{
			if (image.DataLength != value.DataLength)
				throw new TPCException("Cannot apply arithmetic + operation because image tacs lengths [" + image.DataLength + "] and [" + value.DataLength + "] are not the same");
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
				r[i] += value[i];
			return r;
		}

		/// <summary>
		/// Division operator for images.
		/// </summary>
		/// <param name="image">target image</param>
		/// <param name="value">divider</param>
		/// <returns>resulting image</returns>
		public static Image operator /(Image image, Image value)
		{
			if (image.DataLength != value.DataLength)
				throw new TPCException("Cannot apply arithmetic / operation because image tacs lengths [" + image.DataLength + "] and [" + value.DataLength + "] are not the same");
			Image r = new Image(image);
			for (int i = 0; i < r.DataLength; i++)
			{
				if (value[i] == 0) throw new TPCException("Tried to divide with zero intensity value.");
				r[i] /= value[i];
			}
			return r;
		}

		/// <summary>
		/// Implicit conversion from 3-dimensional array. 
		/// </summary>
		/// <param name="dataArray">converted tacs</param>
		/// <returns>image holding the tacs</returns>
		public static implicit operator Image(float[, ,] dataArray)
		{
			Image r = new Image(dataArray.GetLength(0), dataArray.GetLength(1), dataArray.GetLength(2));
			int planeLength = r.Planelength;
			for (int z = 0; z < dataArray.GetLength(0); z++)
				for (int y = 0; y < dataArray.GetLength(1); y++)
					for (int x = 0; x < dataArray.GetLength(2); x++)
						r[x + y * r.DimX + z * planeLength] = dataArray[x, y, z];
			return r;
		}

		/// <summary>
		/// Implicit conversion from 2-dimensional array. 
		/// </summary>
		/// <param name="dataArray">converted tacs</param>
		/// <returns>image holding the tacs</returns>
		public static implicit operator Image(float[,] dataArray)
		{
			Image r = new Image(dataArray.GetLength(0), dataArray.GetLength(1), 1);
			for (int y = 0; y < dataArray.GetLength(1); y++)
				for (int x = 0; x < dataArray.GetLength(2); x++)
					r[x, y, 0] = dataArray[x, y];
			return r;
		}

		/// <summary>
		/// Implicit conversion from 2-dimensional array. 
		/// </summary>
		/// <remarks>tacs is converted to float precision</remarks>
		/// <param name="dataArray">converted tacs</param>
		/// <returns>image holding the tacs</returns>
		public static implicit operator Image(double[,] dataArray)
		{
			Image r = new Image(dataArray.GetLength(0), dataArray.GetLength(1), 1);
			for (int y = 0; y < dataArray.GetLength(0); y++)
				for (int x = 0; x < dataArray.GetLength(1); x++)
					r[y, x, 0] = (float)dataArray[y, x];
			return r;
		}

		/// <summary>
		/// Explicit conversion to 2-dimensional array. 
		/// </summary>
		/// <param name="dataImage">image holding the tacs</param>
		/// <returns>converted tacs</returns>
		public static explicit operator float[,](Image dataImage)
		{
			if (dataImage.Dim.Length > IntLimits.PLANES && dataImage.DimZ > 1)
				throw new TPCException("Cannot cast to 2D array because image has more than one slice");
			float[,] r = new float[dataImage.DimX, dataImage.DimY];
			for (int y = 0; y < dataImage.DimY; y++)
				for (int x = 0; x < dataImage.DimX; x++)
					r[x, y] = dataImage[x, y, 0];
			return r;
		}

		/// <summary>
		/// Explicit conversion to 2-dimensional array. 
		/// </summary>
		/// <remarks>tacs in Image object is converted from float precision</remarks>
		/// <param name="dataImage">image holding the tacs</param>
		/// <returns>converted tacs</returns>
		public static explicit operator double[,](Image dataImage)
		{
			if (dataImage.Dim.Length > IntLimits.PLANES && dataImage.DimZ > 1)
				throw new TPCException("Cannot cast to 2D array because image has more than one slice");
			double[,] r = new double[dataImage.DimX, dataImage.DimY];
			for (int y = 0; y < dataImage.DimY; y++)
				for (int x = 0; x < dataImage.DimX; x++)
					r[x, y] = dataImage[x, y, 0];
			return r;
		}

		/// <summary>
		/// Explicit conversion to 3-dimensional array. 
		/// </summary>
		/// <param name="dataImage">image holding the tacs</param>
		/// <returns>converted tacs</returns>
		public static explicit operator float[, ,](Image dataImage)
		{
			if (dataImage.Dim.Length > IntLimits.FRAMES && dataImage.DimT > 1)
				throw new TPCException("Cannot cast to 3D array because image has more than one frame");
			int newDimz = 1;
			if (dataImage.Dim.Length > IntLimits.PLANES)
				newDimz = dataImage.DimZ;
			float[, ,] r = new float[dataImage.DimX, dataImage.DimY, newDimz];
			for (int z = 0; z < newDimz; z++)
				for (int y = 0; y < dataImage.DimY; y++)
					for (int x = 0; x < dataImage.DimX; x++)
						r[x, y, z] = dataImage[x, y, z];
			return r;
		}

		/// <summary>
		/// Implicit conversion to 1-dimensional array. 
		/// </summary>
		/// <param name="dataImage">image holding the tacs</param>
		/// <returns>converted tacs</returns>
		public static implicit operator float[](Image dataImage)
		{
			float[] r = new float[dataImage.DataLength];
			for (int i = 0; i < dataImage.DataLength; i++)
				r[i] = dataImage[i];
			return r;
		}

		#endregion

		#region Private

		/// <summary>
		/// Update the dataMax and dataMin values in this image.
		/// </summary>
		private void UpdateStatistics()
		{
			double sum = 0.0;
			float min = Single.MaxValue;
			float max = Single.MinValue;
			foreach (float[] farr in this.data)
			{
				Array.ForEach<float>(	farr,
										delegate(float f) {
											if (f > max) max = f;
											if (f < min) min = f;
											sum += f;
										});
			}
			this.dataMax = max;
			this.dataMin = min;
			this.dataMean = (float)(sum / dataLength);
			this.updateNeeded = false;
		}

		#endregion
	}
}
