/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Image
{
	#region Public enumerators

	/// <summary>
	/// Isotope name
	/// </summary>
	public enum Isotope_enumerator
	{
		/// <summary>
		/// Coal-11
		/// </summary>
		C11,
		/// <summary>
		/// Oxygen-14
		/// </summary>
		O14,
		/// <summary>
		/// Oxygen-15
		/// </summary>
		O15,
		/// <summary>
		/// Fluor-18
		/// </summary>
		F18,
		/// <summary>
		/// Germanium-68
		/// </summary>
		Ge68,
		/// <summary>
		/// Gallium-68
		/// </summary>
		Ga68,
		/// <summary>
		/// Bromine-75
		/// </summary>
		Br75,
		/// <summary>
		/// Bromine-76
		/// </summary>
		Br76,
		/// <summary>
		/// Copper-62
		/// </summary>
		Cu62,
		/// <summary>
		/// Copper-64
		/// </summary>
		Cu64,
		/// <summary>
		/// Iron-52
		/// </summary>
		Fe52,
		/// <summary>
		/// Nitrogen-13
		/// </summary>
		N13,
		/// <summary>
		/// Natrium-22
		/// </summary>
		Na22,
		/// <summary>
		/// Bromine-82
		/// </summary>
		Br82,
		/// <summary>
		/// Zinc-62
		/// </summary>
		Zn62,
		/// <summary>
		/// Iodine-124
		/// </summary>
		I124,
		/// <summary>
		/// Rubidium-82
		/// </summary>
		Rb82,
		/// <summary>
		/// No isotope
		/// </summary>
		No_isotope
	}

	/// <summary>
	/// PET image acquisition type
	/// </summary>
	public enum AcquisitionType
	{
		/// <summary>
		/// unknown = static
		/// </summary>
		ACQUISITION_UNKNOWN = 0,
		/// <summary>
		/// 
		/// </summary>
		ACQUISITION_STATIC = 1,
		/// <summary>
		/// dynamic
		/// </summary>
		ACQUISITION_DYNAMIC = 2,
		/// <summary>
		/// tomographic
		/// </summary>
		ACQUISITION_TOMO = 3,
		/// <summary>
		/// gated
		/// </summary>
		ACQUISITION_GATED = 4,
		/// <summary>
		/// gated spect
		/// </summary>
		ACQUISITION_GSPECT = 5
	}
	
	/// <summary>
	/// Weight unit
	/// </summary>
	public enum Weight_unit
	{
		/// <summary>
		/// Unknown weight units
		/// </summary>
		Unknown = 0,
		/// <summary>
		/// Grams
		/// </summary>
		g = 1,
		/// <summary>
		/// Ounces
		/// </summary>
		oz = 2,
		/// <summary>
		/// Kilograms
		/// </summary>
		kg = 3,
		/// <summary>
		/// Pounds
		/// </summary>
		lb = 4
	}

	/// <summary>
	/// Dicom modality type, for MDC library
	/// </summary>
	public enum ImageModality
	{
		/// <summary>
		/// Unknown modality
		/// </summary>
		Unknown = 0,
		/// <summary>
		/// Angioscopy
		/// </summary>
		M_AS = ('A' << 8) | 'S',
		/// <summary>
		/// Audio
		/// </summary>
		M_AU = ('A' << 8) | 'U',
		/// <summary>
		/// Biomagnetic Imaging
		/// </summary>
		M_BI = ('B' << 8) | 'I',
		/// <summary>
		/// Color Flow Doppler
		/// </summary>
		M_CD = ('C' << 8) | 'D',
		/// <summary>
		/// Cinefluorography
		/// </summary>
		M_CF = ('C' << 8) | 'F',
		/// <summary>
		/// Culposcopy
		/// </summary>
		M_CP = ('C' << 8) | 'P',
		/// <summary>
		/// Computed Radiography
		/// </summary>
		M_CR = ('C' << 8) | 'R',
		/// <summary>
		/// Cystoscopy
		/// </summary>
		M_CS = ('C' << 8) | 'S',
		/// <summary>
		/// Computed Tomography
		/// </summary>
		M_CT = ('C' << 8) | 'T',
		/// <summary>
		/// Duplex Doppler
		/// </summary>
		M_DD = ('D' << 8) | 'D',
		/// <summary>
		/// Digital Fluoroscopy
		/// </summary>
		M_DF = ('D' << 8) | 'F',
		/// <summary>
		/// Diaphanography
		/// </summary>
		M_DG = ('D' << 8) | 'G',
		/// <summary>
		/// Digital Microscopy
		/// </summary>
		M_DM = ('D' << 8) | 'M',
		/// <summary>
		/// Digital Substraction Angiography
		/// </summary>
		M_DS = ('D' << 8) | 'S',
		/// <summary>
		/// Digital Radiography
		/// </summary>
		M_DX = ('D' << 8) | 'X',
		/// <summary>
		/// Echocardiography
		/// </summary>
		M_EC = ('E' << 8) | 'C',
		/// <summary>
		/// Endoscopy
		/// </summary>
		M_ES = ('E' << 8) | 'S',
		/// <summary>
		/// Fluorescein Angiography
		/// </summary>
		M_FA = ('F' << 8) | 'A',
		/// <summary>
		/// Fundoscopy
		/// </summary>
		M_FS = ('F' << 8) | 'S',
		/// <summary>
		/// General Microscopy
		/// </summary>
		M_GM = ('G' << 8) | 'M',
		/// <summary>
		/// Hemodynamic Waveform
		/// </summary>
		M_HD = ('H' << 8) | 'D',
		/// <summary>
		/// Intra-Oral Radiography
		/// </summary>
		M_IO = ('I' << 8) | 'O',
		/// <summary>
		/// Hardcopy
		/// </summary>
		M_HC = ('H' << 8) | 'C',
		/// <summary>
		/// Laparoscopy
		/// </summary>
		M_LP = ('L' << 8) | 'P',
		/// <summary>
		/// Magnetic Resonance Angiography
		/// </summary>
		M_MA = ('M' << 8) | 'A',
		/// <summary>
		/// Mammography
		/// </summary>
		M_MG = ('M' << 8) | 'G',
		/// <summary>
		/// Magnetic Resonance
		/// </summary>
		M_MR = ('M' << 8) | 'R',
		/// <summary>
		/// Magnetic Resonance Spectroscopy
		/// </summary>
		M_MS = ('M' << 8) | 'S',
		/// <summary>
		/// Nuclear Medicine
		/// </summary>
		M_NM = ('N' << 8) | 'M',
		/// <summary>
		/// Other
		/// </summary>
		M_OT = ('O' << 8) | 'T',
		/// <summary>
		/// Positron Emission Tomography
		/// </summary>
		M_PT = ('P' << 8) | 'T',
		/// <summary>
		/// Panoramic X-Ray
		/// </summary>
		M_PX = ('P' << 8) | 'X',
		/// <summary>
		/// Radio Fluoroscopy
		/// </summary>
		M_RF = ('R' << 8) | 'F',
		/// <summary>
		/// Radiographic Imaging
		/// </summary>
		M_RG = ('R' << 8) | 'G',
		/// <summary>
		/// Radiotherapy
		/// </summary>
		M_RT = ('R' << 8) | 'T',
		/// <summary>
		/// Slide Microscopy
		/// </summary>
		M_SM = ('S' << 8) | 'M',
		/// <summary>
		/// SR Document
		/// </summary>
		M_SR = ('S' << 8) | 'R',
		/// <summary>
		/// Single-Photon Emission Computed Tomography
		/// </summary>
		M_ST = ('S' << 8) | 'T',
		/// <summary>
		/// Thermography
		/// </summary>
		M_TG = ('T' << 8) | 'G',
		/// <summary>
		/// Ultrasound
		/// </summary>
		M_US = ('U' << 8) | 'S',
		/// <summary>
		/// Videofluorography
		/// </summary>
		M_VF = ('V' << 8) | 'F',
		/// <summary>
		/// X-Ray Angiography
		/// </summary>
		M_XA = ('X' << 8) | 'A',
		/// <summary>
		/// External-Camera Photography
		/// </summary>
		M_XC = ('X' << 8) | 'C'
	}
	#endregion

	/// <summary>
	/// Class for representing person names
	/// </summary>
	public class PersonName : IEquatable<PersonName>, IEquatable<string>
	{
		private string[] names;

		public String FirstNames
		{
			get
			{
				string n = String.Empty;
				for (int i = 1; i < names.Length; i++) n += names[i] + " ";
				return n.Trim();
			}
		}

		public String LastName
		{
			get { return names.Length > 0 ? names[0] : String.Empty; }
		}

		public String FullName
		{
			get
			{
				string n = String.Empty;
				for (int i = 0; i < names.Length; i++) n += names[i] + " ";
				return n.Trim();
			}
		}

		public PersonName(string name)
			: this(name.Split(new char[] { ' ', ',' }, StringSplitOptions.RemoveEmptyEntries)) { }

		private PersonName() { names = new string[0]; }

		public PersonName(params string[] n)
		{
			names = Array.ConvertAll<string, string>(n, delegate(string s) { return s.Trim(new char[] { ' ', '\0' }); });
		}

		public override string ToString()
		{
			return FullName;
		}

		public override bool Equals(object obj)
		{
			return this.ToString().Equals(obj.ToString());
		}

		public override int GetHashCode()
		{
			return FullName.GetHashCode();
		}

		public bool Equals(string other)
		{
			return this.FullName.Equals(other);
		}

		public bool Equals(PersonName other)
		{
			return this.Equals(other.FullName);
		}

		public readonly static PersonName Empty = new PersonName();

		public static implicit operator PersonName(string s) { return new PersonName(s); }

		public static implicit operator String(PersonName p) { return p.ToString(); }
	}

	/// <summary>
	/// Common header information to all images.
	/// </summary>
	public class ImageHeader : HeaderFieldList
	{
		#region Private

		private static readonly DateTime DefaultTime = new DateTime(0, DateTimeKind.Local);

		private Orientation orientation = Orientation.Radiological;
		private string description = String.Empty;
		private PersonName patientName = PersonName.Empty;
		private string patientId = String.Empty;
		private string radiopharma = String.Empty;
		private float dose = 0.0f;
		private DateTime injectionTime = DefaultTime;
		private IntLimits dims = new IntLimits();
		private Voxel size = new Voxel();
		private ImageModality modality = ImageModality.Unknown;
		private DataUnit dataUnit = DataUnit.Unknown;
		private ImageFile.DataType dataType = ImageFile.DataType.BIT16_U;
		private Isotope_enumerator isotope = Isotope_enumerator.No_isotope;
		private bool dynamic = false;

		private float[] frameStarts;
		private float[] frameDurations;

		#endregion

		#region Constructors

		/// <summary>
		/// Default constructor.
		/// </summary>
		public ImageHeader(bool createDynamic = false)
		{
			dynamic = createDynamic;
		}

		/// <summary>
		/// Copy constructor.
		/// </summary>
		public ImageHeader(HeaderFieldList headerlist)
			: this()
		{
			if (headerlist is ImageHeader)
			{
				ImageHeader hdr = headerlist as ImageHeader;
				this.PatientName = hdr.PatientName;
				this.PatientID = hdr.PatientID;
				this.Dim = new IntLimits(hdr.Dim);
				this.Siz = new Voxel(hdr.Siz);
				this.Description = hdr.Description;
				this.Datatype = hdr.Datatype;
				this.Dataunit = hdr.Dataunit;
				this.Modality = hdr.Modality;
				this.DoseStartTime = hdr.DoseStartTime;
				this.InjectedDose = hdr.InjectedDose;
				this.Isotope = hdr.Isotope;
				this.Radiopharma = hdr.Radiopharma;
				this.IsDynamic = hdr.IsDynamic;
				this.FrameDurations = hdr.FrameDurations;
				this.FrameStartTimes = hdr.FrameStartTimes;
				this.Orientation = hdr.Orientation;
			}
			this.AddRange(headerlist);
		}

		#endregion

		#region Public

		/// <summary>
		/// Image orientation
		/// </summary>
		public virtual Orientation Orientation { get { return orientation; } set { orientation = value; } }

		/// <summary>
		/// Header has dynamic image information i.e. the fourth dimension represents time
		/// </summary>
		public virtual bool IsDynamic { get { return dynamic; } set { dynamic = value; } }

		/// <summary>
		/// Frame start times in ms.
		/// </summary>
		public float[] FrameStartTimes
		{
			get
			{
				if (frameStarts != null) return frameStarts;
				else return new float[this.DimT];
			}
			set
			{
				float start = 0;
				int i = 0;
				for (; i < this.DimT && i < value.Length; i++)
				{
					start = value[i];
					this.SetFrameStart(i, start);
				}
				for (; i < this.DimT; i++) this.SetFrameStart(i, start);
			}
		}

		/// <summary>
		/// Frame durations in ms.
		/// </summary>
		public float[] FrameDurations
		{
			get
			{
				if (frameStarts != null) return frameDurations;
				else return new float[this.DimT];
			}
			set
			{
				int i = 0;
				for (; i < this.DimT && i < value.Length; i++)
				{
					this.SetFrameDuration(i, value[i]);
				}
				for (; i < this.DimT; i++) this.SetFrameDuration(i, 0);
			}
		}

		/// <summary>
		/// Frame end times in ms.
		/// </summary>
		public float[] FrameEndTimes
		{
			get
			{
				float[] endTimes = new float[this.Frames];
				for (int i = 0; i < this.Frames; i++)
				{
					endTimes[i] = this.GetFrameStart(i) + this.GetFrameDuration(i);
				}
				return endTimes;
			}
			set
			{
				int i = 0;
				for (; i < this.DimT && i < value.Length; i++)
				{
					this.SetFrameDuration(i, value[i] - this.GetFrameStart(i));
				}
				for (; i < this.DimT; i++) this.SetFrameDuration(i, 0);
			}
		}

		/// <summary>
		/// Set the duration of a single frame.
		/// </summary>
		/// <param name="i">Frame number</param>
		/// <param name="time">Frame duration</param>
		public void SetFrameDuration(int i, double time)
		{
			if (i < this.DimT)
			{
				if (frameDurations == null || i >= frameDurations.Length)
				{
					float[] newDurations = new float[this.DimT];
					if (frameDurations != null) Array.Copy(frameDurations, newDurations, frameDurations.Length);
					frameDurations = newDurations;
				}
				frameDurations[i] = (float)time;
			}
			else throw new TPCInvalidArgumentsException("Dimension mismatch: image does not have " + i + " frames");
		}

		/// <summary>
		/// Set the duration of a single frame
		/// </summary>
		/// <param name="i">Frame number</param>
		public float GetFrameDuration(int i)
		{
			if (i < this.Frames && frameDurations != null && i < frameDurations.Length)
				return frameDurations[i];
			else return 0;
		}

		/// <summary>
		/// Set the duration of a single frame
		/// </summary>
		/// <param name="i">Frame number</param>
		/// <param name="time">Frame duration</param>
		public void SetFrameStart(int i, double time)
		{
			if (i < this.DimT)
			{
				if (frameStarts == null || i >= frameStarts.Length)
				{
					float[] newFrameStarts = new float[this.DimT];
					if (frameStarts != null) Array.Copy(frameStarts, newFrameStarts, frameStarts.Length);
					frameStarts = newFrameStarts;
				}
				frameStarts[i] = (float)time;
			}
			else throw new TPCInvalidArgumentsException("Dimension mismatch: image does not have " + i + " frames");
		}

		/// <summary>
		/// Set the duration of a single fram
		/// </summary>
		/// <param name="i">Frame number</param>
		public float GetFrameStart(int i)
		{
			if (i < this.Frames && frameStarts != null && i < frameStarts.Length)
				return frameStarts[i];
			else return 0;
		}

		/// <summary>
		/// Set the end time of a single frame.
		/// </summary>
		/// <param name="i">Frame number</param>
		/// <param name="time">Frame end time</param>
		public void SetFrameEnd(int i, double time)
		{
			SetFrameDuration(i, time - GetFrameStart(i));
		}

		/// <summary>
		/// Set the end time of a single frame.
		/// </summary>
		/// <param name="i">Frame number</param>
		public float GetFrameEnd(int i)
		{
			return GetFrameStart(i) + GetFrameDuration(i);
		}

		/// <summary>
		/// Dimensions. Low values are allways zero, high value is number of indexes
		/// in axis.
		/// </summary>
		public virtual IntLimits Dim { get { return dims; } set { dims = value; } }

		/// <summary>
		/// image width in voxels.
		/// </summary>
		public virtual int Width
		{
			get { return Dim.Width; }
		}

		/// <summary>
		/// image height in voxels.
		/// </summary>
		public virtual int Height
		{
			get { return Dim.Height; }
		}

		/// <summary>
		/// number of planes.
		/// </summary>
		public virtual int Planes
		{
			get { return Dim.Planes; }
		}

		/// <summary>
		/// frames (time dimension)
		/// </summary>
		public virtual int Frames
		{
			get { return Dim.Frames; }
		}

		/// <summary>
		/// t-dimension. (time dimension)
		/// </summary>
		public virtual int DimT
		{
			get { return this.Frames; }
		}

		/// <summary>
		/// x-dimension.
		/// </summary>
		public virtual int DimX
		{
			get { return this.Width; }
		}

		/// <summary>
		/// y-dimension.
		/// </summary>
		public virtual int DimY
		{
			get { return this.Height; }
		}

		/// <summary>
		/// z-dimension.
		/// </summary>
		public virtual int DimZ
		{
			get { return this.Planes; }
		}

		/// <summary>
		/// Read-only short-cut for voxel size (mm).
		/// </summary>
		public virtual float SizeX
		{
			get { return Siz.SizeX; }
		}

		/// <summary>
		/// Read-only short-cut for voxel size (mm).
		/// </summary>
		public virtual float SizeY
		{
			get { return Siz.SizeY; }
		}

		/// <summary>
		/// Read-only short-cut for voxel size (mm).
		/// </summary>
		public virtual float SizeZ
		{
			get { return Siz.SizeZ; }
		}

		/// <summary>
		/// Dose start time.
		/// </summary>
		public virtual DateTime DoseStartTime
		{
			get { return this.injectionTime; }
			set { this.injectionTime = value; }
		}

		/// <summary>
		/// Injected dose amount in MBq
		/// </summary>
		public virtual float InjectedDose
		{
			get { return this.dose; }
			set { this.dose = value; }
		}

		/// <summary>
		/// Radiopharmaceutical name (DEFAULT == unknown)
		/// </summary>
		public virtual string Radiopharma
		{
			get { return this.radiopharma; }
			set { this.radiopharma = value; }
		}

		/// <summary>
		/// Isotope of image (DEFAULT == no isotope)
		/// </summary>
		public virtual Isotope_enumerator Isotope
		{
			get { return this.isotope; }
			set { this.isotope = value; }
		}

		/// <summary>
		/// Patient name.
		/// </summary>
		public virtual PersonName PatientName
		{
			get { return this.patientName; }
			set { this.patientName = value; }
		}

		/// <summary>
		/// Voxel size
		/// </summary>
		public virtual Voxel Siz
		{
			get { return this.size; }
			set { this.size = value; }
		}

		/// <summary>
		/// patient identification string
		/// </summary>
		public virtual string PatientID
		{
			get { return this.patientId; }
			set { this.patientId = value; }
		}

		/// <summary>
		/// description field of image
		/// </summary>
		public virtual string Description
		{
			get { return this.description; }
			set { this.description = value; }
		}

		/// <summary>
		/// Data binary representation type.
		/// </summary>
		public virtual ImageFile.DataType Datatype
		{
			get { return this.dataType; }
			set { this.dataType = value; }
		}

		/// <summary>
		/// Data representation unit.
		/// </summary>
		public virtual DataUnit Dataunit {
			get { return this.dataUnit; }
			set { this.dataUnit = value;}
		}

		/// <summary>
		/// Image modality type
		/// </summary>
		public virtual ImageModality Modality
		{
			get { return this.modality; }
			set { this.modality = value; }
		}

		/// <summary>
		/// Removes control characters from string. This method is used for 
		/// removing unwanted characters from header fields.
		/// </summary>
		/// <param name="str">string with some characters</param>
		/// <returns>same string wihtout control characters</returns>
		public static string RemoveControlCharacters(string str)
		{
			string tmp = String.Empty;
			foreach (char c in str)
			{
				if (!System.Char.IsControl(c))
					tmp += c;
			}
			return tmp;
		}

		/// <summary>
		/// Returns image header as string
		/// </summary>
		/// <returns>string representation</returns>
		public override string ToString()
		{
			return "ImageHeader[patient_name=" + PatientName + ", patientID=" + PatientID + ", " + Dim + ", " + Siz + ", datatype=" + Datatype + "]";
		}

		/// <summary>
		/// Evaluates if two objects are equal
		/// </summary>
		/// <param name="obj">evaluated object</param>
		/// <returns>true if header tacs is equal to this object's tacs</returns>
		public override bool Equals(object obj)
		{
			ImageHeader h;
			if (obj is ImageHeader)
			{
				h = (obj as ImageHeader);
				if (h.Datatype != Datatype) return false;
				if (h.Dataunit != Dataunit) return false;
				if (h.Description != Description) return false;
				if (h.Dim != Dim) return false;
				return true;
			}
			else
			{
				return false;
			}
		}

		/// <summary>
		/// Override that just calls base class method.
		/// </summary>
		/// <returns></returns>
		public override int GetHashCode()
		{
			return base.GetHashCode();
		}

		/// <summary>
		/// Patient sex type
		/// </summary>
		public enum Patient_sex_e
		{
			/// <summary>
			/// Male, character 'M' value 77 
			/// </summary>
			SEX_MALE = 77,
			/// <summary>
			/// Female, character 'F' value 70
			/// </summary>
			_FEMALE = 70,
			/// <summary>
			/// Unknown, character 'U' value 85
			/// </summary>
			_UNKNOWN = 85
		}
/*
		/// <summary>
		/// IImageheader implementation. Gives access to header fields.
		/// </summary>
		/// <param name="hf">Header field</param>
		/// <returns>Header field value</returns>
		public virtual HeaderValue this[HeaderValue.HeaderField hf]
		{
			get
			{
				switch (hf)
				{
					case HeaderValue.HeaderField.HALFLIFE: return TPClib.Isotope.GetHalflife(this.Isotope);
					case HeaderValue.HeaderField.ISOTOPE: return TPClib.Isotope.ConvertToString(this.Isotope);
					case HeaderValue.HeaderField.MODALITY: return Enum.GetName(typeof(ImageModality), this.Modality);
					case HeaderValue.HeaderField.DATATYPE: return this.Datatype.ToString();
					case HeaderValue.HeaderField.HEIGHT: return this.Height;
					case HeaderValue.HeaderField.WIDTH: return this.Width;
					case HeaderValue.HeaderField.PLANES: return this.Planes;
					case HeaderValue.HeaderField.FRAMES: return this.Frames;
					case HeaderValue.HeaderField.VOXELWIDTH: return this.SizeX;
					case HeaderValue.HeaderField.VOXELHEIGHT: return this.SizeY;
					case HeaderValue.HeaderField.VOXELTHICKNESS: return this.SizeZ;
					case HeaderValue.HeaderField.DYNAMIC: return this.DynamicHeader;
					default: return HeaderValue.EmptyValue;
				}
			}
		}
*/
		#endregion
	}
}
