/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Image
{
    /// <summary>
    /// Object can be read from start to end, but there is no indexing possibility
    /// </summary>
    public interface IReadable<T> where T : struct, IConvertible, IComparable
    {
        /// <summary>
        /// Moves the reading pointer back to beginning
        /// </summary>
        void ResetRead();

        /// <summary>
        /// Reads values from the object
        /// </summary>
        /// <param name="buffer">Buffer, where the read data is put into</param>
        /// <param name="index">Index in the destination buffer, where the data is put</param>
        /// <param name="length">Length of reading</param>
        void Read(T[] buffer, uint index, uint length);
    }

    /// <summary>
    /// Object can be read and indexed
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public interface IArray<T> : IReadable<T> where T : struct, IConvertible, IComparable
    {
        /// <summary>
        /// Total amount of objects in the source array
        /// </summary>
        uint Length { get; }

        /// <summary>
        /// Gets values from the source object
        /// </summary>
        /// <param name="index">Index number</param>
        T this[uint index] { get; }

        /// <summary>
        /// Reading function with index position
        /// </summary>
        /// <param name="buffer">Buffer, where the read data is put into</param>
        /// <param name="index">Index in the destination buffer, where the data is put</param>
        /// <param name="length">Length of reading</param>
        /// <param name="position">Starting position in the source array</param>
        void Read(T[] buffer, uint index, uint length, uint position);
    }

    /// <summary>
    /// Object has many dimensions. It can be read as one block + indexed with coordinates
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public interface IMultiDimensionArray<T> : IArray<T> where T : struct, IConvertible, IComparable
    {
        /// <summary>
        /// Dimensions
        /// </summary>
        uint[] Dimensions { get; }

        /// <summary>
        /// Gets values from source object with coordinates 
        /// </summary>
        /// <param name="coordinates">Coordinates of the value</param>
        T this[uint[] coordinates] { get; }
    }

    /// <summary>
    /// Object can be read, indexed and written
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public interface IWritableArray<T> : IArray<T> where T : struct, IConvertible, IComparable
    {
        /// <summary>
        /// Gets / Sets values from/to the source object
        /// </summary>
        /// <param name="index">Index number</param>
        new T this[uint index] { get; set; }
    }

    /// <summary>
    /// Object has many dimensions. It can be read and written as one block + indexed with coordinates 
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public interface IMultiDimensionWritableArray<T> : IMultiDimensionArray<T>, IWritableArray<T> where T : struct, IConvertible, IComparable
    {
        /// <summary>
        /// Gets/Sets values from/to source object with coordinates 
        /// </summary>
        /// <param name="coordinates">Coordinates of the value</param>
        new T this[uint[] coordinates] { get; set; }
    }
}
