/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System.Collections;
using System;

namespace TPClib.Image
{
    /// <summary>
    /// Binary 3D image containing only logical (1 or 0) values in its voxels. 
    /// Bitwise logical operations can be applied with MaskImage objects.
    /// </summary>
    public class MaskImage: Image
    {
        /// <summary>
        /// Binary values in bit array for optimized memory usage and logical operations speed.
        /// </summary>
        private BitArray data;

		/// <summary>
        /// Length of tacs in image.
        /// </summary>
        public override int DataLength
        {
            get
            {
                return data.Length;
            }
        }
        
		/// <summary>
        /// Constructs image with desired dimensions. Image
        /// has zero at all voxels.
        /// </summary>
        /// <param name="dimx">x-dimension</param>
        /// <param name="dimy">y-dimension</param>
        /// <param name="dimz">z-dimension</param>
		public MaskImage(int dimx, int dimy, int dimz) : this(new int[] { dimx, dimy, dimz }) { }
        
		/// <summary>
        /// Constructs image with desired dimensions. Image
        /// has zero at all voxels.
        /// </summary>
        /// <param name="dimension">dimension values</param>
        public MaskImage(int[] dimension)
        {
            if (dimension.Length > 6)
                throw new TPCInvalidArgumentsException("Only maximum of 6-dimensional images are supported by this class.");
            int product = 1;
            for (int i = 0; i < dimension.Length; i++)
            {
                if (dimension[i] < 0)
                    throw new TPCInvalidArgumentsException("Negative dimension value for creation of image.");
                product *= dimension[i];
            }
            GC.Collect();
            try
            {
                data = new BitArray(product);
            }
            catch (OutOfMemoryException) {
                GC.Collect();
                throw new TPCException("Cannot create image because there is not enough memory.");
            }
            for (int i = 0; i < dimension.Length; i++)
                this.dims.AddLimit(0, dimension[i]);
        }
        
		/// <summary>
        /// Constructs image with desired dimensions. Image
        /// has zero at all voxels.
        /// </summary>
        /// <param name="dimension">dimension object, dimension values will be converted to integer values</param>
        public MaskImage(IntLimits dimension)
        {
            if (dimension.Length > 6) throw new TPCInvalidArgumentsException("Only maximum of 6-dimensional images are supported by this class.");
            for (int i = 0; i < dimension.Length; i++)
                if (dimension.GetLimit(i, Limits.Limit.LOW) != 0)
                    throw new TPCInvalidArgumentsException(i + "'s low limit was non-zero while creating an image:" + dimension.GetLimit(i, Limits.Limit.LOW));
            GC.Collect();
            data = null;
            data = new BitArray((int)dimension.GetProduct());
            for (int i = 0; i < dimension.Length; i++)
                Dim.AddLimit(dimension.GetLimit(i, Limits.Limit.LOW), dimension.GetLimit(i, Limits.Limit.HIGH));
        }
        
		/// <summary>
        /// Voxels of image as an array.
        /// </summary>
        /// <param name="location">index number (0-based)</param>
        /// <returns>value at index </returns>
        public override float this[int location]
        {
            get
            {
                return (data[location] ? 1.0f : 0.0f);
            }
            set
            {
                data[location] = (value != 0 ? true : false);
            }
        }
		
        /// <summary>
        /// Bitwise AND operator of image voxel-by-voxel.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">applied image</param>
        /// <returns>multiplied image</returns>
        public static MaskImage operator &(MaskImage image, MaskImage value)
        {
            if (image.Dim != value.Dim)
                throw new TPCException("Cannot apply arithmetic - operation because image sizes are not the same");
            MaskImage r = new MaskImage(image.Dim);
            r.data = image.data.And(value.data);
            return r;
        }
        
		/// <summary>
        /// Bitwise OR operator of image voxel-by-voxel.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">applied image</param>
        /// <returns>resulting image</returns>
        public static MaskImage operator |(MaskImage image, MaskImage value)
        {
            if (image.DataLength != value.DataLength)
                throw new TPCException("Cannot apply arithmetic - operation because image tacs lengths [" + image.DataLength + "] and [" + value.DataLength + "] are not the same");
            MaskImage r = new MaskImage(image.Dim);
            r.data = image.data.Or(value.data);
            return r;
        }
        
		/// <summary>
        /// Bitwise XOR (exclusive or) operator of image voxel-by-voxel.
        /// </summary>
        /// <param name="image">target image</param>
        /// <param name="value">applied image</param>
        /// <returns>resulting image</returns>
        public static MaskImage operator ^(MaskImage image, MaskImage value)
        {
            if (image.DataLength != value.DataLength)
                throw new TPCException("Cannot apply arithmetic - operation because image tacs lengths [" + image.DataLength + "] and [" + value.DataLength + "] are not the same");
            MaskImage r = new MaskImage(image.Dim);
            r.data = image.data.Xor(value.data);
            return r;
        }
        
		/// <summary>
        /// Multiplies all data with value
        /// </summary>
        /// <param name="value">multiplication value</param>
        public override void Multiply(float value)
        {
            if (value == 0) data.SetAll(false);
        }
        
		/// <summary>
        /// Returns minimum and maximum tacs value
        /// </summary>
        /// <returns>array containing [minimum, maximum] value</returns>
        public override float[] GetMinMax()
        {
            if (data.Length == 0) 
                throw new TPCException("Cannot get minimum and maximum because there are no values in the image.");

            float max = 0.0f;
            float min = 1.0f;
            foreach (bool b in data) {
                if (b) max = 1.0f;
                else min = 0.0f;
            }
            return new float[2] { min, max };
        }
        
		/// <summary>
        /// Returns maximum data value
        /// </summary>
        /// <returns>1.0 if there are one or more ones in the image</returns>
        public override float GetMax()
        {
            foreach (bool b in data) if (b) return 1.0f;
            return 0.0f;
        }
        
		/// <summary>
        /// Returns minimum data value
        /// </summary>
        /// <returns>0.0 if there are one or more zeros in the image</returns>
        public override float GetMin()
        {
            foreach (bool b in data) if (!b) return 0.0f;
            return 1.0f;
        }
        
		/// <summary>
        /// Returns the volume of mask.
        /// </summary>
        /// <returns>number of non-zero elements in this mask</returns>
        public int GetVolume()
        {
            int volume = 0;
            for (int i = 0; i < data.Length; i++)
                if (data[i]) volume++;
            return volume;
        }
        
		/// <summary>
        /// Returns the mean data value.
        /// </summary>
        /// <returns>proportion of ones to total number of voxels</returns>
        public override float GetMean()
        {
            if (data.Length == 0)
                throw new TPCException("Cannot calculate mean of image because there is not voxels.");
            float m = 0.0f;
            for (int i = 0; i < data.Length; i++)
                if (data[i]) m++;
            return m / data.Length;
        }
        
		/// <summary>
        /// Copies mask data into 1-dimensional array
        /// </summary>
        /// <param name="array">target array</param>
        public void CopyTo(Array array) {
            data.CopyTo(array, 0);
        }
		
		/// <summary>
        /// Returns a String that represents the current Object.
        /// </summary>
        /// <returns>A String that represents the current Object.</returns>
        public override string ToString()
        {
            return "MaskImage[" + Dim + ",min=" + GetMin() + ",max=" + GetMax() + "]";
        }
        
		/// <summary>
        /// Evaluates if images equals. Currently only image dimensions and tacs is evaluated.
        /// </summary>
        /// <param name="obj">evaluated object</param>
        /// <returns>false if files differ</returns>
        public override bool Equals(object obj)
        {
            if (obj is MaskImage)
            {
                MaskImage img = (obj as MaskImage);
                if (Dim != img.Dim) return false;
                for (int i = 0; i < data.Length; i++)
                    if (data[i] != img.data[i]) return false;
                return true;
            }
            else
            {
                return false;
            }
        }
        
		/// <summary>
        /// Override that just calls base class method.
        /// </summary>
        /// <returns>hash code of this object</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}
