/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System.Collections.Generic;
using System.IO;
using System;

namespace TPClib.Image
{
	/// <summary>
	/// 
	/// </summary>
	public class BlockStream : Stream
	{
		#region Class DataBlock

		private class DataBlock
		{
			/// <summary></summary>
			public Stream stream;

			/// <summary></summary>
			public int length;

			/// <summary></summary>
			public long start;

			private byte[] blockBuffer = null;
			private bool writePending = false;
			private int writeStart = 0;
			private int writeLength = 0;

			/// <summary>
			/// 
			/// </summary>
			/// <param name="s"></param>
			/// <param name="st"></param>
			/// <param name="ln"></param>
			public DataBlock(Stream s, long st, int ln)
			{
				stream = s;
				length = ln;
				start = st;
			}

			/// <summary>
			/// 
			/// </summary>
			public bool DataAvailable { get { return blockBuffer != null || stream.CanRead; } }

			/// <summary>
			/// 
			/// </summary>
			public void Flush()
			{
				if (writePending)
				{
					long pos = stream.Position;
					stream.Seek(start + writeStart, SeekOrigin.Begin);
					stream.Write(blockBuffer, writeStart, writeLength);
					stream.Seek(pos, SeekOrigin.Begin);
					writePending = false;
				}
				blockBuffer = null;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="buffer"></param>
			/// <param name="offset"></param>
			/// <param name="count"></param>
			/// <param name="position"></param>
			/// <returns></returns>
			public int Read(byte[] buffer, int offset, int count, int position)
			{
				if (blockBuffer == null)
				{
					blockBuffer = new byte[length];
					stream.Seek(start, SeekOrigin.Begin);
					stream.Read(blockBuffer, 0, length);
				}
				int bytesRead = length - position > count ? length - position : count;
				Buffer.BlockCopy(blockBuffer, position, buffer, offset, bytesRead);
				return bytesRead;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="buffer"></param>
			/// <param name="offset"></param>
			/// <param name="count"></param>
			/// <param name="position"></param>
			/// <returns></returns>
			public int Write(byte[] buffer, int offset, int count, int position)
			{
				int bytesWritten = count + position > length ? length - position : count;
				Buffer.BlockCopy(buffer, offset, blockBuffer, position, bytesWritten);
				writePending = true;
				writeStart = position;
				writeLength = bytesWritten;
				return bytesWritten;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="l"></param>
			public void Resize(int l)
			{
				if (blockBuffer != null)
				{
					this.Flush();
				}
				this.length = l;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public DataBlock Split()
			{
				int tailLength = this.length / 2;
				long tailStart = this.start + this.length - tailLength;
				this.length = this.length - tailLength;
				DataBlock tail = new DataBlock(this.stream, tailStart, tailLength);
				if (this.blockBuffer != null)
				{
					tail.blockBuffer = new byte[tail.length];
					Buffer.BlockCopy(this.blockBuffer, this.length, tail.blockBuffer, 0, tail.length);
				}
				if (this.writePending && this.writeStart > this.length)
				{
					tail.writePending = true;
					tail.writeStart = this.writeStart - this.length;
					tail.writeLength = this.writeLength;
					this.writePending = false;
				}
				return tail;
			}
		}

		#endregion

		#region Private fields

		private System.Collections.Generic.LinkedList<DataBlock> blockList = new System.Collections.Generic.LinkedList<DataBlock>();

		private System.Collections.Generic.LinkedListNode<DataBlock> currentBlock;

		private long totalLength = 0;

		private long position = 0;

		private int blockPosition = 0;

		#endregion

		#region Constructors

		/// <summary>
		/// 
		/// </summary>
		public BlockStream(int max = Int32.MaxValue) { maxLength = max; }

		#endregion

		#region Stream interface

		/// <summary>
		/// 
		/// </summary>
		public override bool CanRead
		{
			get { return (currentBlock.Value.DataAvailable); }
		}

		/// <summary>
		/// 
		/// </summary>
		public override bool CanSeek
		{
			get { return true; }
		}

		/// <summary>
		/// 
		/// </summary>
		public override bool CanWrite
		{
			get { return currentBlock.Value.stream.CanWrite; }
		}

		/// <summary>
		/// 
		/// </summary>
		public override void Flush()
		{
			foreach (DataBlock db in blockList)
			{
				db.Flush();
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public override long Length
		{
			get { return totalLength; }
		}

		/// <summary>
		/// 
		/// </summary>
		public override long Position
		{
			get
			{
				return position;
			}
			set
			{
				Seek(value, SeekOrigin.Begin);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="buffer"></param>
		/// <param name="offset"></param>
		/// <param name="count"></param>
		/// <returns></returns>
		public override int Read(byte[] buffer, int offset, int count)
		{
			int bytesRead = currentBlock.Value.Read(buffer, offset, count, blockPosition);
			int lastRead = bytesRead;
			while (bytesRead < count)
			{
				blockPosition = 0;
				currentBlock = currentBlock.Next;
				lastRead = currentBlock.Value.Write(buffer, offset + bytesRead, count - bytesRead, 0);
				bytesRead = lastRead;
			}
			blockPosition = lastRead;
			return bytesRead;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="offset"></param>
		/// <param name="origin"></param>
		/// <returns></returns>
		public override long Seek(long offset, SeekOrigin origin)
		{
			long trueOffset;
			switch (origin)
			{
				case SeekOrigin.End:
					trueOffset = this.Length + offset;
					break;
				case SeekOrigin.Current:
					trueOffset = this.position + offset;
					break;
				default:
					trueOffset = offset;
					break;
			}
			currentBlock = blockList.First;
			while (currentBlock.Value.length < trueOffset)
			{
				trueOffset -= currentBlock.Value.length;
				currentBlock = currentBlock.Next;
			}
			blockPosition = (int)trueOffset;
			return this.position;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void SetLength(long value)
		{
			long diff = value - this.totalLength;

			if (diff > 0)
			{
				blockList.Last.Value.Resize(blockList.Last.Value.length + (int)diff);
				if (blockList.Last.Value.length > maxLength)
				{
					blockList.AddLast(blockList.Last.Value.Split());
				}
			}
			else if (diff < 0)
			{
				while (blockList.Last.Value.length < -diff)
				{
					diff += blockList.Last.Value.length;
					blockList.Last.Value.Flush();
					blockList.RemoveLast();
				}
				blockList.Last.Value.Resize((int)diff);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="buffer"></param>
		/// <param name="offset"></param>
		/// <param name="count"></param>
		public override void Write(byte[] buffer, int offset, int count)
		{
			int bytesWritten = currentBlock.Value.Write(buffer, offset, count, blockPosition);
			int lastWrite = bytesWritten;
			while (bytesWritten < count)
			{
				blockPosition = 0;
				currentBlock = currentBlock.Next;
				lastWrite = currentBlock.Value.Write(buffer, offset + bytesWritten, count - bytesWritten, 0);
				bytesWritten = lastWrite;
			}
			blockPosition = lastWrite;
		}

		#endregion

		#region Public

		/// <summary>
		/// 
		/// </summary>
		public readonly int maxLength;

		/// <summary>
		/// 
		/// </summary>
		/// <param name="s"></param>
		/// <param name="start"></param>
		/// <param name="length"></param>
		public void AddBlock(Stream s, int start, int length)
		{
			this.AddBlock(new DataBlock(s, start, length));
			totalLength += length;
		}

		#endregion

		#region Private methods

		private void AddBlock(DataBlock db)
		{
			if (db.length > maxLength)
			{
				DataBlock tailBlock = db.Split();
				this.AddBlock(db);
				this.AddBlock(tailBlock);
			}
			else blockList.AddLast(db);
		}

		#endregion
	}
}
