/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// Flipped image stream
	/// </summary>
	public class FlippedStream : Stream
	{
		#region Private variables

		private Stream stream;

		private byte[] internalBuffer;
		private int bufferIndex;

		private int blockSize;

		#endregion

		#region Constructors

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="s">Stream to flip</param>
		/// <param name="bSize">Size of the blocks</param>
		/// <param name="bCount">Number of the blocks per flip</param>
		public FlippedStream(Stream s, int bSize, int bCount)
		{
			stream = s;
			blockSize = bSize;
			internalBuffer = new byte[bCount * bSize];

			// Set the index to the end of the buffer to indicate no usable bytes
			bufferIndex = internalBuffer.Length;

			// Seek to the start of the stream
			s.Seek(0, SeekOrigin.Begin);
		}

		#endregion

		#region Stream interface

		/// <summary>
		/// Size of the buffer used by this stream.
		/// </summary>
		public int BufferSize
		{
			get { return internalBuffer.Length; }
		}

		/// <summary>
		/// Read bytes to a buffer. No range checking, buffer must hold at least offset + count bytes.
		/// </summary>
		/// <param name="buffer">Buffer for the bytes read</param>
		/// <param name="offset">Starting position in the buffer for the bytes read</param>
		/// <param name="count">Number of bytes to read</param>
		/// <returns>Number of bytes actually read from the stream</returns>
		public override int Read(byte[] buffer, int offset, int count)
		{
			if (BytesInBuffer == 0) ReadBuffer();

			// Copy the remaining buffer
			int bytesCopied = count > BytesInBuffer ? BytesInBuffer : count;
			Buffer.BlockCopy(internalBuffer, bufferIndex, buffer, offset, bytesCopied);
			bufferIndex += bytesCopied;
			offset += bytesCopied;
			count -= bytesCopied;

			// Copy full buffers
			while (count >= BufferSize)
			{
				ReadBuffer();
				Buffer.BlockCopy(internalBuffer, 0, buffer, offset, BufferSize);
				bytesCopied += BufferSize;
				offset += BufferSize;
				count -= BufferSize;
				bufferIndex = BufferSize;
			}

			// Copy the remaining bytes, if needed
			if (count > 0) bytesCopied += this.Read(buffer, offset, count);

			return bytesCopied;
		}

		/// <summary>
		/// Write bytes from a buffer. No range checking, buffer must hold at least offset + count bytes.
		/// </summary>
		/// <param name="buffer">Buffer to write</param>
		/// <param name="offset">Starting position in the buffer</param>
		/// <param name="count">Number of bytes to write</param>
		public override void Write(byte[] buffer, int offset, int count)
		{
			throw new NotImplementedException();
		}

		/// <summary>
		/// Stream can read.
		/// </summary>
		public override bool CanRead
		{
			get { return stream.CanRead; }
		}

		/// <summary>
		/// Stream can seek.
		/// </summary>
		public override bool CanSeek
		{
			get { return true; }
		}

		/// <summary>
		/// Stream can write.
		/// </summary>
		public override bool CanWrite
		{
			get { return false; }
		}

		/// <summary>
		/// Flush the stream. All pending write operations will be done and internal buffers emptied.
		/// </summary>
		public override void Flush()
		{
			stream.Flush();
		}

		/// <summary>
		/// 
		/// </summary>
		public override long Length
		{
			get { return stream.Length; }
		}

		/// <summary>
		/// 
		/// </summary>
		public override long Position
		{
			get
			{
				return stream.Position - BytesInBuffer;
			}
			set
			{
				this.Seek(value, SeekOrigin.Begin);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="offset"></param>
		/// <param name="origin"></param>
		/// <returns></returns>
		public override long Seek(long offset, SeekOrigin origin)
		{
			long seekDistance = offset;
			switch (origin)
			{
				case SeekOrigin.Begin:
					seekDistance -= this.Position;
					break;
				case SeekOrigin.End:
					seekDistance += this.Length - this.Position;
					break;
				default:
					break;
			}

			// Move within buffer if possible
			if (seekDistance >= 0 && seekDistance < BytesInBuffer)
			{
				bufferIndex += (int)seekDistance;
			}
			else if (seekDistance < 0 && seekDistance > -bufferIndex)
			{
				bufferIndex -= (int)seekDistance;
			}
			// Position outside current buffer
			else
			{
				long trueOffset = this.Position + seekDistance;
				long blocks = trueOffset / internalBuffer.Length;
				stream.Seek(internalBuffer.Length * blocks, SeekOrigin.Begin);
				ReadBuffer();
				bufferIndex = (int)(trueOffset % internalBuffer.Length);
			}

			return this.Position;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		public override void SetLength(long value)
		{
			throw new NotSupportedException();
		}

		#endregion

		#region Private / Protected

		private int BytesInBuffer { get { return internalBuffer.Length - bufferIndex; } }

		private int ReadBuffer()
		{
			stream.Read(internalBuffer, 0, internalBuffer.Length);

			byte[] temp = new byte[blockSize];
			for (int src = 0, dest = internalBuffer.Length - blockSize; src < internalBuffer.Length / 2; src += blockSize, dest -= blockSize)
			{
				Buffer.BlockCopy(internalBuffer, src, temp, 0, blockSize);
				Buffer.BlockCopy(internalBuffer, dest, internalBuffer, src, blockSize);
				Buffer.BlockCopy(temp, 0, internalBuffer, dest, blockSize);
			}

			bufferIndex = 0;
			return internalBuffer.Length;
		}

		/// <summary>
		/// Dispose of this stream
		/// </summary>
		/// <param name="disposing">True, if called by user</param>
		protected override void Dispose(bool disposing)
		{
			if (disposing)
			{
				stream.Dispose();
			}
			base.Dispose(disposing);
		}

		#endregion
	}
}
