/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// Non-continuous stream
	/// </summary>
	public class FragmentedStream : Stream
	{
		#region Private variables

		private long[] dataStarts;

		private long[] dataLengths;

		private long totalLength = 0;

		private Stream stream;

		private long currentPosition = 0;

		private int currentBlock = 0;

		private long blockPosition = 0;

		#endregion

		#region Constructors

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="s">Stream</param>
		/// <param name="starts">Starting bytes for stream segments</param>
		/// <param name="lengths">Lengths of the stream segments</param>
		public FragmentedStream(Stream s, long[] starts, long[] lengths)
		{
			dataStarts = starts;
			dataLengths = lengths;
			stream = s;
			foreach (long l in lengths) totalLength += l;
		}

		#endregion

		#region Stream interface

		/// <summary>
		/// Stream can read.
		/// </summary>
		public override bool CanRead
		{
			get { return stream.CanRead; }
		}

		/// <summary>
		/// Stream can seek.
		/// </summary>
		public override bool CanSeek
		{
			get { return stream.CanSeek; }
		}

		/// <summary>
		/// Stream can write.
		/// </summary>
		public override bool CanWrite
		{
			get { return stream.CanWrite; }
		}

		/// <summary>
		/// Flush the stream.
		/// </summary>
		public override void Flush()
		{
			stream.Flush();
		}

		/// <summary>
		/// Stream length.
		/// </summary>
		public override long Length
		{
			get { return totalLength; }
		}

		/// <summary>
		/// Stream byte position.
		/// </summary>
		public override long Position
		{
			get
			{
				return currentPosition;
			}
			set
			{
				this.Seek(value, SeekOrigin.Begin);
			}
		}

		/// <summary>
		/// Read bytes into a buffer. No range checking, buffer must hold at least offset + count bytes.
		/// </summary>
		/// <param name="buffer">Buffer for the data read</param>
		/// <param name="offset">Starting byte position in buffer</param>
		/// <param name="count">Number of bytes to read into the buffer</param>
		/// <returns>Number of bytes actually read</returns>
		public override int Read(byte[] buffer, int offset, int count)
		{
			long currentBytes = dataLengths[currentBlock] - blockPosition;
			int maxBytes = count > currentBytes ? (int)currentBytes : count;
			int bytesRead = stream.Read(buffer, offset, maxBytes);
			currentPosition += bytesRead;

			if (bytesRead < count)
			{
				SeekToNextBlock();
				bytesRead += this.Read(buffer, offset + bytesRead, count - bytesRead);
			}
			return bytesRead;
		}

		/// <summary>
		/// Seek to a position in this stream.
		/// </summary>
		/// <param name="offset">Offset in bytes from the seek origin</param>
		/// <param name="origin">Seek origin</param>
		/// <returns>Stream position after the seek</returns>
		public override long Seek(long offset, SeekOrigin origin)
		{
			// Calculate offset from the stream start
			switch (origin)
			{
				case SeekOrigin.Current:
					offset += this.Position;
					break;
				case SeekOrigin.End:
					offset += this.Length;
					break;
				default:
					break;
			}

			if (offset < 0 || offset > this.Length) throw new IOException("Can not seek to position" + offset + ": (outside the stream)");

			// Calculate the correct block
			long lengthSum = 0;
			int index;
			for (index = 0; index < dataLengths.Length && lengthSum + dataLengths[index] <= offset; index++)
			{
				lengthSum += dataLengths[index];
			}

			// Offset in the correct data block
			long blockOffset = offset - lengthSum;

			// Offset in the underlying stream
			long trueOffset = dataStarts[index] + blockOffset;

			// Seek to the correct stream position
			stream.Seek(trueOffset, SeekOrigin.Begin);

			// Update position info
			currentPosition = offset;
			currentBlock = index;
			blockPosition = blockOffset;

			// Return curren position
			return currentPosition;
		}

		/// <summary>
		/// Set the length of this stream.
		/// </summary>
		/// <param name="value">New length of the stream</param>
		public override void SetLength(long value)
		{
			throw new NotSupportedException();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="buffer"></param>
		/// <param name="offset"></param>
		/// <param name="count"></param>
		public override void Write(byte[] buffer, int offset, int count)
		{
			long currentBytes = dataLengths[currentBlock] - blockPosition;
			int maxBytes = count > currentBytes ? (int)currentBytes : count;
			stream.Write(buffer, offset, maxBytes);
			currentPosition += maxBytes;

			if (maxBytes < count)
			{
				SeekToNextBlock();
				this.Write(buffer, offset + maxBytes, count - maxBytes);
			}
		}

		#endregion

		#region Private/Protected

		/// <summary>
		/// Dispose of the stream
		/// </summary>
		/// <param name="disposing"></param>
		protected override void Dispose(bool disposing)
		{
			if (disposing) stream.Dispose();
			base.Dispose(disposing);
		}

		private void SeekToNextBlock()
		{
			stream.Seek(dataStarts[++currentBlock], SeekOrigin.Begin);

			currentPosition = 0;
			for (int i = 0; i < currentBlock; i++)
			{
				currentPosition += dataLengths[i];
			}
			blockPosition = 0;
		}

		#endregion
	}
}
