/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image
{
	/// <summary>
	/// Concatenates several files into a single stream
	/// </summary>
	public class MultiFileStream : Stream
	{
		#region Private variables

		private long[] lengths;

		private long[] dataStarts;

		private string[] fileNames;

		private long totalLength;

		private Stream currentStream = null;

		private int currentFile = 0;

		private long currentPosition = 0;

		private int files;

		private FileMode openMode = FileMode.OpenOrCreate;

		private FileAccess accessMode = FileAccess.Read;

		#endregion

		#region Stream interface

		/// <summary>
		/// Stream length
		/// </summary>
		public override long Length
		{
			get { return totalLength; }
		}

		/// <summary>
		/// Current stream position
		/// </summary>
		public override long Position
		{
			get
			{
				return currentPosition;
			}
			set
			{
				this.Seek(value, SeekOrigin.Begin);
			}
		}

		/// <summary>
		/// Stream can read
		/// </summary>
		public override bool CanRead
		{
			get { return currentStream.CanRead; }
		}

		/// <summary>
		/// Stream can write
		/// </summary>
		public override bool CanWrite
		{
			get { return currentStream.CanWrite; }
		}

		/// <summary>
		/// Stream can seek
		/// </summary>
		public override bool CanSeek
		{
			get { return currentStream.CanSeek; }
		}

		/// <summary>
		/// Stream timeouts
		/// </summary>
		public override bool CanTimeout
		{
			get { return currentStream.CanTimeout; }
		}

		/// <summary>
		/// Seek a position in this stream
		/// </summary>
		/// <param name="offset">Position offset from the seek origin</param>
		/// <param name="origin">Base point for the seek offset</param>
		/// <returns>Stream position after seeking</returns>
		public override long Seek(long offset, SeekOrigin origin)
		{
			long truePosition = offset;
			switch (origin)
			{
				case SeekOrigin.Current:
					truePosition += this.Position;
					break;
				case SeekOrigin.End:
					truePosition += this.Length;
					break;
				case SeekOrigin.Begin:
				default:
					break;
			}
			int seekIndex = 0;
			long seekLength = 0;
			while (seekIndex < lengths.Length && truePosition - seekLength > lengths[seekIndex])
			{
				seekLength += lengths[seekIndex++];
			}
			ChangeStream(seekIndex);

			// Set the current position; remember not to include the bytes before the data starts
			currentPosition += currentStream.Seek(truePosition - seekLength, SeekOrigin.Current) - dataStarts[seekIndex];
			return currentPosition;
		}

		/// <summary>
		/// Set the stream length
		/// </summary>
		/// <param name="value">New stream length</param>
		public override void SetLength(long value)
		{
			throw new NotImplementedException();
		}

		/// <summary>
		/// Read from the stream. Output buffer is assumed to be at least offset+count bytes long. This is not checked here.
		/// </summary>
		/// <param name="buffer">Output buffer</param>
		/// <param name="offset">Offset from the start of the output buffer</param>
		/// <param name="count">Number of bytes to read from the stream</param>
		/// <returns>Number of bytes actually read</returns>
		public override int Read(byte[] buffer, int offset, int count)
		{
			long maxBytes = lengths[currentFile] - (currentStream.Position - dataStarts[currentFile]);
			int bytesToRead = maxBytes > count ? count : (int)maxBytes;

			int bytesRead = currentStream.Read(buffer, offset, bytesToRead);
			currentPosition += bytesRead;
			if (bytesRead < count && NextStream())
			{
				bytesRead += this.Read(buffer, offset + bytesRead, count - bytesRead);
			}
			return bytesRead;
		}

		/// <summary>
		/// Write to the stream. Input buffer is assumed to have at least offset+count bytes, length is not checked.
		/// </summary>
		/// <param name="buffer">Input buffer</param>
		/// <param name="offset">Offset from the start of the input buffer</param>
		/// <param name="count">Bytes to write from the input buffer</param>
		public override void Write(byte[] buffer, int offset, int count)
		{
			long bytesLeft = lengths[currentFile] - (currentStream.Position - dataStarts[currentFile]);
			int bytesToWrite = count > bytesLeft ? (int)bytesLeft : count;

			currentStream.Write(buffer, offset, bytesToWrite);

			if (bytesLeft < count && NextStream())
			{
				this.Write(buffer, offset + bytesToWrite, count - bytesToWrite);
			}
			currentPosition += count;
		}

		/// <summary>
		/// Flush the internal stream buffer. Stream position is moved to the start of the next block.
		/// </summary>
		public override void Flush()
		{
			currentStream.Flush();
		}

		/// <summary>
		/// Close the stream.
		/// </summary>
		public override void Close()
		{
			currentStream.Close();
		}

		#endregion

		#region Constructors

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="fnames">File names</param>
		public MultiFileStream(string[] fnames, FileAccess a = FileAccess.ReadWrite) : this(fnames, 0, 0) { }

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="fnames">File names</param>
		/// <param name="head">Offset to the start of the image data from the start of each file</param>
		/// <param name="tail">Offset to the end of the files from the end of the data in each file</param>
		public MultiFileStream(string[] fnames, long head, long tail, FileAccess a = FileAccess.ReadWrite)
		{
			fileNames = fnames;
			lengths = new long[fnames.Length];
			dataStarts = new long[fnames.Length];
			files = fnames.Length;
			accessMode = a;

			totalLength = 0;
			for (int i = 0; i < fnames.Length; i++)
			{
				FileInfo fi = new FileInfo(fnames[i]);
				dataStarts[i] = head;
				lengths[i] = fi.Length - head - tail;
				totalLength += lengths[i];
			}
			ChangeStream(0);
		}

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="fnames">File names</param>
		/// <param name="heads">Offsets to the start of the image data from the start of each file</param>
		/// <param name="ends">Offsets to the end of the files from the end of the data in each file</param>
		/// <param name="endsAreOffsets">If ends values are offsets from file end, set true (default). If ends are datalengths, set to false.</param>
		public MultiFileStream(string[] fnames, long[] heads, long[] ends, bool endsAreOffsets = true, FileAccess a = FileAccess.ReadWrite)
		{
			if (fnames.Length != heads.Length || fnames.Length != ends.Length)
				throw new ArgumentException("Argument array lengths do not match");

			fileNames = fnames;
			lengths = new long[fnames.Length];
			dataStarts = new long[fnames.Length];
			files = fnames.Length;
			accessMode = a;

			totalLength = 0;
			for (int i = 0; i < fnames.Length; i++)
			{
				FileInfo fi = new FileInfo(fnames[i]);
				dataStarts[i] = heads[i];
				if (endsAreOffsets) lengths[i] = fi.Length - heads[i] - ends[i];
				else lengths[i] = ends[i];
				totalLength += lengths[i];
			}
			ChangeStream(0);
		}

		#endregion

		#region Private

		/// <summary>
		/// Change the current stream. Closes any open streams and seeks to the start of data.
		/// </summary>
		/// <param name="index">Index of the file to open</param>
		/// <returns>True, if the stream was succesfully opened</returns>
		private bool ChangeStream(int index)
		{
			if (index >= files || index < 0) return false;

			if (index != currentFile && currentStream != null)
			{
				currentStream.Close();
			}
			if (index != currentFile || currentStream == null)
			{
				currentStream = new FileStream(fileNames[index], openMode, accessMode);
			}

			long newPosition = 0;
			for (int i = 0; i < index; i++) newPosition += lengths[i];

			currentPosition = newPosition;

			currentStream.Seek(dataStarts[index], SeekOrigin.Begin);
			currentFile = index;

			return true;
		}

		/// <summary>
		/// Open the next stream in the list and seeks to the start of data.
		/// </summary>
		/// <returns>True, if the stream was succesfully opened</returns>
		private bool NextStream()
		{
			return ChangeStream(currentFile + 1);
		}

		#endregion
	}
}
