/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.IO;

namespace TPClib.Image.Streaming
{
	/// <summary>
	/// 
	/// </summary>
	public class SubStream : Stream
	{
		#region Private variables

		private enum StreamOperation : byte
		{
			NONE, READ, WRITE
		}

		private StreamOperation pending = StreamOperation.NONE;

		// Underlying file stream
		private Stream fstream;

		// Offset from the start of the underlying file stream
		private int streamOffset;

		// Stream datatype
		private ImageData.DataType dataType;

		// Size of the underlying image
		private IntLimits imageLimits;

		// Size of each dimension in bytes
		private int[] dimensionSize;

		// Limits of the stream window
		private IntLimits windowLimits;
		private int maxDim;
		private int[] windowSize;
		private int[] preBlock;
		private int[] postBlock;

		// Highest dimension with continuous data in the window
		private int continuousDimension;

		// Current internal byteposition inside the stream window
		// Always points to the start of the next unread/unwritten block
		private long windowPosition;

		// Internal buffer
		// bufferIndex will be set at internalBuffer.Length if all bytes from the buffer are already read/written
		private byte[] internalBuffer;
		private int bufferIndex;

		private long streamHeadStart = 0;

		// Maximum internal buffer size
		private const int MAXBUFFER = 0xFFFFF;

		#endregion

		#region Constructors

		/// <summary>
		/// 
		/// </summary>
		/// <param name="fname"></param>
		/// <param name="lim"></param>
		/// <param name="winLim"></param>
		/// <param name="mode"></param>
		/// <param name="head"></param>
		/// <param name="dt"></param>
		public SubStream(string fname, IntLimits lim, IntLimits winLim, int head = 0, ImageFile.DataType dt = ImageFile.DataType.BIT16_S, FileMode mode = FileMode.Open)
			: this(new FileStream(fname, mode), lim, winLim, head, dt) { }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="fname"></param>
		/// <param name="lim"></param>
		public SubStream(string fname, IntLimits lim) : this(fname, lim, lim) { }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="filename"></param>
		/// <param name="header"></param>
		/// <param name="winLim"></param>
		/// <param name="head"></param>
		public SubStream(string filename, ImageHeader header, IntLimits winLim, int head = 0) : this(filename, header.Dim, winLim, head, header.Datatype) { }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="file"></param>
		/// <param name="winLim"></param>
		/// <param name="head"></param>
		public SubStream(ImageFile file, IntLimits winLim, int head = 0) : this(file.filename, file.Header, winLim, head) { }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="s"></param>
		/// <param name="header"></param>
		/// <param name="winLim"></param>
		/// <param name="head"></param>
		public SubStream(Stream s, ImageHeader header, IntLimits winLim, int head = 0) : this(s, header.Dim, winLim, head, header.Datatype) { }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="s"></param>
		/// <param name="lim"></param>
		public SubStream(Stream s, IntLimits lim) : this(s, lim, lim) { }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="s"></param>
		/// <param name="lim"></param>
		/// <param name="winLim"></param>
		/// <param name="head"></param>
		/// <param name="dt"></param>
		/// <param name="littleEndian"></param>
		public SubStream(Stream s, IntLimits lim, IntLimits winLim, int head = 0, ImageFile.DataType dt = ImageFile.DataType.BIT16_S, bool littleEndian = true)
		{
			imageLimits = lim;
			windowLimits = winLim;
			streamOffset = head;

			dataType = new ImageData.DataType();
			dataType.Type = dt;
			dataType.ByteOrder = littleEndian ? ImageData.ByteOrder.LittleEndian : ImageData.ByteOrder.BigEndian;

			Init();

			fstream = s;
			SkipToFirst();
		}

		#endregion

		#region Stream interface

		/// <summary>
		/// End of stream reached
		/// </summary>
		public bool EndOfStream { get { return (this.Position >= this.Length); } }

		/// <summary>
		/// Stream length
		/// </summary>
		public override long Length
		{
			get { return windowSize[maxDim]; }
		}

		/// <summary>
		/// Current stream position
		/// </summary>
		public override long Position
		{
			get
			{
				// Internal position modified by the number of bytes in the buffer
				long pos = windowPosition + bufferIndex;
				if (pending == StreamOperation.READ) pos -= BufferSize;
				return pos;
			}
			set
			{
				Seek(value, SeekOrigin.Begin);
			}
		}

		/// <summary>
		/// Stream can read
		/// </summary>
		public override bool CanRead
		{
			get { return fstream.CanRead; }
		}

		/// <summary>
		/// Stream can write
		/// </summary>
		public override bool CanWrite
		{
			get { return fstream.CanWrite; }
		}

		/// <summary>
		/// Stream can seek
		/// </summary>
		public override bool CanSeek
		{
			get
			{
				return fstream.CanSeek;
			}
		}

		/// <summary>
		/// Stream timeouts
		/// </summary>
		public override bool CanTimeout
		{
			get { return fstream.CanTimeout; }
		}

		/// <summary>
		/// Seek a position in this stream
		/// </summary>
		/// <param name="offset">Position offset from the seek origin</param>
		/// <param name="origin">Base point for the seek offset</param>
		/// <returns>Stream position after seeking</returns>
		public override long Seek(long offset, SeekOrigin origin)
		{
			// Write any pending bytes
			PendingWrite();

			long trueOffset = offset;
			switch (origin)
			{
				case SeekOrigin.Current:
					trueOffset += this.Position;
					break;
				case SeekOrigin.End:
					trueOffset += this.Length;
					break;
				default:
					break;
			}

			// Number of whole blocks in the seek distance
			long blocks = trueOffset / BufferSize;
			long totalSeek = 0;

			// Calculate distance from the start of data
			for (int i = 0; i < blocks; i++)
			{
				totalSeek += BufferSize;
				totalSeek += SkipLength(continuousDimension, totalSeek);
			}

			// Skip to the start of data
			SkipToFirst();

			// Seek the correct block
			fstream.Seek(totalSeek, SeekOrigin.Current);
			windowPosition = blocks * BufferSize;

			// Check if there are any bytes remaining and read a block to the memory if needed
			bufferIndex = (int)(trueOffset - windowPosition);

			if (bufferIndex > 0)
			{
				ReadBlock(internalBuffer, 0);
			}

			return this.Position;
		}

		/// <summary>
		/// Set the stream length
		/// </summary>
		/// <param name="value">New stream length</param>
		public override void SetLength(long value)
		{
			throw new NotSupportedException();
		}

		/// <summary>
		/// Read from the stream. Output buffer is assumed to be at least offset+count bytes long. This is not checked here.
		/// </summary>
		/// <param name="buffer">Output buffer</param>
		/// <param name="offset">Offset from the start of the output buffer</param>
		/// <param name="count">Number of bytes to read from the stream</param>
		/// <returns>Number of bytes actually read</returns>
		public override int Read(byte[] buffer, int offset, int count)
		{
			int byteCount = 0;
			int bytesInBuffer = BufferSize - bufferIndex;

			// Copy all remaining bytes from the read buffer
			if (pending == StreamOperation.READ)
			{
				PendingWrite();

				byteCount = bytesInBuffer > count ? count : bytesInBuffer;
				Buffer.BlockCopy(internalBuffer, bufferIndex, buffer, offset, byteCount);
				bufferIndex += byteCount;
				if (bufferIndex == BufferSize) bufferIndex = 0;
			}

			if (!EndOfStream)
			{
				// Read blocks to the output buffer
				while (count - byteCount >= BufferSize)
				{
					byteCount += ReadBlock(buffer, offset + byteCount);
				}

				// If partial block is still needed, read a block to the internal buffer and copy a part of that.
				if (count > byteCount)
				{
					ReadBlock(internalBuffer, 0);
					bufferIndex = count - byteCount;
					Buffer.BlockCopy(internalBuffer, 0, buffer, offset + byteCount, bufferIndex);
					byteCount = count;
				}
			}

			// Set the last operation status to read, if there are still bytes in the buffer
			if (bufferIndex > 0) pending = StreamOperation.READ;
			else pending = StreamOperation.NONE;

			return byteCount;
		}

		/// <summary>
		/// Write to the stream. Input buffer is assumed to have at least offset+count bytes, length is not checked.
		/// </summary>
		/// <param name="buffer">Input buffer</param>
		/// <param name="offset">Offset from the start of the input buffer</param>
		/// <param name="count">Bytes to write from the input buffer</param>
		public override void Write(byte[] buffer, int offset, int count)
		{
			int byteCount = 0;
			int freeBuffer = BufferSize - bufferIndex;

			// If there are bytes in the internal buffer, replace the unused ones and write if full
			if (pending != StreamOperation.NONE)
			{
				// If last operation was not write, skip back one block,
				// so that the internal buffer is written to the correct position
				if (pending == StreamOperation.READ)
				{
					SkipToPrev();
				}

				byteCount = count > freeBuffer ? freeBuffer : count;
				Buffer.BlockCopy(buffer, offset, internalBuffer, bufferIndex, byteCount);

				// If internal buffer is full, write it
				if (freeBuffer == byteCount)
				{
					WriteBlock(internalBuffer, 0);
				}

				bufferIndex += byteCount;
			}

			// Write whole blocks
			while (count - byteCount >= BufferSize)
			{
				WriteBlock(buffer, offset + byteCount);
				byteCount += BufferSize;
			}

			// If there still are unwritten bytes, store them in the internal buffer
			// This will always result in a partially filled internal buffer
			if (count > byteCount)
			{
				bufferIndex = count - byteCount;
				Buffer.BlockCopy(buffer, offset + byteCount, internalBuffer, 0, bufferIndex);
				pending = StreamOperation.WRITE;
			}
			else pending = StreamOperation.NONE;
		}

		/// <summary>
		/// Flush the internal stream buffer. Stream position is moved to the start of the next block.
		/// </summary>
		public override void Flush()
		{
			PendingWrite();

			// Discard the internal buffer contents and move to the next block
			bufferIndex = BufferSize;

			fstream.Flush();
		}

		/// <summary>
		/// Dispose of this stream
		/// </summary>
		/// <param name="disposing">True, if called by user</param>
		protected override void Dispose(bool disposing)
		{
			if (disposing)
			{
				fstream.Dispose();
			}
			base.Dispose(disposing);
		}

		#endregion

		#region Public

		/// <summary>
		/// Size of the internal buffer
		/// </summary>
		public int BufferSize { get { return internalBuffer.Length; } }

		#endregion

		#region Private

		/// <summary>
		/// 
		/// </summary>
		private long NonWindowedPosition
		{
			get
			{
				long pos = fstream.Position + bufferIndex - streamOffset;
				if (pending == StreamOperation.READ) pos -= streamHeadStart + BufferSize;
				return pos;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		private void PendingWrite()
		{
			// Write remaining bytes from the buffer, if write was the last operation
			if (pending == StreamOperation.WRITE)
			{
				byte[] tempBuffer = new byte[BufferSize];
				ReadBlock(tempBuffer, 0);
				SkipToPrev();
				Buffer.BlockCopy(internalBuffer, 0, tempBuffer, 0, bufferIndex);
				WriteBlock(tempBuffer, 0);
				internalBuffer = tempBuffer;
			}
		}

		/// <summary>
		/// Initialize stream
		/// </summary>
		private void Init()
		{
			dimensionSize = new int[imageLimits.Length];
			windowSize = new int[imageLimits.Length];
			preBlock = new int[imageLimits.Length];
			postBlock = new int[imageLimits.Length];

			maxDim = imageLimits.Length - 1;

			int size = ImageFile.BytesPerPixel(this.dataType.Type);
			int wSize = ImageFile.BytesPerPixel(this.dataType.Type);
			bool continuous = true;

			continuousDimension = 0;
			windowPosition = 0;

			for (int i = 0; i < imageLimits.Length; i++)
			{
				preBlock[i] = (windowLimits.GetLimit(i, Limits.Limit.LOW) - imageLimits.GetLimit(i, Limits.Limit.LOW)) * size;
				postBlock[i] = (imageLimits.GetLimit(i, Limits.Limit.HIGH) - windowLimits.GetLimit(i, Limits.Limit.HIGH)) * size;

				size *= imageLimits[i];
				wSize *= windowLimits[i];

				windowSize[i] = wSize;
				dimensionSize[i] = size;

				// If data up to this dimension has no gaps and windows contents fits into the buffer,
				// set the next dimension as the continuous dimension
				if (continuous && windowSize[i] < MAXBUFFER)
				{
					continuousDimension = i;
				}
				continuous = (size == wSize);
			}

			internalBuffer = new byte[windowSize[continuousDimension]];
		}

		/// <summary>
		/// 
		/// </summary>
		private void SkipToFirst()
		{
			long skipLength = streamOffset;

			for (int i = continuousDimension; i <= maxDim; i++)
			{
				skipLength += preBlock[i];
			}

			fstream.Seek(skipLength, SeekOrigin.Begin);
			windowPosition = 0;
			streamHeadStart = 0;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="dim"></param>
		/// <param name="pos"></param>
		/// <returns></returns>
		private long SkipLength(int dim, long pos)
		{
			long skipLength = 0;
			long currentPosition = pos % windowSize[dim];

			if (currentPosition == 0)
			{
				skipLength += preBlock[dim] + postBlock[dim];
				if (dim < maxDim)
					skipLength += SkipLength(dim + 1, pos);
			}

			return skipLength;
		}

		/// <summary>
		/// 
		/// </summary>
		private void SkipToNext()
		{
			streamHeadStart = SkipLength(continuousDimension, windowPosition);
			fstream.Seek(streamHeadStart, SeekOrigin.Current);
		}

		/// <summary>
		/// 
		/// </summary>
		private void SkipToPrev()
		{
			streamHeadStart = SkipLength(continuousDimension, windowPosition) + windowSize[continuousDimension];
			fstream.Seek(-streamHeadStart, SeekOrigin.Current);
			windowPosition -= windowSize[continuousDimension];
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="buffer"></param>
		/// <param name="offset"></param>
		private void WriteBlock(byte[] buffer, int offset)
		{
			// Write the block to the stream
			fstream.Write(buffer, offset, windowSize[continuousDimension]);
			windowPosition += windowSize[continuousDimension];

			// Skip to the start of next block
			if (!EndOfStream) SkipToNext();
		}

		/// <summary>
		/// Read one datablock from the stream
		/// </summary>
		/// <param name="buffer">Buffer for the block</param>
		/// <param name="offset">Buffer offset</param>
		/// <returns>Number of bytes read</returns>
		private int ReadBlock(byte[] buffer, int offset)
		{
			int bytesRead = 0;

			if (!EndOfStream)
			{
				// Read one block from the stream and skip to the next block
				bytesRead = fstream.Read(buffer, offset, windowSize[continuousDimension]);
				windowPosition += bytesRead;
				if (!EndOfStream) SkipToNext();
			}

			return bytesRead;
		}

		#endregion
	}
}
