/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Reflection;

namespace TPClib.Image
{
	/// <summary>
	/// Static utility functions
	/// </summary>
    public static class Utils
    {
        /// <summary>
        /// Finds the dimension number of the vector. Vector must
        /// be parallel with some of the components (x,y,z ...) and length
        /// must be one. Scanners of Pixel stores are specialised only this
        /// kind of vectors.
        /// </summary>
        /// <param name="vector"></param>
        /// <returns>
        /// One of following following formula:
        ///   1 =  x 
        ///  -1 = -x
        ///   2 =  y
        ///  -2 = -y, etc...
        /// </returns>
        public static int FindDimension(double[] vector)
        {
            int numDims = 0;
            int oneDimNum = -1;

            // Calculating how many dimensions has value other than zero
            for (int i = 0; i < vector.Length; i++) if (vector[i] != 0) { numDims++; oneDimNum = i; }

            // If the vector has only one dimension other than zero,
            // we can create one dimensional vector
            if (numDims != 1) throw new Exception("Vector must be along components and have length of 1");

            // Checking, if the vector is aligned to some component vector
            // and has length of 1
            if (Math.Abs(vector[oneDimNum]) == 1.0) return (oneDimNum + 1) * Math.Sign(vector[oneDimNum]);

            // If not, 0 is returned
            throw new Exception("Vector must have length of 1");
        }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="src"></param>
		/// <param name="factor"></param>
		/// <returns></returns>
        public static double[] VectorMultiply(double[] src, double factor)
        {
            double[] result = new double[src.Length];
            for (int i = 0; i < src.Length; i++) {result[i] = src[i] * factor;}
            return result;
        }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="src1"></param>
		/// <param name="src2"></param>
		/// <returns></returns>
        public static double[] VectorAdd(double[] src1, double[] src2)
        {
            double[] result = new double[src1.Length];
            for (int i = 0; i < src1.Length; i++) { result[i] = src1[i] + src2[i]; }
            return result;
        }

		/// <summary>
		/// 
		/// </summary>
		/// <typeparam name="N"></typeparam>
		/// <param name="name"></param>
		/// <returns></returns>
        public static N readStaticField<N>(string name)
        {
            FieldInfo field = typeof(N).GetField(name, BindingFlags.Public | BindingFlags.Static);
            if (field == null)
            {
                // There's no TypeArgumentException, unfortunately. You might wantto create one :)
                throw new InvalidOperationException("Invalid type argument for NumericUpDown<T>: " +
                    typeof(N).Name);
            }
            return (N)field.GetValue(null);
        }

		/// <summary>
		/// 
		/// </summary>
		/// <typeparam name="N"></typeparam>
		/// <param name="first"></param>
		/// <param name="second"></param>
		/// <returns></returns>
        public static N Multiply<N>(N first, double second)
        {
            Type t = typeof(N);
            if (t == typeof(byte)) return (N)(object)(byte)((byte)(object)first * second);
            if (t == typeof(sbyte)) return (N)(object)(sbyte)((sbyte)(object)first * second);
            if (t == typeof(short)) return (N)(object)(short)((short)(object)first * second);
            if (t == typeof(ushort)) return (N)(object)(ushort)((sbyte)(object)first * second);
            if (t == typeof(char)) return (N)(object)(char)((char)(object)first * second);
            if (t == typeof(Int32)) return (N)(object)(Int32)((Int32)(object)first * second);
            if (t == typeof(UInt32)) return (N)(object)(UInt32)((UInt32)(object)first * second);
            if (t == typeof(Int64)) return (N)(object)(Int64)((Int64)(object)first * second);
            if (t == typeof(UInt64)) return (N)(object)(UInt64)((UInt64)(object)first * second);
            if (t == typeof(float)) return (N)(object)(float)((float)(object)first * second);
            if (t == typeof(double)) return (N)(object)(double)((double)(object)first * second);
            //if (t == typeof(decimal)) return (N)(object)(decimal)((decimal)(object)first * second);
            else throw new Exception("Cannot multiply type " + t.Name);
        }

		/// <summary>
		/// 
		/// </summary>
		/// <typeparam name="N"></typeparam>
		/// <param name="first"></param>
		/// <param name="second"></param>
		/// <returns></returns>
        public static N Multiply<N>(N first, N second)
        {
            Type t = typeof(N);
            if (t == typeof(byte)) return (N)(object)(byte)((byte)(object)first * (byte)(object)second);
            if (t == typeof(sbyte)) return (N)(object)(sbyte)((sbyte)(object)first * (sbyte)(object)second);
            if (t == typeof(short)) return (N)(object)(short)((short)(object)first * (short)(object)second);
            if (t == typeof(ushort)) return (N)(object)(ushort)((sbyte)(object)first * (ushort)(object)second);
            if (t == typeof(char)) return (N)(object)(char)((char)(object)first * (char)(object)second);
            if (t == typeof(Int32)) return (N)(object)(Int32)((Int32)(object)first * (Int32)(object)second);
            if (t == typeof(UInt32)) return (N)(object)(UInt32)((UInt32)(object)first * (UInt32)(object)second);
            if (t == typeof(Int64)) return (N)(object)(Int64)((Int64)(object)first * (Int64)(object)second);
            if (t == typeof(UInt64)) return (N)(object)(UInt64)((UInt64)(object)first * (UInt64)(object)second);
            if (t == typeof(float)) return (N)(object)(float)((float)(object)first * (float)(object)second);
            if (t == typeof(double)) return (N)(object)(double)((double)(object)first * (double)(object)second);
            if (t == typeof(decimal)) return (N)(object)(decimal)((decimal)(object)first * (decimal)(object)second);
            else throw new Exception("Cannot multiply type " + t.Name);
        }

		/// <summary>
		/// 
		/// </summary>
		/// <typeparam name="N"></typeparam>
		/// <param name="first"></param>
		/// <param name="second"></param>
		/// <returns></returns>
        public static N Add<N>(N first, double second)
        {
            Type t = typeof(N);
            if (t == typeof(byte)) return (N)(object)(byte)((byte)(object)first + second);
            if (t == typeof(sbyte)) return (N)(object)(sbyte)((sbyte)(object)first + second);
            if (t == typeof(short)) return (N)(object)(short)((short)(object)first + second);
            if (t == typeof(ushort)) return (N)(object)(ushort)((sbyte)(object)first + second);
            if (t == typeof(char)) return (N)(object)(char)((char)(object)first + second);
            if (t == typeof(Int32)) return (N)(object)(Int32)((Int32)(object)first + second);
            if (t == typeof(UInt32)) return (N)(object)(UInt32)((UInt32)(object)first + second);
            if (t == typeof(Int64)) return (N)(object)(Int64)((Int64)(object)first + second);
            if (t == typeof(UInt64)) return (N)(object)(UInt64)((UInt64)(object)first + second);
            if (t == typeof(float)) return (N)(object)(float)((float)(object)first + second);
            if (t == typeof(double)) return (N)(object)(double)((double)(object)first + second);
            //if (t == typeof(decimal)) return (N)(object)(decimal)((decimal)(object)first + second);
            else throw new Exception("Cannot add type " + t.Name);
        }

		/// <summary>
		/// 
		/// </summary>
		/// <typeparam name="N"></typeparam>
		/// <param name="first"></param>
		/// <param name="second"></param>
		/// <returns></returns>
        public static N Add<N>(N first, N second)
        {
            Type t = typeof(N);
            if (t == typeof(byte)) return (N)(object)(byte)((byte)(object)first + (byte)(object)second);
            if (t == typeof(sbyte)) return (N)(object)(sbyte)((sbyte)(object)first + (sbyte)(object)second);
            if (t == typeof(short)) return (N)(object)(short)((short)(object)first + (short)(object)second);
            if (t == typeof(ushort)) return (N)(object)(ushort)((sbyte)(object)first + (ushort)(object)second);
            if (t == typeof(char)) return (N)(object)(char)((char)(object)first + (char)(object)second);
            if (t == typeof(Int32)) return (N)(object)(Int32)((Int32)(object)first + (Int32)(object)second);
            if (t == typeof(UInt32)) return (N)(object)(UInt32)((UInt32)(object)first + (UInt32)(object)second);
            if (t == typeof(Int64)) return (N)(object)(Int64)((Int64)(object)first + (Int64)(object)second);
            if (t == typeof(UInt64)) return (N)(object)(UInt64)((UInt64)(object)first + (UInt64)(object)second);
            if (t == typeof(float)) return (N)(object)(float)((float)(object)first + (float)(object)second);
            if (t == typeof(double)) return (N)(object)(double)((double)(object)first + (double)(object)second);
            if (t == typeof(decimal)) return (N)(object)(decimal)((decimal)(object)first + (decimal)(object)second);
            else throw new Exception("Cannot add type " + t.Name);
        }

		/// <summary>
		/// 
		/// </summary>
		/// <param name="dims"></param>
		/// <param name="vectors"></param>
		/// <param name="dimensionInfo"></param>
		/// <param name="result"></param>
		/// <param name="newDimensions"></param>
		/// <param name="origo"></param>
        public static void FlipDimensions(uint[] dims, double[][] vectors, int[] dimensionInfo, out double[][] result, out uint[] newDimensions, out double[] origo)
        {
            newDimensions = new uint[dimensionInfo.Length];
            origo = new double[dims.Length];
            result = new double[dims.Length][];

            // Creating the new dimensions and component vectors (length 1)
            // for every dimension
            for (int i = 0; i < dims.Length; i++)
            {
                // creating the new dimension:
                int newDim = Math.Abs(dimensionInfo[i]) - 1;
                newDimensions[i] = dims[newDim];

                // Setting dimension vector (length 1) and origo (center of voxel)
                double[] dimensionVector = new double[dims.Length];
                if (dimensionInfo[i] < 0)
                {
                    // the dimension is read backwards (flipped)
                    for (int w = 0; w < dims.Length; w++) dimensionVector[w] = -vectors[newDim][w];

                    // origo is set to end of the dimension
                    origo[newDim] = newDimensions[i] - 0.5f;
                }
                else
                {
                    for (int w = 0; w < dims.Length; w++) dimensionVector[w] = vectors[newDim][w];
                    //dimensionVector[newDim] = vectors[i];
                    origo[newDim] = 0.5f;
                }
                result[i] = dimensionVector;
            }
        }
    }
}
