/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Image.ValueScales
{
	/// <summary>
	/// 
	/// </summary>
	/// <typeparam name="F"></typeparam>
	/// <typeparam name="T"></typeparam>
	public struct ConversionResult<F,T>
        where F : struct, IConvertible, IComparable
        where T : struct, IConvertible, IComparable
    {
		/// <summary>
		/// 
		/// </summary>
        public readonly ConversionInfo Info;
        
		/// <summary>
		/// 
		/// </summary>
		public readonly F SourceMin;
        
		/// <summary>
		/// 
		/// </summary>
		public readonly F SourceMax;

        /// <summary>
        /// Minimum value of the converted scale
        /// </summary>
        public readonly T Converted_Min;

        /// <summary>
        /// Maximum value of the converted scale
        /// </summary>
        public readonly T Converted_Max;

		/// <summary>
		/// 
		/// </summary>
		/// <param name="info"></param>
		/// <param name="srcMin"></param>
		/// <param name="srcMax"></param>
		/// <param name="destMin"></param>
		/// <param name="destMax"></param>
        public ConversionResult(ConversionInfo info, F srcMin, F srcMax, T destMin, T destMax)
        {
            this.Info = info;
            this.SourceMin = srcMin;
            this.SourceMax = srcMax;
            this.Converted_Min = destMin;
            this.Converted_Max = destMax;
        }
    }

    /// <summary>
    /// Information about conversion from Value Scale to another
    /// </summary>
    public class ConversionInfo
    {
        /// <summary>
        /// True, if the conversion causes data loss
        /// </summary>
        public readonly bool DataLoss;
        /// <summary>
        /// True, if the source scale range is larger than destination.
        /// All values outside are then put to min or max of destination
        /// </summary>
        public readonly bool WindowingNeeded;

        /// <summary>
        /// Constructs conversion info object
        /// </summary>
        /// <param name="dataloss">True, if the conversion causes data loss</param>
        /// <param name="windowingNeeded">True, if the source scale range is larger than destination.
        /// All values outside are then put to min or max of destination</param>
        public ConversionInfo(bool dataloss, bool windowingNeeded)
        {
            this.DataLoss = dataloss;
            this.WindowingNeeded = windowingNeeded;
        }

        /// <summary>
        /// All type combinations that cannot be converted without data loss
        /// </summary>
        private static Dictionary<Type, List<Type>> datalossCombinations = new Dictionary<Type, List<Type>>();
        static ConversionInfo()
        {
            // Byte
            datalossCombinations.Add(typeof(byte), new List<Type>(new Type[] {}));
            // SByte
            datalossCombinations.Add(typeof(sbyte), new List<Type>(new Type[] {}));
            // Short
            datalossCombinations.Add(typeof(short), new List<Type>(new Type[] {}));
            // UShort (char)
            datalossCombinations.Add(typeof(ushort), new List<Type>(new Type[] {}));
            // Char (ushort)
            datalossCombinations.Add(typeof(char), new List<Type>(new Type[] {}));
            // Int32
            datalossCombinations.Add(typeof(Int32), new List<Type>(new Type[] {}));
            // UInt32
            datalossCombinations.Add(typeof(UInt32), new List<Type>(new Type[] {}));
            // Int64
            datalossCombinations.Add(typeof(Int64), new List<Type>(new Type[] {}));
            // UInt64
            datalossCombinations.Add(typeof(UInt64), new List<Type>(new Type[] {}));
            // float
            datalossCombinations.Add(typeof(float), new List<Type>(new Type[] { 
                typeof(bool), typeof(byte), typeof(sbyte), typeof(short), typeof(ushort), typeof(char), 
                typeof(int), typeof(uint), typeof(Int64), typeof(UInt64) }));
            // double
            datalossCombinations.Add(typeof(double), new List<Type>(new Type[] { 
                typeof(bool), typeof(float), typeof(byte), typeof(sbyte), typeof(short), typeof(ushort), 
                typeof(char), typeof(int), typeof(uint), typeof(Int64), typeof(UInt64) }));
        }



        /// <summary>
        /// Creates conversion info object
        /// </summary>
		/// <typeparam name="F"></typeparam>
		/// <typeparam name="T"></typeparam>
        /// <returns></returns>
        public static ConversionResult<F, T> Create<F, T>()
            where F : struct, IConvertible, IComparable
            where T : struct, IConvertible, IComparable
        {
            // If the source does not know it's min and max,
            // we must assume that the whole range of the source
            // scale is in use:
            F minSource = Utils.readStaticField<F>("MinValue");
            F maxSource = Utils.readStaticField<F>("MaxValue");
            return Create<F, T>(minSource, maxSource, 1.0d, 0.0d);
        }

        /// <summary>
        /// Creates conversion info object
        /// </summary>
		/// <typeparam name="F"></typeparam>
		/// <typeparam name="T"></typeparam>
		/// <param name="minSource"></param>
		/// <param name="maxSource"></param>
		/// <param name="factor"></param>
		/// <param name="intersection"></param>
		/// <returns></returns>
        public static ConversionResult<F,T> Create<F, T>(F minSource, F maxSource, double factor, double intersection)
            where F : struct, IConvertible, IComparable
            where T : struct, IConvertible, IComparable
        {
            bool needsWindowing = false;
            bool dataLoss = false;

            // Checking if the given combination contains always dataloss:
            if(datalossCombinations[typeof(F)].Contains(typeof(T))) { dataLoss = true; }
            
            // calculating the transformed min and max
            double tMinSource = (double)Convert.ChangeType(minSource, typeof(double)) * factor + intersection;
            double tMaxSource = (double)Convert.ChangeType(maxSource, typeof(double)) * factor + intersection;

            // Check the windowing:
            T minDest = default(T);
            try { checked { minDest = (T)Convert.ChangeType(tMinSource, typeof(T)); } }
            catch (OverflowException)
            {
                needsWindowing = true;
                minDest = Utils.readStaticField<T>("MinValue");

                // lowest proper value in source:
                double minDest_double = (double)Convert.ChangeType(minDest, typeof(double));
                minSource = (F)Convert.ChangeType((minDest_double - intersection) / factor, typeof(F));
            }

            T maxDest = default(T);
            try { checked { maxDest = (T)Convert.ChangeType(tMaxSource, typeof(T)); } }
            catch (OverflowException)
            {
                needsWindowing = true;
                maxDest = Utils.readStaticField<T>("MaxValue");

                // highest proper value in source:
                double maxDest_double = (double)Convert.ChangeType(maxDest, typeof(double));
                maxSource = (F)Convert.ChangeType((maxDest_double - intersection) / factor, typeof(F));
            }

            // Creating the result:
            return new ConversionResult<F, T>(new ConversionInfo(dataLoss, needsWindowing), minSource, maxSource, minDest, maxDest);
        }
    }

}
