/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Image.ValueScales
{
    /// <summary>
    /// This class creates fastest possible converter object for
    /// one Scale type to another
    /// </summary>
    public class ConverterFactor
    {
        /// <summary>
        /// Creates converter class between given types
        /// </summary>
        /// <typeparam name="T">From - Type</typeparam>
        /// <typeparam name="K">To - Type</typeparam>
        /// <returns>The converter object</returns>
        public static IConverter<T, K> Create<T, K>()
            where T : struct, IConvertible, IComparable
            where K : struct, IConvertible, IComparable
        {
            ValueScale<T> from = new ValueScale<T>();
            ValueScale<K> to = new ValueScale<K>();
            return Create<T, K>(from, to);
        }

        /// <summary>
        /// Creates converter class between given types
        /// </summary>
        /// <typeparam name="T">From - Type</typeparam>
        /// <typeparam name="K">To - Type</typeparam>
        /// <param name="from">Source ValueScale</param>
        /// <param name="to">Destination ValueScale</param>
        /// <returns>The converter object</returns>
        public static IConverter<T, K> Create<T, K>(ValueScale<T> from, ValueScale<K> to)
            where T : struct, IConvertible, IComparable
            where K : struct, IConvertible, IComparable
        {
            
            // calculating the factor and intersection between two types.
            // When source value is transformed with these, we get the
            // result value (in correct scale)
            double factor = from.Factor / to.Factor;
            double intersection = (from.Intersection - to.Intersection) / to.Factor;

            //double factor = from.Factor + from.Intersection - to.Intersection

            // Min and max of the source:
            T minSource = Utils.readStaticField<T>("MinValue");
            T maxSource = Utils.readStaticField<T>("MaxValue");
            if (from.MinMaxGiven) { minSource = from.Min; maxSource = from.Max; }

            // Calculating the conversion information:
            ConversionResult<T, K> convRes = ConversionInfo.Create<T, K>(minSource, maxSource, factor, intersection);

            if (factor == 1 && intersection == 0) // The source and destination are already in same scale:
            {
                // If the types are same
                if (typeof(T) == typeof(K)) return (IConverter<T, K>)new SameTypeConverter<T>();

                // If there is no dataloss and no windowing, fast widening
                // converter can be used
                if (!convRes.Info.WindowingNeeded) return RealConverter_NoWindow<T, K>.Create(convRes.Info.DataLoss);

                // Creating the windowing comparer, if needed or faster
                // one without windowing, if the scale fits the destination
                return RealConverter_window<T, K>.Create(convRes.Info.DataLoss);
            }

            if (convRes.Info.WindowingNeeded) return Rescale_Windowing<T, K>.Create(convRes.Info.DataLoss, factor, intersection, convRes.SourceMin, convRes.SourceMax);                                           
            else return Rescale_NoWindow<T, K>.Create(convRes.Info.DataLoss, factor, intersection);  
        }

        #region RealToRealConversion

        /// <summary>
        /// This converter is created, if the user asks conversion from
        /// type to same type
        /// </summary>
        /// <typeparam name="N">The type</typeparam>
        private class SameTypeConverter<N> : IConverter<N, N>
            where N : struct, IConvertible, IComparable
        {
            protected ConversionInfo info = new ConversionInfo(false, false);
            public ConversionInfo ConversionInfo { get { return info; } }
            public N Convert(N value) { return value; }
            public void Convert(N[] from, int fromIndex, N[] to, int toIndex, int length)
            {
                Array.Copy(from, fromIndex, to, toIndex, length);
            }
            public virtual void Convert(N[] from, int fromIndex, N[] to, int toIndex, int length, int jump)
            {
                for (int i = toIndex; i < toIndex + length; i++) { to[i] = Convert(from[fromIndex]); fromIndex += jump; }
            }
            public N[] Convert(N[] from)
            {
                N[] result = new N[from.Length]; 
                Array.Copy(from, 0, result, 0, from.Length);
                return result;
            }
        }


        /// <summary>
        /// Value type converter that is already in same scale
        /// as the destination and needs no windowing check
        /// </summary>
        /// <typeparam name="F">From - Source value type</typeparam>
        /// <typeparam name="T">To - Destination value type</typeparam>
        private abstract class RealConverter_NoWindow<F, T> : IConverter<F, T>
            where F : struct, IConvertible, IComparable
            where T : struct, IConvertible, IComparable
        {
            protected ConversionInfo info = new ConversionInfo(false, false);
            public ConversionInfo ConversionInfo { get { return info; } }
            public abstract T Convert(F value);
            
            public virtual void Convert(F[] from, int fromIndex, T[] to, int toIndex, int length, int jump)
            {
                for (int i = toIndex; i < toIndex + length; i++) { to[i] = Convert(from[fromIndex]); fromIndex += jump; }
            }

            public virtual void Convert(F[] from, int fromIndex, T[] to, int toIndex, int length)
            {
                Array.Copy(from, fromIndex, to, toIndex, length);
            }
            public T[] Convert(F[] from)
            {
                T[] result = new T[from.Length];
                Convert(from, 0, result, 0, from.Length);
                return result;
            }

            public static RealConverter_NoWindow<F, T> Create(bool dataLoss)
            {
                Type to = typeof(T);
                Type from = typeof(F);
                RealConverter_NoWindow<F, T> c = null;

                if (from == typeof(Byte))
                {
                    if (to == typeof(sbyte)) c = new Byte_sByte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Byte_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Byte_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int32)) c = new Byte_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt32)) c = new Byte_uInt() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new Byte_Int64() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new Byte_uInt64() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Byte_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Byte_Double() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(SByte))
                {
                    if (to == typeof(byte)) c = new sByte_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new sByte_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new sByte_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int32)) c = new sByte_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt32)) c = new sByte_uInt() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new sByte_Int64() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new sByte_uInt64() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new sByte_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new sByte_Double() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(short))
                {
                    if (to == typeof(byte)) c = new Short_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Short_sByte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Short_uShort() as RealConverter_NoWindow<F, T>;

                    else if (to == typeof(ushort)) c = new Short_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int32)) c = new Short_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt32)) c = new Short_uInt() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new Short_Long() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new Short_uLong() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Short_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Short_Double() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Short_Decimal() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(ushort))
                {
                    if (to == typeof(byte)) c = new uShort_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new uShort_sByte() as RealConverter_NoWindow<F, T>;

                    else if (to == typeof(short)) c = new uShort_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int32)) c = new uShort_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt32)) c = new uShort_uInt() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new uShort_Long() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new uShort_uLong() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new uShort_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new uShort_Double() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new uShort_Decimal() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(int))
                {
                    if (to == typeof(byte)) c = new Int_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Int_sByte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Int_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Int_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Int_uInt() as RealConverter_NoWindow<F, T>;

                    else if (to == typeof(Int64)) c = new Int_Long() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new Int_uLong() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Int_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Int_Double() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Int_Decimal() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(uint))
                {
                    if (to == typeof(byte)) c = new uInt_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new uInt_sByte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new uInt_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new uInt_uShort() as RealConverter_NoWindow<F, T>;

                    else if (to == typeof(Int32)) c = new uInt_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new uInt_Long() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new uInt_uLong() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new uInt_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new uInt_Double() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new uInt_Decimal() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(long))
                {
                    if (to == typeof(byte)) c = new Int64_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Int64_sByte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Long_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Long_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new Long_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Long_uInt() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ulong)) c = new Long_uLong() as RealConverter_NoWindow<F, T>;

                    else if (to == typeof(UInt64)) c = new Long_uLong() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Long_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Long_Double() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Long_Decimal() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(ulong))
                {
                    if (to == typeof(byte)) c = new uInt64_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new uInt64_sByte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new uLong_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new uLong_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new uLong_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new uLong_uInt() as RealConverter_NoWindow<F, T>;

                    else if (to == typeof(long)) c = new uLong_Long() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new uLong_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new uLong_Double() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new uLong_Decimal() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(float))
                {
                    if (to == typeof(byte)) c = new Float_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Float_sByte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Float_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Float_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new Float_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Float_uInt() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(long)) c = new Float_Long() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ulong)) c = new Float_uLong() as RealConverter_NoWindow<F, T>;

                    else if (to == typeof(double)) c = new Float_Double() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Float_Decimal() as RealConverter_NoWindow<F, T>;
                }
                else if (from == typeof(double))
                {
                    if (to == typeof(byte)) c = new Double_Byte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Double_sByte() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Double_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Double_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new Double_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Double_uInt() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(long)) c = new Double_Long() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ulong)) c = new Double_uLong() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Double_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Double_Decimal() as RealConverter_NoWindow<F, T>;                       
                }
                else if (from == typeof(decimal))
                {
                    if (to == typeof(short)) c = new Decimal_Short() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Decimal_uShort() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new Decimal_Int() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Decimal_uInt() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(long)) c = new Decimal_Long() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(ulong)) c = new Decimal_uLong() as RealConverter_NoWindow<F, T>;

                    else if (to == typeof(float)) c = new Decimal_Float() as RealConverter_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Decimal_Double() as RealConverter_NoWindow<F, T>;
                }

                if (c == null) throw new Exception("The Real Space conversion class is not implemented from " + from.Name + " to " + to.Name);
                c.info = new ConversionInfo(dataLoss, false);
                return c;
            }

            #region FromByte

            class Byte_sByte : RealConverter_NoWindow<byte, sbyte>
            {
                public override void Convert(byte[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {Buffer.BlockCopy(from, fromIndex, to, toIndex, length);}

                public override sbyte Convert(byte value) { return (sbyte)value; }
            }

            class Byte_Short : RealConverter_NoWindow<byte, short>
            {
                public override short Convert(byte value) { return (byte)value; }
            }

            class Byte_uShort : RealConverter_NoWindow<byte, ushort>
            {
                public override ushort Convert(byte value) { return (ushort)value; }
            }

            class Byte_Int : RealConverter_NoWindow<byte, int>
            {
                public override int Convert(byte value) { return (int)value; }
            }

            class Byte_uInt : RealConverter_NoWindow<byte, uint>
            {
                public override uint Convert(byte value) { return (uint)value; }
            }

            class Byte_Int64 : RealConverter_NoWindow<byte, long>
            {
                public override long Convert(byte value) { return (long)value; }
            }

            class Byte_uInt64 : RealConverter_NoWindow<byte, ulong>
            {
                public override ulong Convert(byte value) { return (ulong)value; }
            }

            class Byte_Float : RealConverter_NoWindow<byte, float>
            {
                public override float Convert(byte value) { return (float)value; }
            }

            class Byte_Double : RealConverter_NoWindow<byte, double>
            {
                public override double Convert(byte value) { return (double)value; }
            }

            #endregion


            #region FromSByte

            class sByte_Byte : RealConverter_NoWindow<sbyte, byte>
            {
                public override void Convert(sbyte[] from, int fromIndex, byte[] to, int toIndex, int length)
                { Buffer.BlockCopy(from, fromIndex, to, toIndex, length); }

                public override byte Convert(sbyte value) { return (byte)value; }
            }

            class sByte_Short : RealConverter_NoWindow<sbyte, short>
            {
                public override short Convert(sbyte value) { return (short)value; }
            }


            class sByte_uShort : RealConverter_NoWindow<sbyte, ushort>
            {
                public override void Convert(sbyte[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    // Array.Copy does not work with this combination for some reason                    
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (ushort)from[fromIndex++];
                }

                public override ushort Convert(sbyte value)
                { return (ushort)value; }
            }


            class sByte_Int : RealConverter_NoWindow<sbyte, int>
            {
                public override void Convert(sbyte[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (sbyte)from[fromIndex++]; }
                }
                public override int Convert(sbyte value)
                { return (int)value; }
            }


            class sByte_uInt : RealConverter_NoWindow<sbyte, uint>
            {
                public override void Convert(sbyte[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    // Array.Copy does not work with this combination for some reason                    
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (uint)from[fromIndex++];
                }

                public override uint Convert(sbyte value)
                { return (uint)value; }
            }


            class sByte_Int64 : RealConverter_NoWindow<sbyte, long>
            {
                public override long Convert(sbyte value)
                { return (long)value; }
            }

            class sByte_uInt64 : RealConverter_NoWindow<sbyte, ulong>
            {
                public override void Convert(sbyte[] from, int fromIndex, ulong[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (ulong)from[fromIndex++]; }

                public override ulong Convert(sbyte value) { return (ulong)value; }
            }

            class sByte_Float : RealConverter_NoWindow<sbyte, float>
            { public override float Convert(sbyte value) { return (float)value; } }

            class sByte_Double : RealConverter_NoWindow<sbyte, double>
            { public override double Convert(sbyte value) { return (double)value; } }


            #endregion


            #region FromShort

            class Short_sByte : RealConverter_NoWindow<short, sbyte>
            {
                public override void Convert(short[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    // The Array.Copy does not work with this combination:
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (sbyte)from[fromIndex++];
                }

                public override sbyte Convert(short value) { return (sbyte)value; }
            }

            class Short_Byte : RealConverter_NoWindow<short, byte>
            {
                public override void Convert(short[] from, int fromIndex, byte[] to, int toIndex, int length) { for (int i = 0; i < length; i++) { to[toIndex++] = (byte)from[fromIndex++]; } }
                public override byte Convert(short value) { return (byte)value; }
            }



            class Short_uShort : RealConverter_NoWindow<short, ushort>
            {
                public override void Convert(short[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    // The Array.Copy does not work with this combination:
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (ushort)from[fromIndex++];
                }

                public override ushort Convert(short value) { return (ushort)value; }
            }

            class Short_Int : RealConverter_NoWindow<short, int>
            {
                public override int Convert(short value) { return (int)value; }
            }

            class Short_uInt : RealConverter_NoWindow<short, uint>
            {
                public override void Convert(short[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    // The Array.Copy does not work with this combination:
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (uint)from[fromIndex++];
                }
                public override uint Convert(short value) { return (uint)value; }
            }

            class Short_Long : RealConverter_NoWindow<short, long>
            {
                public override long Convert(short value) { return (long)value; }
            }

            class Short_uLong : RealConverter_NoWindow<short, ulong>
            {
                public override void Convert(short[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    // The Array.Copy does not work with this combination:
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (ulong)from[fromIndex++];
                }

                public override ulong Convert(short value) { return (ulong)value; }
            }

            class Short_Float : RealConverter_NoWindow<short, float>
            {
                public override float Convert(short value) { return (float)value; }
            }

            class Short_Double : RealConverter_NoWindow<short, double>
            {
                public override double Convert(short value) { return (double)value; }
            }

            class Short_Decimal : RealConverter_NoWindow<short, decimal>
            {
                public override decimal Convert(short value) { return (decimal)value; }
            }

            #endregion


            #region FromUShort

            class uShort_sByte : RealConverter_NoWindow<ushort, sbyte>
            {
                public override void Convert(ushort[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    // The Array.Copy does not work with this combination:
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (sbyte)from[fromIndex++];
                }

                public override sbyte Convert(ushort value) { return (sbyte)value; }
            }

            class uShort_Byte : RealConverter_NoWindow<ushort, byte>
            {
                public override void Convert(ushort[] from, int fromIndex, byte[] to, int toIndex, int length) { for (int i = 0; i < length; i++) { to[toIndex++] = (byte)from[fromIndex++]; } }
                public override byte Convert(ushort value) { return (byte)value; }
            }

            class uShort_Short : RealConverter_NoWindow<ushort, short>
            {
                public override void Convert(ushort[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    // The Array.Copy does not work with this combination:
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (short)from[fromIndex++];
                }
                public override short Convert(ushort value) { return (short)value; }
            }

            class uShort_Int : RealConverter_NoWindow<ushort, int>
            {
                public override int Convert(ushort value) { return (int)value; }
            }

            class uShort_uInt : RealConverter_NoWindow<ushort, uint>
            {
                public override uint Convert(ushort value) { return (uint)value; }
            }

            class uShort_Long : RealConverter_NoWindow<ushort, long>
            {
                public override long Convert(ushort value) { return (long)value; }
            }

            class uShort_uLong : RealConverter_NoWindow<ushort, ulong>
            {
                public override ulong Convert(ushort value) { return (ulong)value; }
            }

            class uShort_Float : RealConverter_NoWindow<ushort, float>
            {
                public override float Convert(ushort value) { return (float)value; }
            }

            class uShort_Double : RealConverter_NoWindow<ushort, double>
            {
                public override double Convert(ushort value) { return (double)value; }
            }

            class uShort_Decimal : RealConverter_NoWindow<ushort, decimal>
            {
                public override decimal Convert(ushort value) { return (decimal)value; }
            }

            #endregion


            #region FromInt


            class Int_Byte : RealConverter_NoWindow<int, byte>
            {
                public override void Convert(int[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    // The Array.Copy does not work with this combination:
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (byte)from[fromIndex++];
                }

                public override byte Convert(int value) { return (byte)value; }
            }

            class Int_sByte : RealConverter_NoWindow<int, sbyte>
            {
                public override void Convert(int[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (sbyte)from[fromIndex++]; }
                }
                public override sbyte Convert(int value)
                { return (sbyte)value; }
            }

            class Int_Short : RealConverter_NoWindow<int, short>
            {
                public override void Convert(int[] from, int fromIndex, short[] to, int toIndex, int length)
                {for (int i = toIndex; i < toIndex + length; i++) to[i] = (short)from[fromIndex++];}

                public override short Convert(int value) { return (short)value; }
            }

            class Int_uShort : RealConverter_NoWindow<int, ushort>
            {
                public override void Convert(int[] from, int fromIndex, ushort[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (ushort)from[fromIndex++]; }

                public override ushort Convert(int value) { return (ushort)value; }
            }


            class Int_uInt : RealConverter_NoWindow<int, uint>
            {
                public override void Convert(int[] from, int fromIndex, uint[] to, int toIndex, int length)
                {for (int i = toIndex; i < toIndex + length; i++) to[i] = (uint)from[fromIndex++];}

                public override uint Convert(int value) { return (uint)value; }
            }

            class Int_Long : RealConverter_NoWindow<int, long>
            {
                public override long Convert(int value) { return (long)value; }
            }

            class Int_uLong : RealConverter_NoWindow<int, ulong>
            {
                public override void Convert(int[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    // The Array.Copy does not work with this combination:
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (ulong)from[fromIndex++];
                }

                public override ulong Convert(int value) { return (ulong)value; }
            }

            class Int_Float : RealConverter_NoWindow<int, float>
            {
                public override float Convert(int value) { return (float)value; }
            }

            class Int_Double : RealConverter_NoWindow<int, double>
            {
                public override double Convert(int value) { return (double)value; }
            }

            class Int_Decimal : RealConverter_NoWindow<int, decimal>
            {
                public override decimal Convert(int value) { return (decimal)value; }
            }


            #endregion


            #region FromUInt

            class uInt_Byte : RealConverter_NoWindow<uint, byte>
            {
                public override void Convert(uint[] from, int fromIndex, byte[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (byte)from[fromIndex++]; }

                public override byte Convert(uint value)
                { return (byte)value; }
            }


            class uInt_sByte : RealConverter_NoWindow<uint, sbyte>
            {
                public override void Convert(uint[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (sbyte)from[fromIndex++]; }

                public override sbyte Convert(uint value)
                { return (sbyte)value; }
            }

            class uInt_Short : RealConverter_NoWindow<uint, short>
            {
                public override void Convert(uint[] from, int fromIndex, short[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (short)from[fromIndex++]; }

                public override short Convert(uint value) { return (short)value; }
            }

            class uInt_uShort : RealConverter_NoWindow<uint, ushort>
            {
                public override void Convert(uint[] from, int fromIndex, ushort[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (ushort)from[fromIndex++]; }

                public override ushort Convert(uint value) { return (ushort)value; }
            }

            class uInt_Int : RealConverter_NoWindow<uint, int>
            {
                public override void Convert(uint[] from, int fromIndex, int[] to, int toIndex, int length)
                {for (int i = toIndex; i < toIndex + length; i++) to[i] = (int)from[fromIndex++];}

                public override int Convert(uint value) { return (int)value; }
            }

            class uInt_Long : RealConverter_NoWindow<uint, long>
            {
                public override long Convert(uint value) { return (long)value; }
            }

            class uInt_uLong : RealConverter_NoWindow<uint, ulong>
            {
                public override ulong Convert(uint value) { return (ulong)value; }
            }

            class uInt_Float : RealConverter_NoWindow<uint, float>
            {
                public override float Convert(uint value) { return (float)value; }
            }

            class uInt_Double : RealConverter_NoWindow<uint, double>
            {
                public override double Convert(uint value) { return (double)value; }
            }

            class uInt_Decimal : RealConverter_NoWindow<uint, decimal>
            {
                public override decimal Convert(uint value) { return (decimal)value; }
            }

            #endregion


            #region FromLong

            class Int64_Byte : RealConverter_NoWindow<long, byte>
            {
                public override void Convert(long[] from, int fromIndex, byte[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (byte)from[fromIndex++]; }

                public override byte Convert(long value) { return (byte)value; }
            }

            class Int64_sByte : RealConverter_NoWindow<long, sbyte>
            {
                public override void Convert(long[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (sbyte)from[fromIndex++]; }

                public override sbyte Convert(long value) { return (sbyte)value; }
            }

            class Long_Short : RealConverter_NoWindow<long, short>
            {
                public override void Convert(long[] from, int fromIndex, short[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (short)from[fromIndex++]; }

                public override short Convert(long value) { return (short)value; }
            }

            class Long_uShort : RealConverter_NoWindow<long, ushort>
            {
                public override void Convert(long[] from, int fromIndex, ushort[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (ushort)from[fromIndex++]; }

                public override ushort Convert(long value) { return (ushort)value; }
            }

            class Long_Int : RealConverter_NoWindow<long, int>
            {
                public override void Convert(long[] from, int fromIndex, int[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (int)from[fromIndex++]; }

                public override int Convert(long value) { return (int)value; }
            }

            class Long_uInt : RealConverter_NoWindow<long, uint>
            {
                public override void Convert(long[] from, int fromIndex, uint[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (uint)from[fromIndex++]; }

                public override uint Convert(long value) { return (uint)value; }
            }

            class Long_uLong : RealConverter_NoWindow<long, ulong>
            {
                public override void Convert(long[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {for (int i = toIndex; i < toIndex + length; i++) to[i] = (ulong)from[fromIndex++];}

                public override ulong Convert(long value) { return (ulong)value; }
            }

            class Long_Float : RealConverter_NoWindow<long, float>
            {
                public override float Convert(long value) { return (float)value; }
            }

            class Long_Double : RealConverter_NoWindow<long, double>
            {
                public override double Convert(long value) { return (double)value; }
            }

            class Long_Decimal : RealConverter_NoWindow<long, decimal>
            {
                public override decimal Convert(long value) { return (decimal)value; }
            }

            #endregion


            #region FromULong

            class uInt64_Byte : RealConverter_NoWindow<ulong, byte>
            {
                public override byte Convert(ulong value) { return (byte)value; }
                public override void Convert(ulong[] from, int fromIndex, byte[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (byte)from[fromIndex++]; }
            }

            class uInt64_sByte : RealConverter_NoWindow<ulong, sbyte>
            {
                public override sbyte Convert(ulong value) { return (sbyte)value; }
                public override void Convert(ulong[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                { for (int i = toIndex; i < toIndex + length; i++) to[i] = (sbyte)from[fromIndex++]; }
            }

            class uLong_Short : RealConverter_NoWindow<ulong, short>
            {
                public override short Convert(ulong value) { return (short)value; }
                public override void Convert(ulong[] from, int fromIndex, short[] to, int toIndex, int length) {
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (short)from[fromIndex++]; }
            }

            class uLong_uShort : RealConverter_NoWindow<ulong, ushort>
            {
                public override ushort Convert(ulong value) { return (ushort)value; }
                public override void Convert(ulong[] from, int fromIndex, ushort[] to, int toIndex, int length) {
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (ushort)from[fromIndex++]; }
            }

            class uLong_Int : RealConverter_NoWindow<ulong, int>
            {
                public override int Convert(ulong value) { return (int)value; }
                public override void Convert(ulong[] from, int fromIndex, int[] to, int toIndex, int length) {
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (int)from[fromIndex++]; }
            }

            class uLong_uInt : RealConverter_NoWindow<ulong, uint>
            {
                public override uint Convert(ulong value) { return (uint)value; }
                public override void Convert(ulong[] from, int fromIndex, uint[] to, int toIndex, int length) {
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (uint)from[fromIndex++]; }
            }

            class uLong_Long : RealConverter_NoWindow<ulong, long>
            {
                public override long Convert(ulong value) { return (long)value; }
                public override void Convert(ulong[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) to[i] = (long)from[fromIndex++];
                }                
            }

            class uLong_Float : RealConverter_NoWindow<ulong, float>
            {
                public override float Convert(ulong value) { return (float)value; }
            }

            class uLong_Double : RealConverter_NoWindow<ulong, double>
            {
                public override double Convert(ulong value) { return (double)value; }
            }

            class uLong_Decimal : RealConverter_NoWindow<ulong, decimal>
            {
                public override decimal Convert(ulong value) { return (decimal)value; }
            }

            #endregion


            #region FromFloat

            class Float_Byte : RealConverter_NoWindow<float, byte>
            {
                public override void Convert(float[] from, int fromIndex, byte[] to, int toIndex, int length) { 
                    for (int i = 0; i < length; i++) { to[toIndex++] = (byte)from[fromIndex++]; } }
                public override byte Convert(float value) { return (byte)value; }
            }

            class Float_sByte : RealConverter_NoWindow<float, sbyte>
            { 
                public override sbyte Convert(float value) { return (sbyte)value; }
                public override void Convert(float[] from, int fromIndex, sbyte[] to, int toIndex, int length) { 
                    for (int i = 0; i < length; i++) { to[toIndex++] = (sbyte)from[fromIndex++]; } }
            }

            class Float_Short : RealConverter_NoWindow<float, short>
            {
                public override short Convert(float value) { return (short)value; }
                public override void Convert(float[] from, int fromIndex, short[] to, int toIndex, int length) { 
                    for (int i = 0; i < length; i++) { to[toIndex++] = (short)from[fromIndex++]; } }
            }

            class Float_uShort : RealConverter_NoWindow<float, ushort>
            {
                public override ushort Convert(float value) { return (ushort)value; }
                public override void Convert(float[] from, int fromIndex, ushort[] to, int toIndex, int length) { 
                    for (int i = 0; i < length; i++) { to[toIndex++] = (ushort)from[fromIndex++]; } }
            }

            class Float_Int : RealConverter_NoWindow<float, int>
            {
                public override int Convert(float value) { return (int)value; }
                public override void Convert(float[] from, int fromIndex, int[] to, int toIndex, int length) { 
                    for (int i = 0; i < length; i++) { to[toIndex++] = (int)from[fromIndex++]; } }
            }

            class Float_uInt : RealConverter_NoWindow<float, uint>
            {
                public override uint Convert(float value) { return (uint)value; }
                public override void Convert(float[] from, int fromIndex, uint[] to, int toIndex, int length) { 
                    for (int i = 0; i < length; i++) { to[toIndex++] = (uint)from[fromIndex++]; } }
            }

            class Float_Long : RealConverter_NoWindow<float, long>
            {
                public override long Convert(float value) { return (long)value; }
                public override void Convert(float[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (long)from[fromIndex++]; }
                }
            }

            class Float_uLong : RealConverter_NoWindow<float, ulong>
            {
                public override ulong Convert(float value) { return (ulong)value; }
                public override void Convert(float[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (ulong)from[fromIndex++]; }
                }
            }

            class Float_Double : RealConverter_NoWindow<float, double>
            {
                public override double Convert(float value) { return (double)value; }
            }

            class Float_Decimal : RealConverter_NoWindow<float, decimal>
            {
                public override decimal Convert(float value) { return (decimal)value; }
            }

            #endregion


            #region FromDouble

            class Double_Byte : RealConverter_NoWindow<double, byte>
            {
                public override byte Convert(double value) { return (byte)value; }
                public override void Convert(double[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (byte)from[fromIndex++]; }
                }
            }

            class Double_sByte : RealConverter_NoWindow<double, sbyte>
            {
                public override sbyte Convert(double value) { return (sbyte)value; }
                public override void Convert(double[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (sbyte)from[fromIndex++]; }
                }
            }

            class Double_Short : RealConverter_NoWindow<double, short>
            {
                public override short Convert(double value) { return (short)value; }
                public override void Convert(double[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (short)from[fromIndex++]; }
                }
            }

            class Double_uShort : RealConverter_NoWindow<double, ushort>
            {
                public override ushort Convert(double value) { return (ushort)value; }
                public override void Convert(double[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (ushort)from[fromIndex++]; }
                }
            }

            class Double_Int : RealConverter_NoWindow<double, int>
            {
                public override int Convert(double value) { return (int)value; }
                public override void Convert(double[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (int)from[fromIndex++]; }
                }
            }

            class Double_uInt : RealConverter_NoWindow<double, uint>
            {
                public override uint Convert(double value) { return (uint)value; }
                public override void Convert(double[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (uint)from[fromIndex++]; }
                }
            }

            class Double_Long : RealConverter_NoWindow<double, long>
            {
                public override long Convert(double value) { return (long)value; }
                public override void Convert(double[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (long)from[fromIndex++]; }
                }
            }

            class Double_uLong : RealConverter_NoWindow<double, ulong>
            {
                public override ulong Convert(double value) { return (ulong)value; }
                public override void Convert(double[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (ulong)from[fromIndex++]; }
                }
            }

            class Double_Float : RealConverter_NoWindow<double, float>
            {
                public override void Convert(double[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = 0; i < length; i++) { to[toIndex++] = (float)from[fromIndex++]; }
                }
                public override float Convert(double value) { return (float)value; }
            }

            class Double_Decimal : RealConverter_NoWindow<double, decimal>
            {
                public override decimal Convert(double value) { return (decimal)value; }
            }

            #endregion


            #region FromDecimal

            class Decimal_Short : RealConverter_NoWindow<decimal, short>
            {
                public override short Convert(decimal value) { return (short)value; }
            }

            class Decimal_uShort : RealConverter_NoWindow<decimal, ushort>
            {
                public override ushort Convert(decimal value) { return (ushort)value; }
            }

            class Decimal_Int : RealConverter_NoWindow<decimal, int>
            {
                public override int Convert(decimal value) { return (int)value; }
            }

            class Decimal_uInt : RealConverter_NoWindow<decimal, uint>
            {
                public override uint Convert(decimal value) { return (uint)value; }
            }

            class Decimal_Long : RealConverter_NoWindow<decimal, long>
            {
                public override long Convert(decimal value) { return (long)value; }
            }

            class Decimal_uLong : RealConverter_NoWindow<decimal, ulong>
            {
                public override ulong Convert(decimal value) { return (ulong)value; }
            }


            class Decimal_Float : RealConverter_NoWindow<decimal, float>
            {
                public override float Convert(decimal value) { return (float)value; }
            }

            class Decimal_Double : RealConverter_NoWindow<decimal, double>
            {
                public override double Convert(decimal value) { return (double)value; }
            }

            #endregion

        }


        /// <summary>
        /// Value type converter that is already in same scale
        /// as the destination, but windowing checking is needed.
        /// 
        /// Example of windowing: source is sbyte value range from -30 to 40
        /// and the destination is uint (no negative values). Now all
        /// values are checked (smaller than 0), and put to 0 if so 
        /// </summary>
        /// <typeparam name="F">From - Source value type</typeparam>
        /// <typeparam name="T">To - Destination value type</typeparam>
        private abstract class RealConverter_window<F, T> : IConverter<F, T>
            where F : struct, IConvertible, IComparable
            where T : struct, IConvertible, IComparable
        {
            public RealConverter_window() { }
            public RealConverter_window(bool windowing) { }
            protected ConversionInfo info = new ConversionInfo(false, true);
            public ConversionInfo ConversionInfo { get { return info; } }
            public abstract T Convert(F value);
            public void Convert(F[] from, int fromIndex, T[] to, int toIndex, int length)
            {
                for (int i = 0; i < length; i++) { to[toIndex++] = Convert(from[fromIndex++]); }
            }
            public virtual void Convert(F[] from, int fromIndex, T[] to, int toIndex, int length, int jump)
            {
                for (int i = toIndex; i < toIndex + length; i++) { to[i] = Convert(from[fromIndex]); fromIndex += jump; }
            }
            public T[] Convert(F[] from)
            {
                T[] result = new T[from.Length];
                Convert(from, 0, result, 0, from.Length);
                return result;
            }

            public static RealConverter_window<F, T> Create(bool dataLoss)
            {
                Type from = typeof(F);
                Type to = typeof(T);
                RealConverter_window<F, T> c = null;

                if (to == typeof(Byte))
                {
                    if (from == typeof(sbyte)) c = new sByte_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(short)) c = new Short_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(ushort)) c = new uShort_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(char)) c = new Char_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(Int32)) c = new Int_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt32)) c = new uInt_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(Int64)) c = new Int64_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt64)) c = new uInt64_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(float)) c = new Float_Byte() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_Byte() as RealConverter_window<F, T>;
                }
                else if (to == typeof(SByte))
                {
                    if (from == typeof(byte)) c = new Byte_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(short)) c = new Short_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(ushort)) c = new uShort_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(char)) c = new Char_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(Int32)) c = new Int_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt32)) c = new uInt_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(Int64)) c = new Int64_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt64)) c = new uInt64_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(float)) c = new Float_sByte() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_sByte() as RealConverter_window<F, T>;
                }
                else if (to == typeof(short))
                {
                    if (from == typeof(ushort)) c = new uShort_Short() as RealConverter_window<F, T>;
                    else if (from == typeof(char)) c = new Char_Short() as RealConverter_window<F, T>;
                    else if (from == typeof(Int32)) c = new Int_Short() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt32)) c = new uInt_Short() as RealConverter_window<F, T>;
                    else if (from == typeof(Int64)) c = new Long_Short() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt64)) c = new uLong_Short() as RealConverter_window<F, T>;
                    else if (from == typeof(float)) c = new Float_Short() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_Short() as RealConverter_window<F, T>;
                    else if (from == typeof(decimal)) c = new Decimal_Short() as RealConverter_window<F, T>;
                }
                else if (to == typeof(ushort))
                {
                    if (from == typeof(sbyte)) c = new sByte_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(short)) c = new Short_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(char)) c = new Char_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(Int32)) c = new Int_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt32)) c = new uInt_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(Int64)) c = new Long_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt64)) c = new uLong_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(float)) c = new Float_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_uShort() as RealConverter_window<F, T>;
                    else if (from == typeof(decimal)) c = new Decimal_uShort() as RealConverter_window<F, T>;
                }
                else if (to == typeof(int))
                {
                    if (from == typeof(UInt32)) c = new uInt_Int() as RealConverter_window<F, T>;
                    else if (from == typeof(Int64)) c = new Long_Int() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt64)) c = new uLong_Int() as RealConverter_window<F, T>;
                    else if (from == typeof(float)) c = new Float_Int() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_Int() as RealConverter_window<F, T>;
                    else if (from == typeof(decimal)) c = new Decimal_Int() as RealConverter_window<F, T>;
                }
                else if (to == typeof(uint))
                {
                    if (from == typeof(sbyte)) c = new sByte_uInt() as RealConverter_window<F, T>;
                    else if (from == typeof(short)) c = new Short_uInt() as RealConverter_window<F, T>;
                    else if (from == typeof(Int32)) c = new Int_uInt() as RealConverter_window<F, T>;
                    else if (from == typeof(Int64)) c = new Long_uInt() as RealConverter_window<F, T>;
                    else if (from == typeof(UInt64)) c = new uLong_uInt() as RealConverter_window<F, T>;
                    else if (from == typeof(float)) c = new Float_uInt() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_uInt() as RealConverter_window<F, T>;
                    else if (from == typeof(decimal)) c = new Decimal_uInt() as RealConverter_window<F, T>;
                }
                else if (to == typeof(long))
                {
                    if (from == typeof(UInt64)) c = new uLong_Long() as RealConverter_window<F, T>;
                    else if (from == typeof(float)) c = new Float_Long() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_Long() as RealConverter_window<F, T>;
                    else if (from == typeof(decimal)) c = new Decimal_Long() as RealConverter_window<F, T>;
                }
                else if (to == typeof(ulong))
                {
                    if (from == typeof(sbyte)) c = new sByte_uLong() as RealConverter_window<F, T>;
                    else if (from == typeof(short)) c = new Short_uLong() as RealConverter_window<F, T>;
                    else if (from == typeof(int)) c = new Int_uLong() as RealConverter_window<F, T>;
                    else if (from == typeof(Int64)) c = new Long_uLong() as RealConverter_window<F, T>;
                    else if (from == typeof(float)) c = new Float_uLong() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_uLong() as RealConverter_window<F, T>;
                    else if (from == typeof(decimal)) c = new Decimal_uLong() as RealConverter_window<F, T>;
                }
                else if (to == typeof(float))
                {
                    if (from == typeof(double)) c = new Double_Float() as RealConverter_window<F, T>;
                    else if (from == typeof(decimal)) c = new Decimal_Float() as RealConverter_window<F, T>;
                }
                else if (to == typeof(decimal))
                {
                    if (from == typeof(float)) c = new Float_Decimal() as RealConverter_window<F, T>;
                    else if (from == typeof(double)) c = new Double_Decimal() as RealConverter_window<F, T>;
                }

                if (c == null) throw new NotImplementedException("The Real Space Windowing conversion class is not implemented from " + from.Name + " to " + to.Name);
                c.info = new ConversionInfo(dataLoss, true);
                return c;
            }

            #region ToByte

            class sByte_Byte : RealConverter_window<sbyte, byte>
            {
                public override byte Convert(sbyte value)
                {
                    if (value < 0) return 0;
                    return (byte)value;
                }
            }

            class Short_Byte : RealConverter_window<short, byte>
            {
                public override byte Convert(short value)
                {
                    if (value < 0) return 0;
                    else if (value > 255) return 255;
                    else return (byte)value;
                }
            }

            class uShort_Byte : RealConverter_window<ushort, byte>
            {
                public override byte Convert(ushort value)
                {
                    if (value > 255) return 255;
                    else return (byte)value;
                }
            }

            class Char_Byte : RealConverter_window<char, byte>
            {
                public override byte Convert(char value)
                {
                    if (value > 255) return 255;
                    else return (byte)value;
                }
            }

            class Int_Byte : RealConverter_window<int, byte>
            {
                public override byte Convert(int value)
                {
                    if (value < 0) return 0;
                    else if (value > 255) return 255;
                    else return (byte)value;
                }
            }

            class uInt_Byte : RealConverter_window<uint, byte>
            {
                public override byte Convert(uint value)
                {
                    if (value > 255) return 255;
                    else return (byte)value;
                }
            }

            class Int64_Byte : RealConverter_window<long, byte>
            {
                public override byte Convert(long value)
                {
                    if (value < 0) return 0;
                    else if (value > 255) return 255;
                    else return (byte)value;
                }
            }

            class uInt64_Byte : RealConverter_window<ulong, byte>
            {
                public override byte Convert(ulong value)
                {
                    if (value > 255) return 255;
                    else return (byte)value;
                }
            }



            class Float_Byte : RealConverter_window<float, byte>
            {
                public override byte Convert(float value)
                {
                    if (value < 0.0f) return 0;
                    else if (value > 255.0f) return 255;
                    else return (byte)value;
                }
            }



            class Double_Byte : RealConverter_window<double, byte>
            {
                public override byte Convert(double value)
                {
                    if (value < 0.0d) return 0;
                    else if (value > 255.0d) return 255;
                    else return (byte)value;
                }
            }

            #endregion


            #region ToSByte

            class Byte_sByte : RealConverter_window<byte, sbyte>
            {
                public override sbyte Convert(byte value)
                {
                    if (value > 127) return 127;
                    else return (sbyte)value;
                }
            }

            class Short_sByte : RealConverter_window<short, sbyte>
            {
                public override sbyte Convert(short value)
                {
                    if (value < -128) return -128;
                    else if (value > 127) return 127;
                    else return (sbyte)value;
                }
            }


            class uShort_sByte : RealConverter_window<ushort, sbyte>
            {
                public override sbyte Convert(ushort value)
                {
                    if (value < 0) return -128;
                    else if (value > 127) return 127;
                    else return (sbyte)value;
                }
            }

            class Char_sByte : RealConverter_window<char, sbyte>
            {
                public override sbyte Convert(char value)
                {
                    if (value > 127) return 127;
                    else return (sbyte)value;
                }
            }

            class Int_sByte : RealConverter_window<int, sbyte>
            {
                public override sbyte Convert(int value)
                {
                    if (value < -128) return -128;
                    else if (value > 127) return 127;
                    else return (sbyte)value;
                }
            }


            class uInt_sByte : RealConverter_window<uint, sbyte>
            {
                public override sbyte Convert(uint value)
                {
                    if (value > 127) return 127;
                    else return (sbyte)value;
                }
            }


            class Int64_sByte : RealConverter_window<long, sbyte>
            {
                public override sbyte Convert(long value)
                {
                    if (value < -128) return -128;
                    else if (value > 127) return 127;
                    else return (sbyte)value;
                }
            }

            class uInt64_sByte : RealConverter_window<ulong, sbyte>
            {
                public override sbyte Convert(ulong value)
                {
                    if (value > 127) return 127;
                    else return (sbyte)value;
                }
            }

            class Float_sByte : RealConverter_window<float, sbyte>
            {
                public override sbyte Convert(float value)
                {
                    if (value < -128.0f) return -128;
                    else if (value > 127.0f) return 127;
                    else return (sbyte)value;
                }
            }

            class Double_sByte : RealConverter_window<double, sbyte>
            {
                public override sbyte Convert(double value)
                {
                    if (value < -128.0d) return -128;
                    else if (value > 127.0d) return 127;
                    else return (sbyte)value;
                }
            }


            #endregion


            #region ToShort

            class uShort_Short : RealConverter_window<ushort, short>
            {
                public override short Convert(ushort value)
                {
                    if (value > short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            class Char_Short : RealConverter_window<char, short>
            {
                public override short Convert(char value)
                {
                    if (value > short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            class Int_Short : RealConverter_window<int, short>
            {
                public override short Convert(int value)
                {
                    if (value < short.MinValue) return short.MinValue;
                    else if (value > short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            class uInt_Short : RealConverter_window<uint, short>
            {
                public override short Convert(uint value)
                {
                    if (value > short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            class Long_Short : RealConverter_window<long, short>
            {
                public override short Convert(long value)
                {
                    if (value < short.MinValue) return short.MinValue;
                    else if (value > short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            class uLong_Short : RealConverter_window<ulong, short>
            {
                public override short Convert(ulong value)
                {
                    if (value > (ulong)short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            class Float_Short : RealConverter_window<float, short>
            {
                public override short Convert(float value)
                {
                    if (value < short.MinValue) return short.MinValue;
                    else if (value > short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            class Double_Short : RealConverter_window<double, short>
            {
                public override short Convert(double value)
                {
                    if (value < short.MinValue) return short.MinValue;
                    else if (value > short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            class Decimal_Short : RealConverter_window<decimal, short>
            {
                public override short Convert(decimal value)
                {
                    if (value < short.MinValue) return short.MinValue;
                    else if (value > short.MaxValue) return short.MaxValue;
                    else return (short)value;
                }
            }

            #endregion


            #region ToUShort

            class sByte_uShort : RealConverter_window<sbyte, ushort>
            {
                public override ushort Convert(sbyte value)
                {
                    if (value < 0) return 0;
                    else return (ushort)value;
                }
            }

            class Short_uShort : RealConverter_window<short, ushort>
            {
                public override ushort Convert(short value)
                {
                    if (value < 0) return 0;
                    else return (ushort)value;
                }
            }

            class Char_uShort : RealConverter_window<char, ushort>
            {
                public override ushort Convert(char value) { return (ushort)value; }
            }

            class Int_uShort : RealConverter_window<int, ushort>
            {
                public override ushort Convert(int value)
                {
                    if (value < ushort.MinValue) return ushort.MinValue;
                    else if (value > ushort.MaxValue) return ushort.MaxValue;
                    else return (ushort)value;
                }
            }

            class uInt_uShort : RealConverter_window<uint, ushort>
            {
                public override ushort Convert(uint value)
                {
                    if (value > ushort.MaxValue) return ushort.MaxValue;
                    else return (ushort)value;
                }
            }

            class Long_uShort : RealConverter_window<long, ushort>
            {
                public override ushort Convert(long value)
                {
                    if (value < 0) return 0;
                    else if (value > ushort.MaxValue) return ushort.MaxValue;
                    else return (ushort)value;
                }
            }

            class uLong_uShort : RealConverter_window<ulong, ushort>
            {
                public override ushort Convert(ulong value)
                {
                    if (value > ushort.MaxValue) return ushort.MaxValue;
                    else return (ushort)value;
                }
            }

            class Float_uShort : RealConverter_window<float, ushort>
            {
                public override ushort Convert(float value)
                {
                    if (value < 0.0f) return 0;
                    else if (value > ushort.MaxValue) return ushort.MaxValue;
                    else return (ushort)value;
                }
            }

            class Double_uShort : RealConverter_window<double, ushort>
            {
                public override ushort Convert(double value)
                {
                    if (value < 0.0d) return 0;
                    else if (value > ushort.MaxValue) return ushort.MaxValue;
                    else return (ushort)value;
                }
            }

            class Decimal_uShort : RealConverter_window<decimal, ushort>
            {
                public override ushort Convert(decimal value)
                {
                    if (value < ushort.MinValue) return ushort.MinValue;
                    else if (value > ushort.MaxValue) return ushort.MaxValue;
                    else return (ushort)value;
                }
            }

            #endregion


            #region ToInt

            class uInt_Int : RealConverter_window<uint, int>
            {
                public override int Convert(uint value)
                {
                    if (value > int.MaxValue) return int.MaxValue;
                    else return (int)value;
                }
            }

            class Long_Int : RealConverter_window<long, int>
            {
                public override int Convert(long value)
                {
                    if (value < int.MinValue) return int.MinValue;
                    else if (value > int.MaxValue) return int.MaxValue;
                    else return (int)value;
                }
            }

            class uLong_Int : RealConverter_window<ulong, int>
            {
                public override int Convert(ulong value)
                {
                    if (value > int.MaxValue) return int.MaxValue;
                    else return (int)value;
                }
            }

            class Float_Int : RealConverter_window<float, int>
            {
                public override int Convert(float value)
                {
                    if (value < (float)int.MinValue) return int.MinValue;
                    else if (value > (float)int.MaxValue) return int.MaxValue;
                    else return (int)value;
                }
            }

            class Double_Int : RealConverter_window<double, int>
            {
                public override int Convert(double value)
                {
                    if (value < (double)int.MinValue) return int.MinValue;
                    else if (value > (double)int.MaxValue) return int.MaxValue;
                    else return (int)value;
                }
            }

            class Decimal_Int : RealConverter_window<decimal, int>
            {
                public override int Convert(decimal value)
                {
                    if (value < int.MinValue) return int.MinValue;
                    else if (value > (decimal)int.MaxValue) return int.MaxValue;
                    else return (int)value;
                }
            }


            #endregion


            #region ToUInt

            class sByte_uInt : RealConverter_window<sbyte, uint>
            {
                public override uint Convert(sbyte value)
                {
                    if (value < 0) return 0;
                    else return (uint)value;
                }
            }

            class Short_uInt : RealConverter_window<short, uint>
            {
                public override uint Convert(short value)
                {
                    if (value < 0) return 0;
                    else return (uint)value;
                }
            }

            class Int_uInt : RealConverter_window<int, uint>
            {
                public override uint Convert(int value)
                {
                    if (value < 0) return 0;
                    else return (uint)value;
                }
            }

            class Long_uInt : RealConverter_window<long, uint>
            {
                public override uint Convert(long value)
                {
                    if (value < uint.MinValue) return uint.MinValue;
                    else if (value > uint.MaxValue) return uint.MaxValue;
                    else return (uint)value;
                }
            }

            class uLong_uInt : RealConverter_window<ulong, uint>
            {
                public override uint Convert(ulong value)
                {
                    if (value < uint.MinValue) return uint.MinValue;
                    else if (value > uint.MaxValue) return uint.MaxValue;
                    else return (uint)value;
                }
            }

            class Float_uInt : RealConverter_window<float, uint>
            {
                public override uint Convert(float value)
                {
                    if (value < (float)uint.MinValue) return uint.MinValue;
                    else if (value > (float)uint.MaxValue) return int.MaxValue;
                    else return (uint)value;
                }
            }

            class Double_uInt : RealConverter_window<double, uint>
            {
                public override uint Convert(double value)
                {
                    if (value < (double)uint.MinValue) return uint.MinValue;
                    else if (value > (double)uint.MaxValue) return uint.MaxValue;
                    else return (uint)value;
                }
            }

            class Decimal_uInt : RealConverter_window<decimal, uint>
            {
                public override uint Convert(decimal value)
                {
                    if (value < (decimal)uint.MinValue) return uint.MinValue;
                    else if (value > (decimal)uint.MaxValue) return uint.MaxValue;
                    else return (uint)value;
                }
            }

            #endregion


            #region ToLong


            class uLong_Long : RealConverter_window<ulong, long>
            {
                public override long Convert(ulong value)
                {
                    if (value > long.MaxValue) return long.MaxValue;
                    else return (long)value;
                }
            }

            class Float_Long : RealConverter_window<float, long>
            {
                public override long Convert(float value)
                {
                    if (value < (float)long.MinValue) return long.MinValue;
                    else if (value > (float)long.MaxValue) return long.MaxValue;
                    else return (long)value;
                }
            }

            class Double_Long : RealConverter_window<double, long>
            {
                public override long Convert(double value)
                {
                    if (value < (double)long.MinValue) return long.MinValue;
                    else if (value > (double)long.MaxValue) return long.MaxValue;
                    else return (long)value;
                }
            }

            class Decimal_Long : RealConverter_window<decimal, long>
            {
                public override long Convert(decimal value)
                {
                    if (value < (decimal)long.MinValue) return long.MinValue;
                    else if (value > (decimal)long.MaxValue) return long.MaxValue;
                    else return (long)value;
                }
            }

            #endregion


            #region ToULong

            class sByte_uLong : RealConverter_window<sbyte, ulong>
            {
                public override ulong Convert(sbyte value)
                {
                    if (value < 0) return 0;
                    else return (ulong)value;
                }
            }

            class Short_uLong : RealConverter_window<short, ulong>
            {
                public override ulong Convert(short value)
                {
                    if (value < 0) return 0;
                    else return (ulong)value;
                }
            }

            class Int_uLong : RealConverter_window<int, ulong>
            {
                public override ulong Convert(int value)
                {
                    if (value < 0) return 0;
                    else return (ulong)value;
                }
            }

            class Long_uLong : RealConverter_window<long, ulong>
            {
                public override ulong Convert(long value)
                {
                    if (value < 0) return 0;
                    else return (ulong)value;
                }
            }

            class Float_uLong : RealConverter_window<float, ulong>
            {
                public override ulong Convert(float value)
                {
                    if (value < 0.0f) return 0;
                    else if (value > (float)ulong.MaxValue) return ulong.MaxValue;
                    else return (ulong)value;
                }
            }

            class Double_uLong : RealConverter_window<double, ulong>
            {
                public override ulong Convert(double value)
                {
                    if (value < (double)ulong.MinValue) return ulong.MinValue;
                    else if (value > (double)ulong.MaxValue) return ulong.MaxValue;
                    else return (ulong)value;
                }
            }

            class Decimal_uLong : RealConverter_window<decimal, ulong>
            {
                public override ulong Convert(decimal value)
                {
                    if (value < (decimal)ulong.MinValue) return ulong.MinValue;
                    else if (value > (decimal)ulong.MaxValue) return ulong.MaxValue;
                    else return (ulong)value;
                }
            }

            #endregion


            #region ToFloat

            class Double_Float : RealConverter_window<double, float>
            {
                public override float Convert(double value)
                {
                    if (value < (double)float.MinValue) return float.MinValue;
                    else if (value > (double)float.MaxValue) return float.MaxValue;
                    else return (float)value;
                }
            }

            class Decimal_Float : RealConverter_window<decimal, float>
            {
                public override float Convert(decimal value)
                {
                    return (float)value;
                }
            }

            #endregion


            #region ToDecimal

            class Float_Decimal : RealConverter_window<float, decimal>
            {
                public override decimal Convert(float value)
                {
                    if (value < (float)decimal.MinValue) return decimal.MinValue;
                    else if (value > (float)decimal.MaxValue) return decimal.MaxValue;
                    else return (decimal)value;
                }
            }

            class Double_Decimal : RealConverter_window<double, decimal>
            {
                public override decimal Convert(double value)
                {
                    if (value < (double)decimal.MinValue) return decimal.MinValue;
                    else if (value > (double)decimal.MaxValue) return decimal.MaxValue;
                    else return (decimal)value;
                }
            }

            #endregion

        }

        #endregion

        #region RescaledTypeConversion


        /// <summary>
        /// Rescaled Value type converter that needs no windowing check
        /// </summary>
        /// <typeparam name="F">From - Source value type</typeparam>
        /// <typeparam name="T">To - Destination value type</typeparam>
        private abstract class Rescale_NoWindow<F, T> : IConverter<F, T>
            where F : struct, IConvertible, IComparable
            where T : struct, IConvertible, IComparable
        {
            protected double factor;
            protected double intersection;

            protected ConversionInfo info = new ConversionInfo(false, false);
            public ConversionInfo ConversionInfo { get { return info; } }
            public abstract T Convert(F value);
            public virtual void Convert(F[] from, int fromIndex, T[] to, int toIndex, int length)
            {
                Array.Copy(from, fromIndex, to, toIndex, length);
            }
            public virtual void Convert(F[] from, int fromIndex, T[] to, int toIndex, int length, int jump)
            {
                for (int i = toIndex; i < toIndex + length; i++) { to[i] = Convert(from[fromIndex]); fromIndex += jump; }
            }
            public T[] Convert(F[] from)
            {
                T[] result = new T[from.Length];
                Convert(from, 0, result, 0, from.Length);
                return result;
            }

            public static Rescale_NoWindow<F, T> Create(bool dataLoss, double factor, double intersection)
            {
                Type to = typeof(T);
                Type from = typeof(F);
                Rescale_NoWindow<F, T> c = null;

                if (from == typeof(Byte))
                {
                    if (to == typeof(byte)) c = new Byte_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Byte_sByte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Byte_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Byte_uShort() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int32)) c = new Byte_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt32)) c = new Byte_uInt() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new Byte_Int64() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new Byte_uInt64() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Byte_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Byte_Double() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(SByte))
                {
                    if (to == typeof(byte)) c = new sByte_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new sByte_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new sByte_uShort() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int32)) c = new sByte_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt32)) c = new sByte_uInt() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new sByte_Int64() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new sByte_uInt64() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new sByte_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new sByte_Double() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(short))
                {
                    if (to == typeof(byte)) c = new Short_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Short_sByte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Short_uShort() as Rescale_NoWindow<F, T>;

                    else if (to == typeof(ushort)) c = new Short_uShort() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int32)) c = new Short_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt32)) c = new Short_uInt() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new Short_Long() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new Short_uLong() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Short_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Short_Double() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Short_Decimal() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(ushort))
                {
                    if (to == typeof(byte)) c = new uShort_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new uShort_sByte() as Rescale_NoWindow<F, T>;

                    else if (to == typeof(short)) c = new uShort_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int32)) c = new uShort_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt32)) c = new uShort_uInt() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new uShort_Long() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new uShort_uLong() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new uShort_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new uShort_Double() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new uShort_Decimal() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(int))
                {
                    if (to == typeof(byte)) c = new Int_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Int_sByte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Int_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Int_uShort() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Int_uInt() as Rescale_NoWindow<F, T>;

                    else if (to == typeof(Int64)) c = new Int_Long() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new Int_uLong() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Int_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Int_Double() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Int_Decimal() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(uint))
                {
                    if (to == typeof(byte)) c = new uInt_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new uInt_sByte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new uInt_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new uInt_uShort() as Rescale_NoWindow<F, T>;

                    else if (to == typeof(Int32)) c = new uInt_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(Int64)) c = new uInt_Long() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(UInt64)) c = new uInt_uLong() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new uInt_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new uInt_Double() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new uInt_Decimal() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(long))
                {
                    if (to == typeof(byte)) c = new Int64_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Int64_sByte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Long_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Long_uShort() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new Long_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Long_uInt() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ulong)) c = new Long_uLong() as Rescale_NoWindow<F, T>;

                    else if (to == typeof(UInt64)) c = new Long_uLong() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Long_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Long_Double() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Long_Decimal() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(ulong))
                {
                    if (to == typeof(byte)) c = new uInt64_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new uInt64_sByte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new uLong_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new uLong_uShort() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new uLong_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new uLong_uInt() as Rescale_NoWindow<F, T>;

                    else if (to == typeof(long)) c = new uLong_Long() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new uLong_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new uLong_Double() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new uLong_Decimal() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(float))
                {
                    if (to == typeof(byte)) c = new Float_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Float_sByte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Float_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Float_uShort() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new Float_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Float_uInt() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(long)) c = new Float_Long() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ulong)) c = new Float_uLong() as Rescale_NoWindow<F, T>;

                    else if (to == typeof(float)) c = new Float_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Float_Double() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Float_Decimal() as Rescale_NoWindow<F, T>;
                }
                else if (from == typeof(double))
                {
                    if (to == typeof(byte)) c = new Double_Byte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(sbyte)) c = new Double_sByte() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(short)) c = new Double_Short() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ushort)) c = new Double_uShort() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(int)) c = new Double_Int() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(uint)) c = new Double_uInt() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(long)) c = new Double_Long() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(ulong)) c = new Double_uLong() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(float)) c = new Double_Float() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(double)) c = new Double_Double() as Rescale_NoWindow<F, T>;
                    else if (to == typeof(decimal)) c = new Double_Decimal() as Rescale_NoWindow<F, T>;
                }

                if (c == null) throw new NotImplementedException(
                    "There is no Rescaled method implemented to convert value scale " + from.Name + " to " + to.Name);

                c.info = new ConversionInfo(dataLoss, false);
                c.factor = factor;
                c.intersection = intersection;
                return c;
            }

            #region FromByte

            class Byte_Byte : Rescale_NoWindow<byte, byte>
            {
                public override byte Convert(byte value) { return (byte)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, byte[] to, int toIndex, int length) {
                    for (int i = toIndex; i < toIndex + length; i++) {
                        to[i] = (byte)(from[fromIndex++] * factor + intersection); }}                
            }

            class Byte_sByte : Rescale_NoWindow<byte, sbyte>
            {
                public override sbyte Convert(byte value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) {to[i] = (sbyte)(from[fromIndex++] * factor + intersection);}
                }   
            }

            class Byte_Short : Rescale_NoWindow<byte, short>
            {
                public override short Convert(byte value) { return (short)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class Byte_uShort : Rescale_NoWindow<byte, ushort>
            {
                public override ushort Convert(byte value) { return (ushort)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class Byte_Int : Rescale_NoWindow<byte, int>
            {
                public override int Convert(byte value) { return (int)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class Byte_uInt : Rescale_NoWindow<byte, uint>
            {
                public override uint Convert(byte value) { return (uint)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class Byte_Int64 : Rescale_NoWindow<byte, long>
            {
                public override long Convert(byte value) { return (long)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class Byte_uInt64 : Rescale_NoWindow<byte, ulong>
            {
                public override ulong Convert(byte value) { return (ulong)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class Byte_Float : Rescale_NoWindow<byte, float>
            {
                public override float Convert(byte value) { return (float)(value * factor + intersection); }
                public override void Convert(byte[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class Byte_Double : Rescale_NoWindow<byte, double>
            {
                public override double Convert(byte value) { return (double)value * factor + intersection; }
                public override void Convert(byte[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }

            #endregion


            #region FromSByte

            class sByte_Byte : Rescale_NoWindow<sbyte, byte>
            {
                public override byte Convert(sbyte value) { return (byte)(value * factor + intersection); }
                public override void Convert(sbyte[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class sByte_Short : Rescale_NoWindow<sbyte, short>
            {
                public override short Convert(sbyte value) { return (short)(value * factor + intersection); }
                public override void Convert(sbyte[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                } 
            }


            class sByte_uShort : Rescale_NoWindow<sbyte, ushort>
            {
                public override ushort Convert(sbyte value) { return (ushort)(value * factor + intersection); }
                public override void Convert(sbyte[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                } 
            }


            class sByte_Int : Rescale_NoWindow<sbyte, int>
            {
                public override int Convert(sbyte value) { return (int)(value * factor + intersection); }
                public override void Convert(sbyte[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                } 
            }


            class sByte_uInt : Rescale_NoWindow<sbyte, uint>
            {
                public override uint Convert(sbyte value) { return (uint)(value * factor + intersection); }
                public override void Convert(sbyte[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                } 
            }


            class sByte_Int64 : Rescale_NoWindow<sbyte, long>
            {
                public override long Convert(sbyte value) { return (long)(value * factor + intersection); }
                public override void Convert(sbyte[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class sByte_uInt64 : Rescale_NoWindow<sbyte, ulong>
            {
                public override ulong Convert(sbyte value) { return (ulong)(value * factor + intersection); }
                public override void Convert(sbyte[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                } 
            }

            class sByte_Float : Rescale_NoWindow<sbyte, float>
            {
                public override float Convert(sbyte value) { return (float)(value * factor + intersection); }
                public override void Convert(sbyte[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class sByte_Double : Rescale_NoWindow<sbyte, double>
            {
                public override double Convert(sbyte value) { return (double)value * factor + intersection; }
                public override void Convert(sbyte[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }


            #endregion


            #region FromShort

            class Short_Byte : Rescale_NoWindow<short, byte>
            {
                public override byte Convert(short value) { return (byte)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Short_sByte : Rescale_NoWindow<short, sbyte>
            {
                public override sbyte Convert(short value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (sbyte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Short_uShort : Rescale_NoWindow<short, ushort>
            {
                public override ushort Convert(short value) { return (ushort)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Short_Int : Rescale_NoWindow<short, int>
            {
                public override int Convert(short value) { return (int)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Short_uInt : Rescale_NoWindow<short, uint>
            {
                public override uint Convert(short value) { return (uint)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Short_Long : Rescale_NoWindow<short, long>
            {
                public override long Convert(short value) { return (long)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Short_uLong : Rescale_NoWindow<short, ulong>
            {
                public override ulong Convert(short value) { return (ulong)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Short_Float : Rescale_NoWindow<short, float>
            {
                public override float Convert(short value) { return (float)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Short_Double : Rescale_NoWindow<short, double>
            {
                public override double Convert(short value) { return (double)value * factor + intersection; }
                public override void Convert(short[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }

            class Short_Decimal : Rescale_NoWindow<short, decimal>
            {
                public override decimal Convert(short value) { return (decimal)(value * factor + intersection); }
                public override void Convert(short[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (decimal)(from[fromIndex++] * factor + intersection); }
                }
            }

            #endregion


            #region FromUShort

            class uShort_sByte : Rescale_NoWindow<ushort, sbyte>
            {
                public override sbyte Convert(ushort value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (sbyte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uShort_Byte : Rescale_NoWindow<ushort, byte>
            {
                public override byte Convert(ushort value) { return (byte)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uShort_Short : Rescale_NoWindow<ushort, short>
            {
                public override short Convert(ushort value) { return (short)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uShort_Int : Rescale_NoWindow<ushort, int>
            {
                public override int Convert(ushort value) { return (int)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uShort_uInt : Rescale_NoWindow<ushort, uint>
            {
                public override uint Convert(ushort value) { return (uint)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uShort_Long : Rescale_NoWindow<ushort, long>
            {
                public override long Convert(ushort value) { return (long)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uShort_uLong : Rescale_NoWindow<ushort, ulong>
            {
                public override ulong Convert(ushort value) { return (ulong)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uShort_Float : Rescale_NoWindow<ushort, float>
            {
                public override float Convert(ushort value) { return (float)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uShort_Double : Rescale_NoWindow<ushort, double>
            {
                public override double Convert(ushort value) { return (double)value * factor + intersection; }
                public override void Convert(ushort[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }

            class uShort_Decimal : Rescale_NoWindow<ushort, decimal>
            {
                public override decimal Convert(ushort value) { return (decimal)(value * factor + intersection); }
                public override void Convert(ushort[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (decimal)(from[fromIndex++] * factor + intersection); }
                }
            }

            #endregion


            #region FromInt


            class Int_Byte : Rescale_NoWindow<int, byte>
            {
                public override byte Convert(int value) { return (byte)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Int_sByte : Rescale_NoWindow<int, sbyte>
            {
                public override sbyte Convert(int value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (sbyte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Int_Short : Rescale_NoWindow<int, short>
            {
                public override short Convert(int value) { return (short)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Int_uShort : Rescale_NoWindow<int, ushort>
            {
                public override ushort Convert(int value) { return (ushort)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                }
            }


            class Int_uInt : Rescale_NoWindow<int, uint>
            {
                public override uint Convert(int value) { return (uint)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Int_Long : Rescale_NoWindow<int, long>
            {
                public override long Convert(int value) { return (long)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Int_uLong : Rescale_NoWindow<int, ulong>
            {
                public override ulong Convert(int value) { return (ulong)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Int_Float : Rescale_NoWindow<int, float>
            {
                public override float Convert(int value) { return (float)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Int_Double : Rescale_NoWindow<int, double>
            {
                public override double Convert(int value) { return (double)value * factor + intersection; }
                public override void Convert(int[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }

            class Int_Decimal : Rescale_NoWindow<int, decimal>
            {
                public override decimal Convert(int value) { return (decimal)(value * factor + intersection); }
                public override void Convert(int[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (decimal)(from[fromIndex++] * factor + intersection); }
                }
            }


            #endregion


            #region FromUInt

            class uInt_Byte : Rescale_NoWindow<uint, byte>
            {
                public override byte Convert(uint value) { return (byte)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                }
            }


            class uInt_sByte : Rescale_NoWindow<uint, sbyte>
            {
                public override sbyte Convert(uint value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (sbyte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uInt_Short : Rescale_NoWindow<uint, short>
            {
                public override short Convert(uint value) { return (short)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uInt_uShort : Rescale_NoWindow<uint, ushort>
            {
                public override ushort Convert(uint value) { return (ushort)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uInt_Int : Rescale_NoWindow<uint, int>
            {
                public override int Convert(uint value) { return (int)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uInt_Long : Rescale_NoWindow<uint, long>
            {
                public override long Convert(uint value) { return (long)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uInt_uLong : Rescale_NoWindow<uint, ulong>
            {
                public override ulong Convert(uint value) { return (ulong)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uInt_Float : Rescale_NoWindow<uint, float>
            {
                public override float Convert(uint value) { return (float)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uInt_Double : Rescale_NoWindow<uint, double>
            {
                public override double Convert(uint value) { return (double)value * factor + intersection; }
                public override void Convert(uint[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }

            class uInt_Decimal : Rescale_NoWindow<uint, decimal>
            {
                public override decimal Convert(uint value) { return (decimal)(value * factor + intersection); }
                public override void Convert(uint[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (decimal)(from[fromIndex++] * factor + intersection); }
                }
            }

            #endregion


            #region FromLong

            class Int64_Byte : Rescale_NoWindow<long, byte>
            {
                public override byte Convert(long value) { return (byte)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Int64_sByte : Rescale_NoWindow<long, sbyte>
            {
                public override sbyte Convert(long value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (sbyte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Long_Short : Rescale_NoWindow<long, short>
            {
                public override short Convert(long value) { return (short)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Long_uShort : Rescale_NoWindow<long, ushort>
            {
                public override ushort Convert(long value) { return (ushort)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Long_Int : Rescale_NoWindow<long, int>
            {
                public override int Convert(long value) { return (int)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Long_uInt : Rescale_NoWindow<long, uint>
            {
                public override uint Convert(long value) { return (uint)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Long_uLong : Rescale_NoWindow<long, ulong>
            {
                public override ulong Convert(long value) { return (ulong)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Long_Float : Rescale_NoWindow<long, float>
            {
                public override float Convert(long value) { return (float)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Long_Double : Rescale_NoWindow<long, double>
            {
                public override double Convert(long value) { return (double)value * factor + intersection; }
                public override void Convert(long[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }

            class Long_Decimal : Rescale_NoWindow<long, decimal>
            {
                public override decimal Convert(long value) { return (decimal)(value * factor + intersection); }
                public override void Convert(long[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (decimal)(from[fromIndex++] * factor + intersection); }
                }
            }

            #endregion


            #region FromULong

            class uInt64_Byte : Rescale_NoWindow<ulong, byte>
            {
                public override byte Convert(ulong value) { return (byte)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uInt64_sByte : Rescale_NoWindow<ulong, sbyte>
            {
                public override sbyte Convert(ulong value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (sbyte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uLong_Short : Rescale_NoWindow<ulong, short>
            {
                public override short Convert(ulong value) { return (short)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uLong_uShort : Rescale_NoWindow<ulong, ushort>
            {
                public override ushort Convert(ulong value) { return (ushort)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uLong_Int : Rescale_NoWindow<ulong, int>
            {
                public override int Convert(ulong value) { return (int)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uLong_uInt : Rescale_NoWindow<ulong, uint>
            {
                public override uint Convert(ulong value) { return (uint)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uLong_Long : Rescale_NoWindow<ulong, long>
            {
                public override long Convert(ulong value) { return (long)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uLong_Float : Rescale_NoWindow<ulong, float>
            {
                public override float Convert(ulong value) { return (float)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class uLong_Double : Rescale_NoWindow<ulong, double>
            {
                public override double Convert(ulong value) { return (double)value * factor + intersection; }
                public override void Convert(ulong[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }

            class uLong_Decimal : Rescale_NoWindow<ulong, decimal>
            {
                public override decimal Convert(ulong value) { return (decimal)(value * factor + intersection); }
                public override void Convert(ulong[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (decimal)(from[fromIndex++] * factor + intersection); }
                }
            }

            #endregion


            #region FromFloat

            class Float_Byte : Rescale_NoWindow<float, byte>
            {
                public override byte Convert(float value) { return (byte)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_sByte : Rescale_NoWindow<float, sbyte>
            {
                public override sbyte Convert(float value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (sbyte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_Short : Rescale_NoWindow<float, short>
            {
                public override short Convert(float value) { return (short)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_uShort : Rescale_NoWindow<float, ushort>
            {
                public override ushort Convert(float value) { return (ushort)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_Int : Rescale_NoWindow<float, int>
            {
                public override int Convert(float value) { return (int)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_uInt : Rescale_NoWindow<float, uint>
            {
                public override uint Convert(float value) { return (uint)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_Long : Rescale_NoWindow<float, long>
            {
                public override long Convert(float value) { return (long)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_uLong : Rescale_NoWindow<float, ulong>
            {
                public override ulong Convert(float value) { return (ulong)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_Float : Rescale_NoWindow<float, float>
            {
                public override float Convert(float value) { return (float)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Float_Double : Rescale_NoWindow<float, double>
            {
                public override double Convert(float value) { return (double)value * factor + intersection; }
                public override void Convert(float[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (double)from[fromIndex++] * factor + intersection; }
                }
            }

            class Float_Decimal : Rescale_NoWindow<float, decimal>
            {
                public override decimal Convert(float value) { return (decimal)(value * factor + intersection); }
                public override void Convert(float[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (decimal)(from[fromIndex++] * factor + intersection); }
                }
            }

            #endregion


            #region FromDouble

            class Double_Byte : Rescale_NoWindow<double, byte>
            {
                public override byte Convert(double value) { return (byte)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (byte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_sByte : Rescale_NoWindow<double, sbyte>
            {
                public override sbyte Convert(double value) { return (sbyte)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (sbyte)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_Short : Rescale_NoWindow<double, short>
            {
                public override short Convert(double value) { return (short)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (short)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_uShort : Rescale_NoWindow<double, ushort>
            {
                public override ushort Convert(double value) { return (ushort)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ushort)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_Int : Rescale_NoWindow<double, int>
            {
                public override int Convert(double value) { return (int)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (int)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_uInt : Rescale_NoWindow<double, uint>
            {
                public override uint Convert(double value) { return (uint)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (uint)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_Long : Rescale_NoWindow<double, long>
            {
                public override long Convert(double value) { return (long)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (long)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_uLong : Rescale_NoWindow<double, ulong>
            {
                public override ulong Convert(double value) { return (ulong)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (ulong)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_Float : Rescale_NoWindow<double, float>
            {
                public override float Convert(double value) { return (float)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (float)(from[fromIndex++] * factor + intersection); }
                }
            }

            class Double_Double : Rescale_NoWindow<double, double>
            {
                public override double Convert(double value) { return value * factor + intersection; }
                public override void Convert(double[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = from[fromIndex++] * factor + intersection; }
                }
            }

            class Double_Decimal : Rescale_NoWindow<double, decimal>
            {
                public override decimal Convert(double value) { return (decimal)(value * factor + intersection); }
                public override void Convert(double[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++) { to[i] = (decimal)(from[fromIndex++] * factor + intersection); }
                }
            }


            #endregion


        }


        /// <summary>
        /// Rescaled Value type converter that needs no windowing check
        /// </summary>
        /// <typeparam name="F">From - Source value type</typeparam>
        /// <typeparam name="T">To - Destination value type</typeparam>
        private abstract class Rescale_Windowing<F, T> : IConverter<F, T>
            where F : struct, IConvertible, IComparable
            where T : struct, IConvertible, IComparable
        {
            protected double factor;
            protected double intersection;

            F firstFrom_Inside;
            F lastFrom_Inside;

            protected ConversionInfo info = new ConversionInfo(false, false);
            public ConversionInfo ConversionInfo { get { return info; } }
            public abstract T Convert(F value);
            public virtual void Convert(F[] from, int fromIndex, T[] to, int toIndex, int length)
            {
                Array.Copy(from, fromIndex, to, toIndex, length);
            }
            public virtual void Convert(F[] from, int fromIndex, T[] to, int toIndex, int length, int jump)
            {
                for (int i = toIndex; i < toIndex + length; i++) { to[i] = Convert(from[fromIndex]); fromIndex += jump; }
            }
            public T[] Convert(F[] from)
            {
                T[] result = new T[from.Length];
                Convert(from, 0, result, 0, from.Length);
                return result;
            }

            public static Rescale_Windowing<F, T> Create(bool dataLoss, double factor, double intersection, F firstFromInside, F lastFromInside)
            {
                Type to = typeof(T);
                Type from = typeof(F);
                Rescale_Windowing<F, T> c = null;

                if (from == typeof(Byte))
                {
                    if (to == typeof(byte)) c = new Byte_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new Byte_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new Byte_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new Byte_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int32)) c = new Byte_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt32)) c = new Byte_uInt() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int64)) c = new Byte_Int64() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt64)) c = new Byte_uInt64() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new Byte_Float() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(SByte))
                {
                    if (to == typeof(byte)) c = new sByte_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new sByte_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new sByte_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new sByte_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int32)) c = new sByte_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt32)) c = new sByte_uInt() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int64)) c = new sByte_Int64() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt64)) c = new sByte_uInt64() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new sByte_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new sByte_Double() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(short))
                {
                    if (to == typeof(byte)) c = new Short_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new Short_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new Short_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new Short_uShort() as Rescale_Windowing<F, T>;

                    else if (to == typeof(ushort)) c = new Short_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int32)) c = new Short_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt32)) c = new Short_uInt() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int64)) c = new Short_Long() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt64)) c = new Short_uLong() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new Short_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new Short_Double() as Rescale_Windowing<F, T>;
                    else if (to == typeof(decimal)) c = new Short_Decimal() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(ushort))
                {
                    if (to == typeof(byte)) c = new uShort_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new uShort_sByte() as Rescale_Windowing<F, T>;

                    else if (to == typeof(short)) c = new uShort_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new uShort_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int32)) c = new uShort_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt32)) c = new uShort_uInt() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int64)) c = new uShort_Long() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt64)) c = new uShort_uLong() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new uShort_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new uShort_Double() as Rescale_Windowing<F, T>;
                    else if (to == typeof(decimal)) c = new uShort_Decimal() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(int))
                {
                    if (to == typeof(byte)) c = new Int_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new Int_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new Int_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new Int_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(int)) c = new Int_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(uint)) c = new Int_uInt() as Rescale_Windowing<F, T>;

                    else if (to == typeof(Int64)) c = new Int_Long() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt64)) c = new Int_uLong() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new Int_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new Int_Double() as Rescale_Windowing<F, T>;
                    else if (to == typeof(decimal)) c = new Int_Decimal() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(uint))
                {
                    if (to == typeof(byte)) c = new uInt_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new uInt_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new uInt_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new uInt_uShort() as Rescale_Windowing<F, T>;

                    else if (to == typeof(uint)) c = new uInt_uInt() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int32)) c = new uInt_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(Int64)) c = new uInt_Long() as Rescale_Windowing<F, T>;
                    else if (to == typeof(UInt64)) c = new uInt_uLong() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new uInt_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new uInt_Double() as Rescale_Windowing<F, T>;
                    else if (to == typeof(decimal)) c = new uInt_Decimal() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(long))
                {
                    if (to == typeof(byte)) c = new Int64_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new Int64_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new Long_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new Long_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(int)) c = new Long_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(uint)) c = new Long_uInt() as Rescale_Windowing<F, T>;
                    else if (to == typeof(long)) c = new Long_Long() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ulong)) c = new Long_uLong() as Rescale_Windowing<F, T>;

                    else if (to == typeof(UInt64)) c = new Long_uLong() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new Long_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new Long_Double() as Rescale_Windowing<F, T>;
                    else if (to == typeof(decimal)) c = new Long_Decimal() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(ulong))
                {
                    if (to == typeof(byte)) c = new uInt64_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new uInt64_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new uLong_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new uLong_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(int)) c = new uLong_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(uint)) c = new uLong_uInt() as Rescale_Windowing<F, T>;

                    else if (to == typeof(long)) c = new uLong_Long() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ulong)) c = new uLong_uLong() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new uLong_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new uLong_Double() as Rescale_Windowing<F, T>;
                    else if (to == typeof(decimal)) c = new uLong_Decimal() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(float))
                {
                    if (to == typeof(byte)) c = new Float_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new Float_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new Float_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new Float_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(int)) c = new Float_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(uint)) c = new Float_uInt() as Rescale_Windowing<F, T>;
                    else if (to == typeof(long)) c = new Float_Long() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ulong)) c = new Float_uLong() as Rescale_Windowing<F, T>;

                    else if (to == typeof(float)) c = new Float_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new Float_Double() as Rescale_Windowing<F, T>;
                    else if (to == typeof(decimal)) c = new Float_Decimal() as Rescale_Windowing<F, T>;
                }
                else if (from == typeof(double))
                {
                    if (to == typeof(byte)) c = new Double_Byte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(sbyte)) c = new Double_sByte() as Rescale_Windowing<F, T>;
                    else if (to == typeof(short)) c = new Double_Short() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ushort)) c = new Double_uShort() as Rescale_Windowing<F, T>;
                    else if (to == typeof(int)) c = new Double_Int() as Rescale_Windowing<F, T>;
                    else if (to == typeof(uint)) c = new Double_uInt() as Rescale_Windowing<F, T>;
                    else if (to == typeof(long)) c = new Double_Long() as Rescale_Windowing<F, T>;
                    else if (to == typeof(ulong)) c = new Double_uLong() as Rescale_Windowing<F, T>;
                    else if (to == typeof(float)) c = new Double_Float() as Rescale_Windowing<F, T>;
                    else if (to == typeof(double)) c = new Double_Double() as Rescale_Windowing<F, T>;
                    else if (to == typeof(decimal)) c = new Double_Decimal() as Rescale_Windowing<F, T>;
                }

                if (c == null) throw new NotImplementedException(
                    "There is no Rescaled+Windowed method implemented to convert value scale " + from.Name + " to " + to.Name);

                c.info = new ConversionInfo(dataLoss, true);
                c.factor = factor;
                c.intersection = intersection;
                c.firstFrom_Inside = firstFromInside;
                c.lastFrom_Inside = lastFromInside;
                return c;
            }

            #region FromByte

            class Byte_Byte : Rescale_Windowing<byte, byte>
            {
                public override void Convert(byte[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = 0;
                        else if (val > lastFrom_Inside) to[i] = 255;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(byte value)
                {
                    if (value < firstFrom_Inside) return 0;
                    else if (value > lastFrom_Inside) return 255;
                    else return (byte)(value * factor + intersection);
                }
            }

            class Byte_sByte : Rescale_Windowing<byte, sbyte>
            {
                public override void Convert(byte[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(byte value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class Byte_Short : Rescale_Windowing<byte, short>
            {
                public override void Convert(byte[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(byte value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class Byte_uShort : Rescale_Windowing<byte, ushort>
            {
                public override void Convert(byte[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(byte value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class Byte_Int : Rescale_Windowing<byte, int>
            {
                public override void Convert(byte[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(byte value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class Byte_uInt : Rescale_Windowing<byte, uint>
            {
                public override void Convert(byte[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(byte value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class Byte_Int64 : Rescale_Windowing<byte, long>
            {
                public override void Convert(byte[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(byte value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class Byte_uInt64 : Rescale_Windowing<byte, ulong>
            {
                public override void Convert(byte[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(byte value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class Byte_Float : Rescale_Windowing<byte, float>
            {
                public override void Convert(byte[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        byte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(byte value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }


            #endregion


            #region FromSByte

            class sByte_Byte : Rescale_Windowing<sbyte, byte>
            {
                public override void Convert(sbyte[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }

            class sByte_sByte : Rescale_Windowing<sbyte, sbyte>
            {
                public override void Convert(sbyte[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class sByte_Short : Rescale_Windowing<sbyte, short>
            {
                public override void Convert(sbyte[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override short Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }


            class sByte_uShort : Rescale_Windowing<sbyte, ushort>
            {
                public override void Convert(sbyte[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override ushort Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }


            class sByte_Int : Rescale_Windowing<sbyte, int>
            {
                public override void Convert(sbyte[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override int Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }


            class sByte_uInt : Rescale_Windowing<sbyte, uint>
            {
                public override void Convert(sbyte[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }


            class sByte_Int64 : Rescale_Windowing<sbyte, long>
            {
                public override void Convert(sbyte[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class sByte_uInt64 : Rescale_Windowing<sbyte, ulong>
            {
                public override void Convert(sbyte[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class sByte_Float : Rescale_Windowing<sbyte, float>
            {
                public override void Convert(sbyte[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class sByte_Double : Rescale_Windowing<sbyte, double>
            {
                public override void Convert(sbyte[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        sbyte val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(sbyte value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)(value * factor + intersection);
                }
            }


            #endregion


            #region FromShort

            class Short_sByte : Rescale_Windowing<short, sbyte>
            {
                public override void Convert(short[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(short value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class Short_Byte : Rescale_Windowing<short, byte>
            {
                public override void Convert(short[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(short value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }

            class Short_Short : Rescale_Windowing<short, short>
            {
                public override void Convert(short[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(short value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class Short_uShort : Rescale_Windowing<short, ushort>
            {
                public override void Convert(short[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(short value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class Short_Int : Rescale_Windowing<short, int>
            {
                public override void Convert(short[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(short value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class Short_uInt : Rescale_Windowing<short, uint>
            {
                public override void Convert(short[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(short value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class Short_Long : Rescale_Windowing<short, long>
            {
                public override void Convert(short[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(short value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class Short_uLong : Rescale_Windowing<short, ulong>
            {
                public override void Convert(short[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(short value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class Short_Float : Rescale_Windowing<short, float>
            {
                public override void Convert(short[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(short value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class Short_Double : Rescale_Windowing<short, double>
            {
                public override void Convert(short[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(short value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)(value * factor + intersection);
                }
            }

            class Short_Decimal : Rescale_Windowing<short, decimal>
            {
                public override void Convert(short[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        short val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = decimal.MinValue;
                        else if (val > lastFrom_Inside) to[i] = decimal.MaxValue;
                        else to[i] = (decimal)(val * factor + intersection);
                    }
                }

                public override decimal Convert(short value)
                {
                    if (value < firstFrom_Inside) return decimal.MinValue;
                    else if (value > lastFrom_Inside) return decimal.MaxValue;
                    else return (decimal)(value * factor + intersection);
                }
            }

            #endregion


            #region FromUShort

            class uShort_sByte : Rescale_Windowing<ushort, sbyte>
            {
                public override void Convert(ushort[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class uShort_Byte : Rescale_Windowing<ushort, byte>
            {
                public override void Convert(ushort[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }

            class uShort_Short : Rescale_Windowing<ushort, short>
            {
                public override void Convert(ushort[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class uShort_uShort : Rescale_Windowing<ushort, ushort>
            {
                public override void Convert(ushort[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class uShort_Int : Rescale_Windowing<ushort, int>
            {
                public override void Convert(ushort[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class uShort_uInt : Rescale_Windowing<ushort, uint>
            {
                public override void Convert(ushort[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class uShort_Long : Rescale_Windowing<ushort, long>
            {
                public override void Convert(ushort[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class uShort_uLong : Rescale_Windowing<ushort, ulong>
            {
                public override void Convert(ushort[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class uShort_Float : Rescale_Windowing<ushort, float>
            {
                public override void Convert(ushort[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class uShort_Double : Rescale_Windowing<ushort, double>
            {
                public override void Convert(ushort[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)value * factor + intersection;
                }
            }

            class uShort_Decimal : Rescale_Windowing<ushort, decimal>
            {
                public override void Convert(ushort[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ushort val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = decimal.MinValue;
                        else if (val > lastFrom_Inside) to[i] = decimal.MaxValue;
                        else to[i] = (decimal)(val * factor + intersection);
                    }
                }

                public override decimal Convert(ushort value)
                {
                    if (value < firstFrom_Inside) return decimal.MinValue;
                    else if (value > lastFrom_Inside) return decimal.MaxValue;
                    else return (decimal)(value * factor + intersection);
                }
            }

            #endregion


            #region FromInt


            class Int_Byte : Rescale_Windowing<int, byte>
            {
                public override void Convert(int[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(int value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }

            class Int_sByte : Rescale_Windowing<int, sbyte>
            {
                public override void Convert(int[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(int value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class Int_Short : Rescale_Windowing<int, short>
            {
                public override void Convert(int[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(int value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class Int_uShort : Rescale_Windowing<int, ushort>
            {
                public override void Convert(int[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(int value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class Int_Int : Rescale_Windowing<int, int>
            {
                public override void Convert(int[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(int value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class Int_uInt : Rescale_Windowing<int, uint>
            {
                public override void Convert(int[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(int value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class Int_Long : Rescale_Windowing<int, long>
            {
                public override void Convert(int[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(int value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class Int_uLong : Rescale_Windowing<int, ulong>
            {
                public override void Convert(int[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(int value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class Int_Float : Rescale_Windowing<int, float>
            {
                public override void Convert(int[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(int value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class Int_Double : Rescale_Windowing<int, double>
            {
                public override void Convert(int[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(int value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)(value * factor + intersection);
                }
            }

            class Int_Decimal : Rescale_Windowing<int, decimal>
            {
                public override void Convert(int[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        int val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = decimal.MinValue;
                        else if (val > lastFrom_Inside) to[i] = decimal.MaxValue;
                        else to[i] = (decimal)(val * factor + intersection);
                    }
                }

                public override decimal Convert(int value)
                {
                    if (value < firstFrom_Inside) return decimal.MinValue;
                    else if (value > lastFrom_Inside) return decimal.MaxValue;
                    else return (decimal)(value * factor + intersection);
                }
            }


            #endregion


            #region FromUInt

            class uInt_Byte : Rescale_Windowing<uint, byte>
            {
                public override void Convert(uint[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(uint value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }


            class uInt_sByte : Rescale_Windowing<uint, sbyte>
            {
                public override void Convert(uint[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(uint value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class uInt_Short : Rescale_Windowing<uint, short>
            {
                public override void Convert(uint[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(uint value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class uInt_uShort : Rescale_Windowing<uint, ushort>
            {
                public override void Convert(uint[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(uint value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class uInt_Int : Rescale_Windowing<uint, int>
            {
                public override void Convert(uint[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(uint value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class uInt_uInt : Rescale_Windowing<uint, uint>
            {
                public override void Convert(uint[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(uint value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class uInt_Long : Rescale_Windowing<uint, long>
            {
                public override void Convert(uint[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(uint value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class uInt_uLong : Rescale_Windowing<uint, ulong>
            {
                public override void Convert(uint[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(uint value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class uInt_Float : Rescale_Windowing<uint, float>
            {
                public override void Convert(uint[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(uint value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class uInt_Double : Rescale_Windowing<uint, double>
            {
                public override void Convert(uint[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(uint value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)(value * factor + intersection);
                }
            }

            class uInt_Decimal : Rescale_Windowing<uint, decimal>
            {
                public override void Convert(uint[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        uint val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = decimal.MinValue;
                        else if (val > lastFrom_Inside) to[i] = decimal.MaxValue;
                        else to[i] = (decimal)(val * factor + intersection);
                    }
                }

                public override decimal Convert(uint value)
                {
                    if (value < firstFrom_Inside) return decimal.MinValue;
                    else if (value > lastFrom_Inside) return decimal.MaxValue;
                    else return (decimal)(value * factor + intersection);
                }
            }

            #endregion


            #region FromLong

            class Int64_Byte : Rescale_Windowing<long, byte>
            {
                public override void Convert(long[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(long value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }

            class Int64_sByte : Rescale_Windowing<long, sbyte>
            {
                public override void Convert(long[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(long value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class Long_Short : Rescale_Windowing<long, short>
            {
                public override void Convert(long[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(long value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class Long_uShort : Rescale_Windowing<long, ushort>
            {
                public override void Convert(long[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(long value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class Long_Int : Rescale_Windowing<long, int>
            {
                public override void Convert(long[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(long value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class Long_uInt : Rescale_Windowing<long, uint>
            {
                public override void Convert(long[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(long value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class Long_Long : Rescale_Windowing<long, long>
            {
                public override void Convert(long[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(long value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class Long_uLong : Rescale_Windowing<long, ulong>
            {
                public override void Convert(long[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(long value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class Long_Float : Rescale_Windowing<long, float>
            {
                public override void Convert(long[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(long value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class Long_Double : Rescale_Windowing<long, double>
            {
                public override void Convert(long[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(long value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)(value * factor + intersection);
                }
            }

            class Long_Decimal : Rescale_Windowing<long, decimal>
            {
                public override void Convert(long[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        long val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = decimal.MinValue;
                        else if (val > lastFrom_Inside) to[i] = decimal.MaxValue;
                        else to[i] = (decimal)(val * factor + intersection);
                    }
                }

                public override decimal Convert(long value)
                {
                    if (value < firstFrom_Inside) return decimal.MinValue;
                    else if (value > lastFrom_Inside) return decimal.MaxValue;
                    else return (decimal)(value * factor + intersection);
                }
            }

            #endregion


            #region FromULong

            class uInt64_Byte : Rescale_Windowing<ulong, byte>
            {
                public override void Convert(ulong[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }

            class uInt64_sByte : Rescale_Windowing<ulong, sbyte>
            {
                public override void Convert(ulong[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class uLong_Short : Rescale_Windowing<ulong, short>
            {
                public override void Convert(ulong[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class uLong_uShort : Rescale_Windowing<ulong, ushort>
            {
                public override void Convert(ulong[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class uLong_Int : Rescale_Windowing<ulong, int>
            {
                public override void Convert(ulong[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class uLong_uInt : Rescale_Windowing<ulong, uint>
            {
                public override void Convert(ulong[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class uLong_Long : Rescale_Windowing<ulong, long>
            {
                public override void Convert(ulong[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class uLong_uLong : Rescale_Windowing<ulong, ulong>
            {
                public override void Convert(ulong[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class uLong_Float : Rescale_Windowing<ulong, float>
            {
                public override void Convert(ulong[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class uLong_Double : Rescale_Windowing<ulong, double>
            {
                public override void Convert(ulong[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)(value * factor + intersection);
                }
            }

            class uLong_Decimal : Rescale_Windowing<ulong, decimal>
            {
                public override void Convert(ulong[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        ulong val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = decimal.MinValue;
                        else if (val > lastFrom_Inside) to[i] = decimal.MaxValue;
                        else to[i] = (decimal)(val * factor + intersection);
                    }
                }

                public override decimal Convert(ulong value)
                {
                    if (value < firstFrom_Inside) return decimal.MinValue;
                    else if (value > lastFrom_Inside) return decimal.MaxValue;
                    else return (decimal)(value * factor + intersection);
                }
            }

            #endregion


            #region FromFloat

            class Float_Byte : Rescale_Windowing<float, byte>
            {
                public override void Convert(float[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(float value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }

            class Float_sByte : Rescale_Windowing<float, sbyte>
            {
                public override void Convert(float[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(float value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class Float_Short : Rescale_Windowing<float, short>
            {
                public override void Convert(float[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(float value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class Float_uShort : Rescale_Windowing<float, ushort>
            {
                public override void Convert(float[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(float value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class Float_Int : Rescale_Windowing<float, int>
            {
                public override void Convert(float[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(float value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class Float_uInt : Rescale_Windowing<float, uint>
            {
                public override void Convert(float[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(float value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class Float_Long : Rescale_Windowing<float, long>
            {
                public override void Convert(float[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(float value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class Float_uLong : Rescale_Windowing<float, ulong>
            {
                public override void Convert(float[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(float value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class Float_Float : Rescale_Windowing<float, float>
            {
                public override void Convert(float[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(float value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class Float_Double : Rescale_Windowing<float, double>
            {
                public override void Convert(float[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(float value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)(value * factor + intersection);
                }
            }

            class Float_Decimal : Rescale_Windowing<float, decimal>
            {
                public override void Convert(float[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        float val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = decimal.MinValue;
                        else if (val > lastFrom_Inside) to[i] = decimal.MaxValue;
                        else to[i] = (decimal)(val * factor + intersection);
                    }
                }

                public override decimal Convert(float value)
                {
                    if (value < firstFrom_Inside) return decimal.MinValue;
                    else if (value > lastFrom_Inside) return decimal.MaxValue;
                    else return (decimal)(value * factor + intersection);
                }
            }

            #endregion


            #region FromDouble

            class Double_Byte : Rescale_Windowing<double, byte>
            {
                public override void Convert(double[] from, int fromIndex, byte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = byte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = byte.MaxValue;
                        else to[i] = (byte)(val * factor + intersection);
                    }
                }

                public override byte Convert(double value)
                {
                    if (value < firstFrom_Inside) return byte.MinValue;
                    else if (value > lastFrom_Inside) return byte.MaxValue;
                    else return (byte)(value * factor + intersection);
                }
            }

            class Double_sByte : Rescale_Windowing<double, sbyte>
            {
                public override void Convert(double[] from, int fromIndex, sbyte[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = sbyte.MinValue;
                        else if (val > lastFrom_Inside) to[i] = sbyte.MaxValue;
                        else to[i] = (sbyte)(val * factor + intersection);
                    }
                }

                public override sbyte Convert(double value)
                {
                    if (value < firstFrom_Inside) return sbyte.MinValue;
                    else if (value > lastFrom_Inside) return sbyte.MaxValue;
                    else return (sbyte)(value * factor + intersection);
                }
            }

            class Double_Short : Rescale_Windowing<double, short>
            {
                public override void Convert(double[] from, int fromIndex, short[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = short.MinValue;
                        else if (val > lastFrom_Inside) to[i] = short.MaxValue;
                        else to[i] = (short)(val * factor + intersection);
                    }
                }

                public override short Convert(double value)
                {
                    if (value < firstFrom_Inside) return short.MinValue;
                    else if (value > lastFrom_Inside) return short.MaxValue;
                    else return (short)(value * factor + intersection);
                }
            }

            class Double_uShort : Rescale_Windowing<double, ushort>
            {
                public override void Convert(double[] from, int fromIndex, ushort[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ushort.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ushort.MaxValue;
                        else to[i] = (ushort)(val * factor + intersection);
                    }
                }

                public override ushort Convert(double value)
                {
                    if (value < firstFrom_Inside) return ushort.MinValue;
                    else if (value > lastFrom_Inside) return ushort.MaxValue;
                    else return (ushort)(value * factor + intersection);
                }
            }

            class Double_Int : Rescale_Windowing<double, int>
            {
                public override void Convert(double[] from, int fromIndex, int[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = int.MinValue;
                        else if (val > lastFrom_Inside) to[i] = int.MaxValue;
                        else to[i] = (int)(val * factor + intersection);
                    }
                }

                public override int Convert(double value)
                {
                    if (value < firstFrom_Inside) return int.MinValue;
                    else if (value > lastFrom_Inside) return int.MaxValue;
                    else return (int)(value * factor + intersection);
                }
            }

            class Double_uInt : Rescale_Windowing<double, uint>
            {
                public override void Convert(double[] from, int fromIndex, uint[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = uint.MinValue;
                        else if (val > lastFrom_Inside) to[i] = uint.MaxValue;
                        else to[i] = (uint)(val * factor + intersection);
                    }
                }

                public override uint Convert(double value)
                {
                    if (value < firstFrom_Inside) return uint.MinValue;
                    else if (value > lastFrom_Inside) return uint.MaxValue;
                    else return (uint)(value * factor + intersection);
                }
            }

            class Double_Long : Rescale_Windowing<double, long>
            {
                public override void Convert(double[] from, int fromIndex, long[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = long.MinValue;
                        else if (val > lastFrom_Inside) to[i] = long.MaxValue;
                        else to[i] = (long)(val * factor + intersection);
                    }
                }

                public override long Convert(double value)
                {
                    if (value < firstFrom_Inside) return long.MinValue;
                    else if (value > lastFrom_Inside) return long.MaxValue;
                    else return (long)(value * factor + intersection);
                }
            }

            class Double_uLong : Rescale_Windowing<double, ulong>
            {
                public override void Convert(double[] from, int fromIndex, ulong[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = ulong.MinValue;
                        else if (val > lastFrom_Inside) to[i] = ulong.MaxValue;
                        else to[i] = (ulong)(val * factor + intersection);
                    }
                }

                public override ulong Convert(double value)
                {
                    if (value < firstFrom_Inside) return ulong.MinValue;
                    else if (value > lastFrom_Inside) return ulong.MaxValue;
                    else return (ulong)(value * factor + intersection);
                }
            }

            class Double_Float : Rescale_Windowing<double, float>
            {
                public override void Convert(double[] from, int fromIndex, float[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = float.MinValue;
                        else if (val > lastFrom_Inside) to[i] = float.MaxValue;
                        else to[i] = (float)(val * factor + intersection);
                    }
                }

                public override float Convert(double value)
                {
                    if (value < firstFrom_Inside) return float.MinValue;
                    else if (value > lastFrom_Inside) return float.MaxValue;
                    else return (float)(value * factor + intersection);
                }
            }

            class Double_Double : Rescale_Windowing<double, double>
            {
                public override void Convert(double[] from, int fromIndex, double[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = double.MinValue;
                        else if (val > lastFrom_Inside) to[i] = double.MaxValue;
                        else to[i] = (double)(val * factor + intersection);
                    }
                }

                public override double Convert(double value)
                {
                    if (value < firstFrom_Inside) return double.MinValue;
                    else if (value > lastFrom_Inside) return double.MaxValue;
                    else return (double)(value * factor + intersection);
                }
            }

            class Double_Decimal : Rescale_Windowing<double, decimal>
            {
                public override void Convert(double[] from, int fromIndex, decimal[] to, int toIndex, int length)
                {
                    for (int i = toIndex; i < toIndex + length; i++)
                    {
                        double val = from[fromIndex++];
                        if (val < firstFrom_Inside) to[i] = decimal.MinValue;
                        else if (val > lastFrom_Inside) to[i] = decimal.MaxValue;
                        else to[i] = (decimal)(val * factor + intersection);
                    }
                }

                public override decimal Convert(double value)
                {
                    if (value < firstFrom_Inside) return decimal.MinValue;
                    else if (value > lastFrom_Inside) return decimal.MaxValue;
                    else return (decimal)(value * factor + intersection);
                }
            }

            #endregion


        }

        #endregion
    }
}
