/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System.Collections.Generic;

namespace TPClib
{
    /// <summary>
    /// Represents dimension object that consists of integer coodinate index values. 
    /// <remarks>low coordinate is lowest index that is included in coordinates (usually 0)
    ///  while high value is not included in coordinates (256 means that highest index is 255)</remarks>
    /// </summary>
    public class IntLimits : Limits
    {
        /// <summary>
        /// Read-only width
        /// </summary>
        public new int DimX
        {
            get
            {
                return (int)Width;
            }
        }
        /// <summary>
        /// Read-only height
        /// </summary>
        public new int DimY
        {
            get
            {
                return (int)Height;
            }
        }
        /// <summary>
        /// Read-only planes
        /// </summary>
        public new int DimZ
        {
            get
            {
                return (int)Planes;
            }
        }
        /// <summary>
        /// Read-only frames
        /// </summary>
        public new int DimT
        {
            get
            {
                return (int)Frames;
            }
        }
        /// <summary>
        /// Read-only width
        /// </summary>
        public new int Width
        {
            get
            {
                if (hi_dim.Length <= WIDTH) return 1;
                return (int)(hi_dim[WIDTH] - lo_dim[WIDTH]);
            }
        }
        /// <summary>
        /// Read-only height
        /// </summary>
        public new int Height
        {
            get
            {
                if (hi_dim.Length <= HEIGHT) return 1;
                return (int)(hi_dim[HEIGHT] - lo_dim[HEIGHT]);
            }
        }
        /// <summary>
        /// Read-only planes
        /// </summary>
        public new int Planes
        {
            get
            {
                if (hi_dim.Length <= PLANES) return 1;
                return (int)(hi_dim[PLANES] - lo_dim[PLANES]);
            }
        }
        /// <summary>
        /// Read-only frames
        /// </summary>
        public new int Frames
        {
            get
            {
                if (hi_dim.Length <= FRAMES) return 1;
                return (int)(hi_dim[FRAMES] - lo_dim[FRAMES]);
            }
        }
        /// <summary>
        /// Read-only gates
        /// </summary>
        public new int Gates
        {
            get
            {
                if (hi_dim.Length <= GATES) return 1;
                return (int)(hi_dim[GATES] - lo_dim[GATES]);
            }
        }
        /// <summary>
        /// Read-only beds
        /// </summary>
        public new int Beds
        {
            get
            {
                if (hi_dim.Length <= BEDS) return 1;
                return (int)(hi_dim[BEDS] - lo_dim[BEDS]);
            }
        }
        /// <summary>
        /// Read-only detectorvectors
        /// </summary>
        public new int DetectorVectors
        {
            get
            {
                if (hi_dim.Length <= DETECTORVECTOR) return 1;
                return (int)(hi_dim[DETECTORVECTOR] - lo_dim[DETECTORVECTOR]);
            }
        }
        /// <summary>
        /// Read-only energywindows
        /// </summary>
        public new int EnergyWindows
        {
            get
            {
                if (hi_dim.Length <= ENERGYWINDOW) return 1;
                return (int)(hi_dim[ENERGYWINDOW] - lo_dim[ENERGYWINDOW]);
            }
        }
        /// <summary>
        /// Implicit cast into long values array.
        /// </summary>
        /// <param name="dim">converted object</param>
        /// <returns>dimension sizes in array (x,y,z,..etc.)</returns>
        public static implicit operator long[](IntLimits dim)
        {
            long[] r = new long[dim.Length];
            for (int i = 0; i < dim.Length; i++)
                r[i] = (long)dim.hi_dim[i] - (long)dim.lo_dim[i];
            return r;
        }
        /// <summary>
        /// Implicit cast into double values array.
        /// </summary>
        /// <param name="dim">converted object</param>
        /// <returns>dimension sizes in array (x,y,z,..etc.)</returns>
        public static implicit operator double[](IntLimits dim)
        {
            double[] r = new double[dim.Length];
            for (int i = 0; i < dim.Length; i++)
                r[i] = (double)(dim.hi_dim[i] - dim.lo_dim[i]);
            return r;
        }
        /// <summary>
        /// Implicit cast from int values array.
        /// </summary>
        /// <param name="dim">dimension sizes in array (x,y,z,..etc.)</param>
        /// <returns>dimensions object</returns>
        public static implicit operator IntLimits(int[] dim)
        {
            return new IntLimits(dim);
        }
        /// <summary>  
        /// Constructs dimension object without dimensions.
        /// </summary> 
        public IntLimits()
            : base()
        {
        }
		/// <summary>
		/// Copyconstructor
		/// </summary>
		/// <param name="il">IntLimits to copy</param>
		public IntLimits(IntLimits il)
		{
			lo_dim = new double[il.Length];
			hi_dim = new double[il.Length];

			System.Array.Copy(il.lo_dim, this.lo_dim, il.Length);
			System.Array.Copy(il.hi_dim, this.hi_dim, il.Length);
		}

		/// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(long[] dimensions)
            : base(dimensions)
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(uint[] dimensions)
            : base(dimensions)
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimx">x-dimension</param>
        /// <exception cref="TPCInvalidArgumentsException">dimension is not above zero</exception>
        public IntLimits(int dimx)
			: this(new int[] { dimx })
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimx">x-dimension</param>
        /// <param name="dimy">y-dimension</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(int dimx, int dimy)
			: this(new int[] { dimx, dimy })
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimx">x-dimension</param>
        /// <param name="dimy">y-dimension</param>
        /// <param name="dimz">z-dimension</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(int dimx, int dimy, int dimz)
			: this(new int[] { dimx, dimy, dimz })
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimx">x-dimension</param>
        /// <param name="dimy">y-dimension</param>
        /// <param name="dimz">z-dimension</param>
        /// <param name="dimt">t-dimension (time points)</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(int dimx, int dimy, int dimz, int dimt)
            : this(new int[] { dimx, dimy, dimz, dimt })
        {
        }
        /// <summary>  
        /// Constructs dimension from value array.
        /// </summary> 
        /// <param name="dimensions">dimension values</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
		public IntLimits(int[] dimensions) : this(new int[dimensions.Length], dimensions) { }

		/// <summary>  
        /// Constructs dimension from value array. Low limits are set to zero.
        /// </summary> 
        /// <param name="lo_limits">low limits</param>
        /// <param name="hi_limits">high limits</param>
        /// <exception cref="TPCInvalidArgumentsException">dimensions are not above zero</exception>
        public IntLimits(int[] lo_limits, int[] hi_limits)
        {
			lo_dim = new double[lo_limits.Length];
			hi_dim = new double[lo_limits.Length];
			
			for (int i = 0; i < lo_limits.Length; i++)
            {
                if (lo_limits[i] < 0) lo_dim[i] = 0;
				else lo_dim[i] = lo_limits[i];

				if (hi_limits[i] - lo_limits[i] < 1) hi_dim[i] = 1;
				else hi_dim[i] = hi_limits[i];
            }
        }
        /// <summary>
        /// Returns desired dimension value
        /// </summary>
        /// <param name="i">dimension number (0-based)</param>
        /// <returns>dimension as double value. </returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public new int GetDimension(int i)
        {
			if (i >= this.Length) return 1;
            return (int)base.GetDimension(i);
        }
        
		/// <summary>
        /// Returns all dimensions as new limits object (all low limits to zero).
        /// </summary>
        /// <returns>dimension as double value. All dimensions greater than number of actual defined dimensions are regarded as 1.</returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public new IntLimits GetDimensions()
        {
            IntLimits r = new IntLimits();
            for (int i = 0; i < Length; i++)
                r.AddLimit(0, GetDimension(i));
            return r;
        }
        
		/// <summary>
        /// Returns desired limit value.
        /// </summary>
        /// <param name="i">dimension number [1..number of dimensions]</param>
        /// <param name="limit">limit</param>
        /// <returns>limit as double value.</returns>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public new int GetLimit(int i, Limit limit)
        {
			if (i < this.Length) return (int)base.GetLimit(i, limit);
			else return (limit == Limit.HIGH) ? 1 : 0;
        }

        /// <summary>
        /// Sets desired limit value.
        /// </summary>
        /// <param name="i">dimension number</param>
        /// <param name="limit">limit</param>
        /// <param name="value">value that is set</param>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public void SetLimit(int i, Limit limit, int value)
        {
            base.SetLimit(i, limit, value);
        }
        /// <summary>
        /// Sets desired limit values.
        /// </summary>
        /// <param name="i">dimension number</param>
        /// <param name="low">low limit</param>
        /// <param name="high">high limit</param>
        /// <exception cref="TPCInvalidArgumentsException">i is below zero</exception>
        public void SetLimits(int i, int low, int high)
        {
            base.SetLimits(i, low, high);
        }
        /// <summary>
        /// Calculates n first dimension values multiplied together.
        /// </summary>
        /// <param name="dimensions">last dimension that is multiplied [0..No dimensions-1]</param>
        /// <returns>product of dimensions</returns>
        public new double GetProduct(int dimensions)
        {
            return (int)base.GetProduct(dimensions);
        }
        /// <summary>
        /// Calculates dimension values multiplied together.
        /// </summary>
        /// <returns>product of dimensions</returns>
        public new int GetProduct()
        {
            return (int)base.GetProduct();
        }

		/// <summary>
		/// Compare limits
		/// </summary>
		/// <param name="obj">Object to compare with</param>
		/// <returns>True, if limit values are equal</returns>
		public override bool Equals(object obj)
		{
			if (!(obj is Limits)) return false;

			Limits lim = obj as Limits;

			int i;
			double aSize, bSize, aLow, bLow;

			// Compare dimension lengths and lower limits.
			// Assume limits [0,1] for dimensions not listed.
			for (i = 0; i < this.Length || i < lim.Length; i++)
			{
				aLow = 0;
				bLow = 0;
				aSize = 1;
				bSize = 1;

				if (lim.Length > i)
				{
					aLow = lim.GetLimit(i, Limit.LOW);
					aSize = lim.GetDimension(i);
				}
				if (this.Length > i)
				{
					bLow = lo_dim[i];
					bSize = this.GetDimension(i);
				}

				if (aLow != bLow || aSize != bSize) return false;
			}

			return true;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override int GetHashCode()
		{
			return this.GetProduct().GetHashCode();
		}

		/// <summary>
		/// Inclusion in the limits of this object.
		/// </summary>
		/// <param name="sublim">Limits to check for inclusion</param>
		/// <returns>True, if the sub limits are completely included</returns>
		public bool Includes(IntLimits sublim)
		{
			int i;
			double subSize, fullSize, subLow, fullLow;

			// Compare dimension lengths and lower limits.
			// Assume limits [0,1] for dimensions not listed.
			for (i = 0; i < this.Length || i < sublim.Length; i++)
			{
				subLow = 0;
				fullLow = 0;
				subSize = 1;
				fullSize = 1;

				if (sublim.Length > i)
				{
					subLow = sublim.GetLimit(i, Limit.LOW);
					subSize = sublim.GetDimension(i);
				}
				if (this.Length > i)
				{
					fullLow = lo_dim[i];
					fullSize = this.GetDimension(i);
				}

				if (subLow < fullLow || (subLow + subSize) > (fullLow + fullSize)) return false;
			}

			return true;
		}

		/// <summary>
        /// Converts limits to dimension array. Array has difference between 
        /// low and high values in is indexes.
        /// </summary>
        /// <returns>array if indexes in limits object</returns>
        new public int[] ToArray()
        {
            int[] r = new int[lo_dim.Length];
            for (int i = 0; i < r.Length; i++)
            {
                r[i] = (int)(hi_dim[i] - lo_dim[i]);
            }
            return r;
        }
        
		/// <summary>
        /// Gest dimension size as integer.
        /// </summary>
        /// <param name="i">0-based index of dimension number</param>
        /// <returns>size of dimension at index</returns>
        public int this[int i]
        {
			get { return this.GetDimension(i); }
        }

		/// <summary>
        /// Removes singleton dimensions (length == 1) from dimension object.
        /// Note that 0-length dimensions are not removed, since they should not exist.
        /// </summary>
        public IntLimits RemoveSingletons()
        {
            List<double> lo_dim_tmp = new List<double>();
            List<double> hi_dim_tmp = new List<double>();
            for (int i = 0; i < lo_dim.Length; i++)
            {
                if ((int)(hi_dim[i] - lo_dim[i]) != 1)
                {
                    lo_dim_tmp.Add((double)lo_dim[i]);
                    hi_dim_tmp.Add((double)hi_dim[i]);
                }
            }
            lo_dim = lo_dim_tmp.ToArray();
            hi_dim = hi_dim_tmp.ToArray();
            return new IntLimits(this);
        }
        
		/// <summary>
        /// Resolves limits for subregion.
        /// </summary>
        /// <see cref="IntLimits"/>
        /// <param name="index">index (0-based)</param>
        /// <param name="dimension">dimension number</param>
        /// <returns>limits object that represent index in dimension</returns>
        /// <exception cref="TPCInvalidArgumentsException">in case of invalid parameters</exception>
        public IntLimits GetDimensionLimits(int index, int dimension)
        {
            if (dimension < 0) throw new TPCInvalidArgumentsException("Dimension argument is negative");
            if (dimension > this.Length)
                throw new TPCInvalidArgumentsException("Dimension argument " + dimension + " is larger than image dimensions " + this.Length);
            if (index < 0 || index >= this.GetDimension(dimension))
                throw new TPCInvalidArgumentsException("Index number " + index + " is out of bounds [0.." + (this.GetDimension(dimension) - 1) + "]");
            IntLimits r = new IntLimits();
            int i = 0;
            while (r.Length < dimension)
                r.AddLimit(0, this.GetDimension(i++));
            r.AddLimit(index, index + 1);
            while (r.Length < this.Length)
                r.AddLimit(0, 1);
            return r;
        }
    }
}
