/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Image;

namespace TPClib
{
    /// <summary>
    /// Class representing isotope. Only predefined values can be used.
    /// </summary>
    public sealed class Isotope
    {
        /// <summary>
        /// Structure containing isotope values
        /// </summary>
        private struct Isotope_values {
            /// <summary>
            /// Isotope name.
            /// </summary>
            public string name;
            /// <summary>
            /// Isotope halflife in seconds.
            /// </summary>
            public double halflife;
            /// <summary>
            /// Isotope enumerator value.
            /// </summary>
            public Isotope_enumerator isotopeCode;
            /// <summary>
            /// Contrustor 
            /// </summary>
            /// <param name="isotopeCode">isotope</param>
            /// <param name="halflife">halflife in minutes</param>
            /// <param name="name">isotope name as string</param>
            public Isotope_values(Isotope_enumerator isotopeCode, double halflife, string name)
            {
                this.halflife = halflife*60.0;
                this.isotopeCode = isotopeCode;
                this.name = name;
            }
        }
        
		/// <summary>
        /// List of available isotopes.
        /// </summary>
        private static Isotope_values[] isotopes = new Isotope_values[] { 
            new Isotope_values(Isotope_enumerator.C11, 20.38,"Carbon-11"),
            new Isotope_values(Isotope_enumerator.O14, 1.17676666667, "Oxygen-14"),
            new Isotope_values(Isotope_enumerator.O15, 2.03666666667, "Oxygen-15"),
            new Isotope_values(Isotope_enumerator.F18, 109.77, "Fluorine-18"),
            new Isotope_values(Isotope_enumerator.Ge68, 396000.0, "Germanium-68"),
            new Isotope_values(Isotope_enumerator.Ga68, 67.629, "Gallium-68"),
            new Isotope_values(Isotope_enumerator.Br75, 96.7, "Bromine-75"),
            new Isotope_values(Isotope_enumerator.Br76, 978.33, "Bromine-76"),
            new Isotope_values(Isotope_enumerator.Cu62, 9.74, "Copper-62"),
            new Isotope_values(Isotope_enumerator.Cu64, 768.0, "Copper-64"),
            new Isotope_values(Isotope_enumerator.Fe52, 496.5, "Iron-52"),
            new Isotope_values(Isotope_enumerator.N13, 9.965, "Nitrogen-13"),
            new Isotope_values(Isotope_enumerator.Na22, 1368000.0, "Natrium-22"),
            new Isotope_values(Isotope_enumerator.Br82, 2118.0, "Bromine-82"),
            new Isotope_values(Isotope_enumerator.Zn62, 551.16, "Zinc-62"),
            new Isotope_values(Isotope_enumerator.I124, 6013.44, "Iodine-124"),
            new Isotope_values(Isotope_enumerator.Rb82, 1.25, "Rubidium-82"),
            new Isotope_values(Isotope_enumerator.No_isotope, 0, "No isotope")
        };

        /// <summary>
        /// Returns isotope halflife in seconds.
        /// </summary>
        /// <param name="isotope">isotope</param>
        /// <returns>halflife in seconds</returns>
        public static double GetHalflifeInMinutes(Isotope_enumerator isotope)
        {
            return GetHalflife(isotope) / 60.0;
        }
        
		/// <summary>
        /// Returns isotope halflife in seconds.
        /// </summary>
        /// <param name="isotope">isotope</param>
        /// <returns>halflife in seconds</returns>
        public static double GetHalflife(Isotope_enumerator isotope) {
            foreach(Isotope_values i in isotopes) {
                if (i.isotopeCode == isotope) {
                    return i.halflife;
                }
            }
            throw new Exception("Internal error: all isotope code values are not defined");
        }
        
		/// <summary>
        /// Constructs string from unit. 
        /// </summary>
        /// <param name="isotope">isotope name, for example 'Bq/ml'</param>
        public static string ConvertToString(Isotope_enumerator isotope)
        {
            string str = isotope.ToString();
            char[] char_str = str.ToCharArray();
            //search for first digit character
            int i = 0;
            for (; i < char_str.Length; i++) {
                if (Char.IsDigit(char_str[i])) break;
            }
            //insert '-' between letters and digits
            if(i < char_str.Length)
                str.Insert(i, "-");
            return str;
        }
        
		/// <summary>
        /// Creates isotope from isotope name. 'No_isotope' is used if 
        /// no proper match was found.
        /// </summary>
        /// <param name="name">isotope name, for example 'Oxygen-15'</param>
        public static Isotope_enumerator CreateIsotope(string name)
        {
            name = name.ToUpperInvariant();
            //convert from synonym
            if (name.Contains("11C") || name.Contains("C-11") || name.Contains("C11")) return Isotope_enumerator.C11;
            else if (name.Contains("14O") || name.Contains("O-14") || name.Contains("O14")) return Isotope_enumerator.O14;
            else if (name.Contains("15O") || name.Contains("O-15") || name.Contains("O15")) return Isotope_enumerator.O15;
            else if (name.Contains("18F") || name.Contains("F-18") || name.Contains("F18")) return Isotope_enumerator.F18;
            else if (name.Contains("68GE") || name.Contains("GE-68") || name.Contains("GE68")) return Isotope_enumerator.Ge68;
            else if (name.Contains("68GA") || name.Contains("GA-68") || name.Contains("Ga68")) return Isotope_enumerator.Ga68;
            else if (name.Contains("75BR") || name.Contains("BR-75") || name.Contains("BR75")) return Isotope_enumerator.Br75;
            else if (name.Contains("BR76") || name.Contains("BR-76") || name.Contains("BR76")) return Isotope_enumerator.Br76;
            else if (name.Contains("62CU") || name.Contains("CU-62") || name.Contains("CU62")) return Isotope_enumerator.Cu62;
            else if (name.Contains("52FE") || name.Contains("FE-52") || name.Contains("FE52")) return Isotope_enumerator.Fe52;
            else if (name.Contains("13N") || name.Contains("N-13") || name.Contains("N^13")) return Isotope_enumerator.N13;
            else if (name.Contains("22NA") || name.Contains("NA-22") || name.Contains("NA22")) return Isotope_enumerator.Na22;
            else if (name.Contains("82BR") || name.Contains("BR-82") || name.Contains("BR82")) return Isotope_enumerator.Br82;
            else if (name.Contains("62ZN") || name.Contains("ZN-62") || name.Contains("ZN62")) return Isotope_enumerator.Zn62;
			else if (name.Contains("82RB") || name.Contains("RB-82") || name.Contains("RB82")) return Isotope_enumerator.Rb82;
			else if (name.Contains("24I") || name.Contains("I-24") || name.Contains("I24")) return Isotope_enumerator.I124;
            return Isotope_enumerator.No_isotope;
        }

        /// <summary>
        /// Returns names of all available isotopes.
        /// </summary>
        /// <returns>array having unit names</returns>
        public static string[] GetAllIsotopenames()
        {
            string[] r = new string[isotopes.Length];
            for (int i = 0; i < isotopes.Length; i++)
                r[i] = isotopes[i].name;
            return r;
        }
        
		/// <summary>
        /// Returns lambda value for this isotope. Halflife is used in 
        /// seconds in calculation.
        /// </summary>
        /// <returns>lambda</returns>
        public double GetLambda(Isotope_enumerator isotope)
        {
            foreach (Isotope_values i in isotopes)
            {
                if (i.isotopeCode == isotope)
                {
                    return (Math.Log(2.0, Math.E) / i.halflife);
                }
            }
            throw new Exception("Internal error: all isotope code values are not defined");
        }
    }
}
