/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Modeling
{
	/// <summary>
	/// Modeling attribute for metrics.
	/// </summary>
	public class MetricAttribute : ModelingAttribute { }

	/// <summary>
	/// Maximum difference metric. Calculates the maximum difference in two sets of values.
	/// </summary>
	[Metric(Name = "Maximum", Description = "Maximum absolute deviation")]
	public class MaximumMetric : IFitMetric
	{
		private MaximumMetric() { }

		/// <summary>
		/// Maximum metric.
		/// </summary>
		public static MaximumMetric Maximum = new MaximumMetric();

		/// <summary>
		/// Calculate the metric for two value sets.
		/// </summary>
		/// <param name="a">First value set</param>
		/// <param name="b">Second value set</param>
		/// <returns>Metric value for the sets</returns>
		public double Calculate(double[] a, double[] b)
		{
			double max = Double.MinValue;
			double dif;
			for (int i = 0; i < a.Length; i++)
			{
				dif = Math.Abs(a[i] - b[i]);
				if (max < dif) max = dif;
			}
			return max;
		}
	}

	/// <summary>
	/// Manhattan distance.
	/// </summary>
	[Metric(Name = "Manhattan", Description = "Manhattan distance")]
	public class ManhattanMetric : IFitMetric
	{
		private ManhattanMetric() { }

		/// <summary>
		/// Manhattan distance metric.
		/// </summary>
		public static ManhattanMetric Manhattan = new ManhattanMetric();

		/// <summary>
		/// Calculate the metric for two value sets.
		/// </summary>
		/// <param name="a">First value set</param>
		/// <param name="b">Second value set</param>
		/// <returns>Metric value for the sets</returns>
		public double Calculate(double[] a, double[] b)
		{
			double sum = 0.0;
			for (int i = 0; i < a.Length; i++)
			{
				sum += Math.Abs(a[i] - b[i]);
			}
			return sum;
		}
	}

	/// <summary>
	/// Euclidean metric.
	/// </summary>
	[Metric(Name = "Euclid", Description = "Euclidean distance")]
	public class EuclideanMetric : IFitMetric
	{
		private EuclideanMetric() { }

		/// <summary>
		/// Euclidean metric.
		/// </summary>
		public static EuclideanMetric Euclidean = new EuclideanMetric();

		/// <summary>
		/// Calculate the metric for two value sets.
		/// </summary>
		/// <param name="a">First value set</param>
		/// <param name="b">Second value set</param>
		/// <returns>Metric value for the sets</returns>
		public double Calculate(double[] a, double[] b)
		{
			double sum = 0.0;
			for (int i = 0; i < a.Length; i++)
			{
				sum += a[i] * b[i];
			}
			return Math.Sqrt(sum);
		}
	}

	/// <summary>
	/// Sum of squares.
	/// </summary>
	[Metric(Name = "Sum of Squares", Description = "Sum of Squares")]
	public class SumOfSquaresMetric : IFitMetric
	{
		private SumOfSquaresMetric() { }

		/// <summary>
		/// Sum of squares.
		/// </summary>
		public static SumOfSquaresMetric SumOfSquares = new SumOfSquaresMetric();

		/// <summary>
		/// Calculate the metric for two value sets.
		/// </summary>
		/// <param name="a">First value set</param>
		/// <param name="b">Second value set</param>
		/// <returns>Metric value for the sets</returns>
		public double Calculate(double[] a, double[] b)
		{
			double dif, dif2;
			double sum = 0.0;
			for (int i = 0; i < a.Length; i++)
			{
				dif = a[i] - b[i];
				dif2 = dif * dif;
				sum += dif2;
			}

			return sum;
		}
	}

	/// <summary>
	/// Weighted sum of squares.
	/// </summary>
	[Metric(Name = "Weighted Sum of Squares", Description = "Weighted Sum of Squares")]
	public class WeightedSumOfSquaresMetric : IFitMetric
	{
		/// <summary>
		/// Weights.
		/// </summary>
		[Input(Name=@"Weights")]
		public double[] weights;

		/// <summary>
		/// Calculate the metric for two value sets.
		/// </summary>
		/// <param name="a">First value set</param>
		/// <param name="b">Second value set</param>
		/// <returns>Metric value for the sets</returns>
		public double Calculate(double[] a, double[] b)
		{
			double dif, dif2;
			double sum = 0.0;
			for (int i = 0; i < a.Length; i++)
			{
				dif = a[i] - b[i];
				dif2 = dif * dif * weights[i];
				sum += dif2;
			}

			return sum;
		}
	}
}
