/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Linear regression fit
	/// </summary>
	[FitProcess(Name = @"Linear regression", Description = @"Linear regression fit")]
	public class LinearRegressionFit : SingleFitProcess
	{
		/// <summary>
		/// Model to fit
		/// </summary>
		[Input(Name = @"Model", Optional = true)]
		public StraightLine Model = new StraightLine();

		/// <summary>
		/// 
		/// </summary>
		[Output(Name = @"Slope standard deviation")]
		public double SlopeSD = 0.0f;

		/// <summary>
		/// 
		/// </summary>
		[Output(Name = @"Intercept standard deviation")]
		public double InterceptSD = 0.0f;

		/// <summary>
		/// 
		/// </summary>
		[Output(Name = @"Pearson's correlation coefficient")]
		public double Pearson = 1.0f;

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			Model.ReferencePoints = this.MeasuredData.ReferenceValues;
			Model.Init();
		}

		/// <summary>
		/// Fit model parameters to input data
		/// </summary>
		/// <returns>Fitted parameters</returns>
		public override ParameterList Run()
		{
			// Fixed intercept
			if (Model.Intercept.Hidden && !Model.Slope.Hidden)
			{
				Vector x = new Vector(MeasuredData.ReferenceValues.ToArray());
				Vector y = new Vector((double[])MeasuredData) - Model.Intercept;
				this.Model.Slope = (x * y) / (x * x);
			}
			// Fixed slope
			else if (Model.Slope.Hidden && !Model.Intercept.Hidden)
			{
				double xMean = 0.0;
				double yMean = 0.0;
				for (int i = 0; i < MeasuredData.Length;i++ )
				{
					yMean += MeasuredData[i];
					xMean += MeasuredData.ReferenceValues[i];
				}

				yMean /= MeasuredData.Length;
				xMean /= MeasuredData.Length;
				Model.Intercept = yMean - Model.Slope * xMean;
			}
			// No fixed parameters
			else
			{
				double[] res = LinearRegression.Pearson(MeasuredData.ReferenceValues.ToArray(), MeasuredData);
				Model.Slope = res[0];
				this.SlopeSD = (double)res[1];
				Model.Intercept = res[2];
				this.InterceptSD = (double)res[3];
				this.Pearson = (double)res[4];
			}

			return Model.GetParameters();
		}
	}

	/// <summary>
	/// Iterative method for linear least-squares fit with errors in both
	/// coordinates. This method is fully based on article [3].
	/// 
	/// For tacs-point pairs (x, y) each point has its own weighting factors
	/// in (wx, wy). This routine finds the values of the parameters m (slope)
	/// and c (intercept, ic) that yield the "best-fit" of the model equation
	/// Y = mX + c to the tacs, where X and Y are the predicted or calculated values
	/// of the tacs points.
	/// 
	/// Weighting factors wx and wy must be assigned as the inverses of the variances
	/// or squares of the measurement uncertainties (SDs), i.e. w=1/(sd)^2
	/// 
	/// If true weights are unknown but yet the relative weights are correct,
	/// the slope, intercept and residuals (WSS) will be correct.
	/// The applied term S/(N-2) makes also the estimate of sigma (sd) of slope
	/// less dependent on the scaling of weights. The sigmas are not exact, since
	/// only the lowest-order terms in Taylor-series expansion are incorporated;
	/// anyhow sigmas are more accurate than the ones based on York algorithm.
	/// 
	/// One or more tacs points can be excluded from the fit by setting either x or y
	/// weight to 0.
	/// 
	/// References:
	/// 1. York, D. Linear-squares fitting of a straight line.
	///    Can. J. Phys. 1966;44:1079-1086.
	/// 2. Lybanon, M. A better least squares method when both variables have
	///    uncertainties. Am. J. Phys. 1984;52:22-26 and 276-278.
	/// 3. Reed BC. Linear least-squares fits with errors in both coordinates. II:
	///    Comments on parameter variances. Am. J. Phys. 1992;60:59-62.
	/// </summary>
	[FitProcess(Name = @"Weighted linear regression", Description = @"Linear regression fit with weights")]
	public class WeightedLinearRegressionFit : SingleFitProcess
	{
		/// <summary>
		/// Model to fit
		/// </summary>
		[Input(Name = @"Model", Optional = true)]
		public StraightLine Model = new StraightLine();

		/// <summary>
		/// X-axis weights for data
		/// </summary>
		[Input(Name = @"X-axis weights for data", Optional = true)]
		public double[] XWeights = null;

		/// <summary>
		/// Y-axis weights for data
		/// </summary>
		[Input(Name = @"Y-axis weights for data", Optional = true)]
		public double[] YWeights = null;

		/// <summary>
		/// Tolerance
		/// </summary>
		[Input(Name = @"Tolerance", Optional = true)]
		public double Tolerance = 1.0e-5;

		/// <summary>
		/// Expected slope standard deviation
		/// </summary>
		[Output(Name = @"Expected slope standard deviation")]
		public double SlopeSD = 0.0;

		/// <summary>
		/// Expected intercept standard deviation
		/// </summary>
		[Output(Name = @"Expected intercept standard deviation")]
		public double InterceptSD = 0.0;

		/// <summary>
		/// Effective weights
		/// </summary>
		[Output(Name = @"Effective weights")]
		public double[] EffectiveWeights;

		/// <summary>
		/// Sqrt(WSS)/wsum of the residuals
		/// </summary>
		[Output(Name = @"Sqrt(WSS)/wsum of the residuals")]
		public double Residual = 0.0;

		/// <summary>
		/// Estimated points
		/// </summary>
		[Output(Name = @"Estimated points")]
		public Plot TacEstimate;

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			Model.ReferencePoints = MeasuredData.ReferenceValues;
			Model.Init();
			if (XWeights == null)
			{
				XWeights = new double[MeasuredData.Length];
				for (int i = 0; i < XWeights.Length; i++) XWeights[i] = 1.0f;
			}
			if (YWeights == null)
			{
				YWeights = new double[MeasuredData.Length];
				for (int i = 0; i < YWeights.Length; i++) YWeights[i] = 1.0f;
			}
		}

		/// <summary>
		/// Fit model parameters to input data
		/// </summary>
		/// <returns>Fitted parameters</returns>
		public override ParameterList Run()
		{
			double[] cx, cy;
			double slopeEstimate, interceptEstimate;
			LinearLeastSquares.Calculate(MeasuredData.ReferenceValues.ToArray(), MeasuredData.Values.ToArray(), YWeights, XWeights, Tolerance, out EffectiveWeights, out interceptEstimate, out slopeEstimate, out Residual, out InterceptSD, out SlopeSD, out cx, out cy);
			TacEstimate = new Plot((DataSeries)cy, (DataSeries)cx);
			Model.Intercept = interceptEstimate;
			Model.Slope = slopeEstimate;
			return new ParameterList(interceptEstimate, slopeEstimate);
		}
	}
}
