/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// O-15 labeled H20 for myocardial perfusion study
	///    k1
	/// Ca----C1
	///    k2
	/// </summary>
	[Model(Name = @"CM1T", Description = @"O-15 labeled H20 for myocardial perfusion study")]
	public class CM1T : SingleInputCompartmentModel
	{
		/// <summary>
		/// 
		/// </summary>
		[Parameter(Name = @"K1")]
		public OptimizationParameter K1 = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// 
		/// </summary>
		[Parameter(Name = @"K2")]
		public OptimizationParameter K2 = new OptimizationParameter(0.5, 0, 5);

		/// <summary>
		/// 
		/// </summary>
		[Output(Name = @"C1")]
		public TimeDataSeries C1;

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			base.Init();
			C1 = new TimeDataSeries(new double[Samples], Ca.Times);
		}

		/// <summary>
		/// Simulate data with current parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			double iC1 = 0;

			double iC1_last = 0;
			double iInput = 0;

			double C1_last = 0;

			double dt2 = 0;

			for (int i = 0; i < Samples; i++)
			{
				dt2 = HalfDeltaTimes[i];
				iInput = IntegratedInput[i];

				C1[i] = (K1 * iInput - K2 * (iC1_last + dt2 * C1_last)) / (1 + dt2 * K2);
				iC1 = iC1_last + dt2 * (C1_last + C1[i]);
				C1_last = C1[i];
				iC1_last = iC1;
			}
			CPet = C1;
			return C1;
		}
	}

	/// <summary>
	/// General 1-tissue compartmental model
	///    k1    
	/// Ca----C1
	///    k2    
	/// Cpet=Va*Ca+(1-Va)*(C1)
	/// </summary>
	[Model(Name = @"General 1-tissue compartmental model", Description = @"General 1-tissue compartmental model")]
	public class General1TCModel : CM1T
	{
		/// <summary>
		/// 
		/// </summary>
		[Parameter(Name = @"VA")]
		public OptimizationParameter Va = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// Initialization.
		/// </summary>
		public override void Init()
		{
			base.Init();
			CPet = new TimeDataSeries(new double[Samples], Ca.Times);
		}

		/// <summary>
		/// Simulate data with current parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			double[] baseSimulation = base.Simulate();

			for (int i = 0; i < baseSimulation.Length; i++)
			{
				CPet[i] = (1 - Va) * baseSimulation[i] + Va * Ca[i];
			}
			return CPet;
		}
	}
}
