/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// By Chunlei Han. May 10, 2009
	/// 3-tissue compartmental model in parallel
	///    k1     k3     
	/// Ca----C1-----C2
	///    k2  |  k4
	///        |
	///     k5 | k6
	///        |
	///        |
	///       C3
	///          
	/// reference: http://www.turkupetcentre.net/reports/tpcmod0001.pdf
	/// </summary>
	[Model(Name = @"CM3T parallel", Description = @"3-tissue compartmental model in parallel")]
	public class CM3TParallel : SingleInputCompartmentModel
	{
		/// <summary>
		/// K1
		/// </summary>
		[Parameter(Name = @"K1")]
		public OptimizationParameter K1 = new OptimizationParameter(0.5, 0, 10);

		/// <summary>
		/// K2
		/// </summary>
		[Parameter(Name = @"K2")]
		public OptimizationParameter K2 = new OptimizationParameter(0.5, 0, 3);

		/// <summary>
		/// K3
		/// </summary>
		[Parameter(Name = @"K3")]
		public OptimizationParameter K3 = new OptimizationParameter(0.5, 0, 3);

		/// <summary>
		/// K4
		/// </summary>
		[Parameter(Name = @"K4")]
		public OptimizationParameter K4 = new OptimizationParameter(0.5, 0, 3);

		/// <summary>
		/// K5
		/// </summary>
		[Parameter(Name = @"K5")]
		public OptimizationParameter K5 = new OptimizationParameter(0.5, 0, 3);

		/// <summary>
		/// K6
		/// </summary>
		[Parameter(Name = @"K6")]
		public OptimizationParameter K6 = new OptimizationParameter(0.5, 0, 3);

		/// <summary>
		/// C1
		/// </summary>
		[Output(Name = @"C1")]
		public TimeDataSeries C1;

		/// <summary>
		/// C2
		/// </summary>
		[Output(Name = @"C2")]
		public TimeDataSeries C2;

		/// <summary>
		/// C3
		/// </summary>
		[Output(Name = @"C3")]
		public TimeDataSeries C3;

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			base.Init();
			C1 = new TimeDataSeries(new double[Samples], Ca.Times);
			C2 = new TimeDataSeries(new double[Samples], Ca.Times);
			C3 = new TimeDataSeries(new double[Samples], Ca.Times);
		}

		/// <summary>
		/// Simulate data with current parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			//Cumulative integral values;
			double iC1, iC2, iC3;

			// Other temp variables
			double iC1_last = 0;
			double iC2_last = 0;
			double iC3_last = 0;
			double C1_last = 0;
			double C2_last = 0;
			double C3_last = 0;
			double iInput;
			double dt2 = 0;
			double ca, c1f, c1s, c2f, c2s, c3f, c3s, c1base;

			for (int i = 0; i < Samples; i++)
			{
				dt2 = HalfDeltaTimes[i];
				iInput = IntegratedInput[i];

				//calculate C1;
				//Ca term in upper part 
				ca = K1 * iInput;
				//C1 term in upper part 
				c1f = K2 + K3 / (1 + dt2 * K4) + K5 / (1 + dt2 * K6);
				c1s = iC1_last + dt2 * C1_last;
				//C2 term
				c2f = K4 / (1 + dt2 * K4);
				c2s = iC2_last + dt2 * C2_last;
				//C3 term
				c3f = K6 / (1 + dt2 * K6);
				c3s = iC3_last + dt2 * C3_last;

				c1base = 1 + dt2 * c1f;

				//new c1
				C1[i] = (ca - c1f * c1s + c2f * c2s + c3f * c3s) / c1base;
				iC1 = iC1_last + dt2 * (C1_last + C1[i]);

				//calculate c2
				C2[i] = (K3 * iC1 - K4 * c2s) / (1 + dt2 * K4);
				iC2 = iC2_last + dt2 * (C2_last + C2[i]);

				C3[i] = (K5 * iC1 - K6 * c3s) / (1 + dt2 * K6);
				iC3 = iC3_last + dt2 * (C3_last + C3[i]);

				//non-negative constraint
				if (C1[i] < 0) C1[i] = 0;
				if (C2[i] < 0) C2[i] = 0;
				if (C3[i] < 0) C3[i] = 0;

				CPet[i] = C1[i] + C2[i] + C3[i];

				C1_last = C1[i];
				C2_last = C2[i];
				C3_last = C3[i];

				iC1_last = iC1;
				iC2_last = iC2;
				iC3_last = iC3;

			}
			return CPet;
		}
	}

	/// <summary>
	/// General 3-tissue compartmental model parallel
	///    k1    k3
	/// Ca----C1----C2
	///    k2  | k4
	///        |
	///     k5 | k6
	///        |
	///       C3
	/// Cpet=Va*Ca+(1-Va)*(C1+C2+C3)
	/// </summary>
	[Model(Name = @"General 3-tissue parallel compartmental model", Description = @"General 3-tissue parallel compartmental model")]
	public class General3TCModelParallel : CM3TParallel
	{
		/// <summary>
		/// VA
		/// </summary>
		[Parameter(Name = @"VA")]
		public OptimizationParameter Va = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// Simulate data with current parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			base.Simulate();

			for (int i = 0; i < CPet.Length; i++)
			{
				CPet[i] = (1 - Va) * CPet[i] + Va * Ca[i];
			}
			return CPet;
		}
	}
}
