/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// By Chunlei Han. Oct 28, 2009
	/// 3-tissue compartmental model in parallel
	///    K1     k3    k5
	/// Ca----C1-----C2-----C3
	///    k2  ^  k4    k6
	///          
	/// This is a general compartment model, it can be used for lower compartment
	/// model by setting to zero and hiding the corresponding parameters.
	/// Example:
	/// 2-tissue compartment model: K5=0, K6=0; K5.Hidden = true, K6.Hidden = true;
	/// 
	/// reference: http://www.turkupetcentre.net/reports/tpcmod0001.pdf
	/// </summary>
	[Model(Name = @"CM3T series", Description = @"3-tissue compartmental model in series")]
	public class CM3TSeries : SingleInputCompartmentModel
	{
		/// <summary>
		/// K1
		/// </summary>
		[Parameter(Name = @"K1")]
		public OptimizationParameter K1 = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// K2
		/// </summary>
		[Parameter(Name = @"K2")]
		public OptimizationParameter K2 = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// K3
		/// </summary>
		[Parameter(Name = @"K3")]
		public OptimizationParameter K3 = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// K4
		/// </summary>
		[Parameter(Name = @"K4")]
		public OptimizationParameter K4 = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// K5
		/// </summary>
		[Parameter(Name = @"K5")]
		public OptimizationParameter K5 = new OptimizationParameter(0, 0, 1);

		/// <summary>
		/// K6
		/// </summary>
		[Parameter(Name = @"K6")]
		public OptimizationParameter K6 = new OptimizationParameter(0, 0, 1);

		/// <summary>
		/// C1
		/// </summary>
		[Output(Name = @"C1")]
		public TimeDataSeries C1;

		/// <summary>
		/// C2
		/// </summary>
		[Output(Name = @"C2")]
		public TimeDataSeries C2;

		/// <summary>
		/// C3
		/// </summary>
		[Output(Name = @"C3")]
		public TimeDataSeries C3;

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			base.Init();
			C1 = new TimeDataSeries(new double[Samples], Ca.Times);
			C2 = new TimeDataSeries(new double[Samples], Ca.Times);
			C3 = new TimeDataSeries(new double[Samples], Ca.Times);
		}

		/// <summary>
		/// Simulate data with current parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			//Cumulative integral values;
			double iC1 = 0;
			double iC2 = 0;
			double iC3 = 0;
			double iC1_last = 0;
			double iC2_last = 0;
			double iC3_last = 0;
			double iInput = 0;

			double c1_last = 0;
			double c2_last = 0;
			double c3_last = 0;

			double dt2 = 0;

			double c1f, c1s, c2f, c2s, element_a, element_b, element_c;

			for (int i = 0; i < Samples; i++)
			{
				dt2 = HalfDeltaTimes[i];
				iInput = IntegratedInput[i];

				//element
				element_a = K5 * K6 * dt2 / (1 + dt2 * K6);
				element_b = K4 * K6 * dt2 / (1 + dt2 * K6);
				element_c = K4 + K5 - element_a;

				//C1
				c1f =	(K1 * (1 + dt2 * element_c * iInput) +
						(K3 * K4 * dt2 - (K2 + K3) * (1 + dt2 * element_c)) * (iC1_last + dt2 * c1_last) +
						K4 * (iC2_last + dt2 * c2_last) +
						element_b * (iC3_last + dt2 * c3_last));

				c1s =	(1 + dt2 * element_c) * (1 + dt2 * (K2 + K3)) -
						K3 * K4 * (dt2 * dt2);

				C1[i] = c1f / c1s;

				iC1 = iC1_last + dt2 * (c1_last + C1[i]);

				//C2
				c2f =	(K3 * iInput -
						element_a * (iC2_last + dt2 * c2_last) +
						(K6 / (1 + dt2 * K6)) *	(element_c * (iC3_last + dt2 * c3_last)));

				c2s = 1 + dt2 * element_c;
				C2[i] = c2f / c2s;
				iC2 = iC2_last + dt2 * (c2_last + C2[i]);

				// C3
				C3[i] = (K5 * dt2 * iC2 + -K6 * (iC3_last + dt2 * c3_last)) / (1 + dt2 * K6);
				iC3 = iC3_last + dt2 * (c3_last + C3[i]);

				iC1_last = iC1;
				iC2_last = iC2;
				iC3_last = iC3;

				//non-negative constraint
				if (C1[i] < 0) C1[i] = 0;
				if (C2[i] < 0) C2[i] = 0;
				if (C3[i] < 0) C3[i] = 0;

				CPet[i] = C1[i] + C2[i] + C3[i];
				c1_last = C1[i];
				c2_last = C2[i];
				c3_last = C3[i];
			}

			return CPet;
		}
	}

	/// <summary>
	/// General 3-tissue compartmental model series
	///    k1    k3    k5
	/// Ca----C1----C2----C3
	///    k2    k4    k6
	/// Cpet=Va*Ca+(1-Va)*(C1+C2)
	/// </summary>
	[Model(Name = @"General 3-tissue compartmental model series", Description = @"General 3-tissue compartmental model series")]
	public class General3TCModelSeries : CM3TSeries
	{
		/// <summary>
		/// VA
		/// </summary>
		[Parameter(Name = @"VA")]
		public OptimizationParameter Va = new OptimizationParameter(0.5, 0, 1);

		/// <summary>
		/// Simulate data with the current parameters
		/// </summary>
		/// <returns>Simulated data</returns>
		public override DataSeries Simulate()
		{
			for (int i = 0; i < CPet.Length; i++)
			{
				CPet[i] = (1 - Va) * CPet[i] + Va * Ca[i];
			}
			return CPet;
		}
	}
}
