/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Model Kmono. Linear fit to a logarithmic scale.
	/// </summary>
	[Model(Name = @"Kmono model", Description = @"Linearized logarithmic data model", Category = ModelAttribute.ClassCategory.TEST)]
	public class Kmono : BaseModel
	{
		/// <summary>
		/// Reference times
		/// </summary>
		[Input(Name = @"Reference values", Optional = false)]
		public ISeries ReferencePoints;

		/// <summary>
		/// Linear slope
		/// </summary>
		[Parameter(Name = @"Slope", Description = @"Kmono (1/min)")]
		public OptimizationParameter Slope;

		/// <summary>
		/// Linear intercept
		/// </summary>
		[Parameter(Name = @"Intercept", Description = @"Amplitude (log scale)")]
		public OptimizationParameter Intercept;

		/// <summary>
		/// Result plot
		/// </summary>
		[Output(Name = @"Result")]
		public Plot Result;

		/// <summary>
		/// Number of samples simulated
		/// </summary>
		public int Samples
		{
			get { return ReferencePoints.Count; }
		}

		/// <summary>
		/// Initialize
		/// </summary>
		public override void Init()
		{
			Result = new Plot(new DataSeries(Samples), ReferencePoints);
		}

		/// <summary>
		/// Simulate data using the current model parameters
		/// </summary>
		/// <returns></returns>
		public override DataSeries Simulate()
		{
			for (int i = 0; i < Samples; i++)
				Result[i] = Math.Exp(Intercept - ReferencePoints[i] * Slope);

			return Result;
		}
	}
}
