/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
    /// <summary>
    /// RTCM (Referencetimes Tissue Compartment Model) model
    /// </summary>
	[Model(Name = @"RTCM", Description = @"Reference Tissue Compartment Model")]
	public class RTCMModel : SingleInputCompartmentModel
	{
		/// <summary>
		/// R1 = K1/K1'
		/// </summary>
		[Parameter(Name = @"R1", Description = @"Ratio K1/K1'", Optional = false)]
		public OptimizationParameter R1;

		/// <summary>
		/// K2
		/// </summary>
		[Parameter(Name = @"K2", Description = @"", Optional = false)]
		public OptimizationParameter K2;

		/// <summary>
		/// K3
		/// </summary>
		[Parameter(Name = @"K3", Description = @"", Optional = false)]
		public OptimizationParameter K3;

		/// <summary>
		/// Bp = K3 / K4
		/// </summary>
		[Parameter(Name = @"BP", Description = @"", Optional = false)]
		public OptimizationParameter Bp;

		/// <summary>
		/// Cf
		/// </summary>
		[Output(Name = @"CF", Description = @"First tissue compartment")]
		public TimeDataSeries Cf;

		/// <summary>
		/// Cb
		/// </summary>
		[Output(Name = @"CB", Description = @"Second tissue compartment")]
		public TimeDataSeries Cb;

		/// <summary>
		/// Initialization.
		/// </summary>
		public override void Init()
		{
			base.Init();
			Cb = new TimeDataSeries(new double[Samples], Ca.Times);
			Cf = new TimeDataSeries(new double[Samples], Ca.Times);
		}

		/// <summary>
		/// Simulates tissue TAC using reference_times tissue compartment model (original) and reference_times region TAC, at reference_times region TAC tacs.
		/// The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
		/// </summary>
		/// <returns>Vector of simulated values</returns>
		public override DataSeries Simulate()
		{
			double k4 = K3 / Bp;

			//Cumulative integral values
			double cbi = 0.0;
			double cfi = 0.0;
			double cbiLast = 0.0;
			double cfiLast = 0.0;

			double cbLast = 0.0;
			double cfLast = 0.0;
			double cbCurrent, cfCurrent;

			// Variables for intermediate values
			double f, b, w, dt2, cr, cri;

			for (int i = 0; i < Samples; i++)
			{
				dt2 = HalfDeltaTimes[i];
				cr = Ca[i];
				cri = IntegratedInput[i];

				w = K2 + K3 + K2 * k4 * dt2;
				f = cfiLast + dt2 * cfLast;
				b = cbiLast + dt2 * cbLast;

				cfCurrent = ((1.0 + k4 * dt2) * (R1 * cr + K2 * cri) + k4 * b - w * f) / (1.0 + dt2 * (w + k4));
				cfi = cfiLast + dt2 * (cfLast + cfCurrent);

				cbCurrent = (K3 * cfi - k4 * b) / (1.0 + k4 * dt2);
				cbi = cbiLast + dt2 * (cbLast + cbCurrent);

				// Save the last values
				cbLast = cbCurrent;
				cfLast = cfCurrent;
				cfiLast = cfi;
				cbiLast = cbi;

				// Save the results in the output arrays
				Cb[i] = cbCurrent;
				Cf[i] = cfCurrent;
				CPet[i] = cfCurrent + cbCurrent;
			}
			return CPet;
		}
	}
}
