/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.Modeling
{
    /// <summary>
    /// SRTM (Simplified Reference Tissue Model) model.
	/// reference: Lammertsma AA, Hume SP. Simplified reference tissue model for PET receptor studies. NeuroImage 1996; 4:153-15
	/// </summary>
	[Model(Name = @"SRTM", Description = @"Simplified Reference Tissue Model")]
	public class SRTMModel : SingleInputCompartmentModel
    {
		/// <summary>
		/// R1
		/// </summary>
		[Parameter(Name=@"R1", Description=@"")]
		public OptimizationParameter R1;

		/// <summary>
		/// Bp
		/// </summary>
		[Parameter(Name=@"BP", Description=@"")]
		public OptimizationParameter Bp;

		/// <summary>
		/// K2
		/// </summary>
		[Parameter(Name=@"K2", Description=@"")]
		public OptimizationParameter K2;

        /// <summary>
        /// Simulates tissue TAC using reference_times tissue compartment model (simplified) and reference_times region TAC, at reference_times region TAC tacs.
        /// The units of rate constants must be related to the time unit; 1/min and min, or 1/sec and sec.
        /// </summary>
        /// <returns>Vector of simulated values</returns>
		public override DataSeries Simulate()
        {
            double k2bp = K2 / (1.0 + Bp);

            //Cumulative integral value
            double cti = 0.0;

            double last_val = 0.0;
            double cr, cri, dt2, ci;

            for (int i = 0; i < Samples; i++)
            {
				cr = Ca[i];
                cri = IntegratedInput[i];
                dt2 = HalfDeltaTimes[i];

                ci = ((R1 * cr) + (K2 * cri) - k2bp * (dt2 * last_val + cti)) /
                            (1.0 + dt2 * k2bp);

				cti += dt2 * (last_val + ci);
				last_val = ci;
				CPet[i] = ci;
            }
			return CPet;
        }
	}
}
