/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using TPClib.Common;

namespace TPClib.Modeling
{
	/// <summary>
	/// Model parameter
	/// </summary>
	public struct OptimizationParameter : IComparable, IFormattable
	{
		private double value;

		private double min;

		private double max;

		private bool hidden;

		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="d">Value</param>
		/// <param name="minVal">Minimum value</param>
		/// <param name="maxVal">Maximum value</param>
		public OptimizationParameter(double d = 0.0, double minVal = Double.NegativeInfinity, double maxVal = Double.PositiveInfinity)
		{
			if (d < minVal || d > maxVal) throw new TPCException("Parameter value is outside it's limits");
			value = d;
			hidden = false;
			min = minVal;
			max = maxVal;
		}

		/// <summary>
		/// Copy constructor
		/// </summary>
		/// <param name="p">Parameter to copy</param>
		public OptimizationParameter(OptimizationParameter p)
		{
			this.value = p.value;
			this.hidden = p.Hidden;
			this.max = p.max;
			this.min = p.min;
		}

		/// <summary>
		/// Get lower bound
		/// </summary>
		public double Minimum { get { return min; } }

		/// <summary>
		/// Get upper bound
		/// </summary>
		public double Maximum { get { return max; } }

		/// <summary>
		/// Parameter value.
		/// </summary>
		public double Value { get { return value; } set { this.value = value; } }

		/// <summary>
		/// Hidden parameter
		/// </summary>
		public bool Hidden { get { return hidden; } set { hidden = value; } }

		/// <summary>
		/// Check whether the current value is inside the parameter limits
		/// </summary>
		public bool InsideBounds { get { return (value <= max && value >= min); } }

		/// <summary>
		/// String representation of this parameter
		/// </summary>
		/// <returns>String representation</returns>
		public override string ToString()
		{
			return value.ToString();
		}

		/// <summary>
		/// String representation of this parameter
		/// </summary>
		/// <param name="s">Format string</param>
		/// <param name="ifp">IFormatProvider</param>
		/// <returns>String representation</returns>
		public string ToString(string s, IFormatProvider ifp)
		{
			return value.ToString(s, ifp);
		}

		/// <summary>
		/// Compare to other object
		/// </summary>
		/// <param name="o">Object to compare</param>
		/// <returns>0 if equal, 1 if this instance is larger, -1 if smaller</returns>
		public int CompareTo(object o)
		{
			return value.CompareTo(o);
		}

		/// <summary>
		/// Get type code for this class
		/// </summary>
		/// <returns>Type code</returns>
		public TypeCode GetTypeCode()
		{
			return TypeCode.Object;
		}

		/// <summary>
		/// Equality comparison
		/// </summary>
		/// <param name="obj">Object to compare with</param>
		/// <returns>True, if object is equal to this instance</returns>
		public override bool Equals(object obj)
		{
			return value.Equals(obj);
		}

		/// <summary>
		/// Get a hash code for this instance
		/// </summary>
		/// <returns>Hashcode</returns>
		public override int GetHashCode()
		{
			return value.GetHashCode();
		}

		/// <summary>
		/// Conversion operator from double
		/// </summary>
		/// <param name="d">Value to convert</param>
		/// <returns>Parameter with the argument value</returns>
		public static implicit operator OptimizationParameter(double d)
		{
			return new OptimizationParameter(d);
		}

		/// <summary>
		/// Conversion operator to double
		/// </summary>
		/// <param name="p">Parameter to convert</param>
		/// <returns>Parameter value in double</returns>
		public static implicit operator double(OptimizationParameter p)
		{
			return p.value;
		}

		/// <summary>
		/// Conversion operator to double
		/// </summary>
		/// <param name="p">Parameter to convert</param>
		/// <returns>Parameter value in double</returns>
		public static implicit operator float(OptimizationParameter p)
		{
			return (float)p.value;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="p"></param>
		/// <returns></returns>
		public static explicit operator int(OptimizationParameter p)
		{
			return (int)(p.value);
		}

		/// <summary>
		/// Multiplication operator
		/// </summary>
		/// <param name="p">Parameter</param>
		/// <param name="d">Double</param>
		/// <returns>Parameter with value equal to the multiplication result</returns>
		public static OptimizationParameter operator *(OptimizationParameter p, double d)
		{
			OptimizationParameter pp = new OptimizationParameter(p);
			pp.value = pp.value * d;
			return pp;
		}
	}

	/// <summary>
	/// List of parameters.
	/// </summary>
	public class ParameterList : IList<OptimizationParameter>
	{
		private IList<OptimizationParameter> parameters;

		/// <summary>
		/// Default constructor. Creates an empty list.
		/// </summary>
		/// <param name="n">Length of the list</param>
		public ParameterList(int n = 0)
		{
			parameters = new OptimizationParameter[n];
		}

		/// <summary>
		/// Constructor. Creates a new list from a set of parameters.
		/// </summary>
		/// <param name="p">Set of parameters</param>
		public ParameterList(params OptimizationParameter[] p)
		{
			OptimizationParameter[] pArr = new OptimizationParameter[p.Length];
			Array.Copy(p, pArr, p.Length);
			p.Clone();
			parameters = pArr;
		}

		#region IList interface

		/// <summary>
		/// Index of an item in this list.
		/// </summary>
		/// <param name="item">Item</param>
		/// <returns>Index of the item in this list</returns>
		public int IndexOf(OptimizationParameter item)
		{
			return parameters.IndexOf(item);
		}

		/// <summary>
		/// Insert a new item at a specified index.
		/// </summary>
		/// <param name="index">Index to add the new item.</param>
		/// <param name="item">The new item</param>
		public void Insert(int index, OptimizationParameter item)
		{
			parameters.Insert(index, item);
		}

		/// <summary>
		/// Remove the item at a specified index.
		/// </summary>
		/// <param name="index">Index</param>
		public void RemoveAt(int index)
		{
			parameters.RemoveAt(index);
		}

		/// <summary>
		/// Get the parameter at an index.
		/// </summary>
		/// <param name="index">Index</param>
		/// <returns>Item at the index</returns>
		public OptimizationParameter this[int index]
		{
			get
			{
				return parameters[index];
			}
			set
			{
				parameters[index] = value;
			}
		}

		/// <summary>
		/// Add a new parameter at the end of this list.
		/// </summary>
		/// <param name="item">Item to add</param>
		public void Add(OptimizationParameter item)
		{
			parameters.Add(item);
		}

		/// <summary>
		/// Clear all parameters from this list.
		/// </summary>
		public void Clear()
		{
			parameters.Clear();
		}

		/// <summary>
		/// Check, if this list contains a parameter.
		/// </summary>
		/// <param name="item">Parameter to look for</param>
		/// <returns>True, if the parameter was found</returns>
		public bool Contains(OptimizationParameter item)
		{
			return parameters.Contains(item);
		}

		/// <summary>
		/// Copy the contents of this list to an array.
		/// </summary>
		/// <param name="array">Destination array</param>
		/// <param name="arrayIndex">Starting index</param>
		public void CopyTo(OptimizationParameter[] array, int arrayIndex)
		{
			parameters.CopyTo(array, arrayIndex);
		}

		/// <summary>
		/// Number of items in this list.
		/// </summary>
		public int Count
		{
			get { return parameters.Count; }
		}

		/// <summary>
		/// Read only list.
		/// </summary>
		public bool IsReadOnly
		{
			get { return parameters.IsReadOnly; }
		}

		/// <summary>
		/// Remove an item from this list.
		/// </summary>
		/// <param name="item">Item to remove</param>
		/// <returns>True, if the item was found and removed</returns>
		public bool Remove(OptimizationParameter item)
		{
			return parameters.Remove(item);
		}

		/// <summary>
		/// Get a list enumerator.
		/// </summary>
		/// <returns>Enumerator for this list</returns>
		public IEnumerator<OptimizationParameter> GetEnumerator()
		{
			return parameters.GetEnumerator();
		}

		/// <summary>
		/// Get a list enumerator.
		/// </summary>
		/// <returns>Enumerator for this list</returns>
		System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
		{
			return parameters.GetEnumerator();
		}

		#endregion

		#region Conversion operators

		/// <summary>
		/// Conversion to float array.
		/// </summary>
		/// <param name="pList">Parameter list</param>
		/// <returns>List contents in an array</returns>
		public static implicit operator float[](ParameterList pList)
		{
			float[] fArr = new float[pList.Count];
			for (int i = 0; i < fArr.Length; i++)
			{
				fArr[i] = pList[i];
			}
			return fArr;
		}

		/// <summary>
		/// Conversion from a float array.
		/// </summary>
		/// <param name="fArr">Float array</param>
		/// <returns>Parameter list</returns>
		public static implicit operator ParameterList(float[] fArr)
		{
			ParameterList pList = new ParameterList(fArr.Length);
			for (int i = 0; i < fArr.Length; i++)
			{
				pList[i] = new OptimizationParameter(fArr[i]);
			}
			return pList;
		}

		/// <summary>
		/// Conversion to a double array.
		/// </summary>
		/// <param name="pList">Parameter list</param>
		/// <returns>Double array of list values</returns>
		public static implicit operator double[](ParameterList pList)
		{
			double[] dArr = new double[pList.Count];
			for (int i = 0; i < dArr.Length; i++)
			{
				dArr[i] = pList[i];
			}
			return dArr;
		}

		/// <summary>
		/// Conversion from a double array.
		/// </summary>
		/// <param name="dArr">Double array</param>
		/// <returns>Parameter list</returns>
		public static implicit operator ParameterList(double[] dArr)
		{
			ParameterList pList = new ParameterList(dArr.Length);
			for (int i = 0; i < dArr.Length; i++)
			{
				pList[i] = new OptimizationParameter(dArr[i]);
			}
			return pList;
		}

		/// <summary>
		/// Conversion to a vector.
		/// </summary>
		/// <param name="pList">Parameter list</param>
		/// <returns>Vector containing the list values</returns>
		public static implicit operator Vector(ParameterList pList)
		{
			return (double[])pList;
		}

		/// <summary>
		/// Conversion from a vector.
		/// </summary>
		/// <param name="v">Vector</param>
		/// <returns>Parameter list</returns>
		public static implicit operator ParameterList(Vector v)
		{
			return (double[])v;
		}

		/// <summary>
		/// Conversion to an array of parameters.
		/// </summary>
		/// <param name="pList">Parameter list</param>
		/// <returns>Array of parameters</returns>
		public static implicit operator OptimizationParameter[](ParameterList pList)
		{
			OptimizationParameter[] pArr = new OptimizationParameter[pList.Count];
			pList.CopyTo(pArr, 0);
			return pArr;
		}

		/// <summary>
		/// Conversion from an array of parameters
		/// </summary>
		/// <param name="pList">Array of parameters</param>
		/// <returns>List of parameters</returns>
		public static implicit operator ParameterList(OptimizationParameter[] pList)
		{
			return new ParameterList(pList);
		}

		#endregion
	}
}
