/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using TPClib.Common;

namespace DraftLib.Modeling
{
    /// <summary>
    /// The algoritm is based on:
    /// Numerical Recipes, Third Edition, 2007
    /// 15.2 Fitting Data to a Straight Line (page 780)
    /// </summary>
    public static class LinearRegression
    {
        /// <summary>
        /// Estimates parameterNames for interpolation.
        /// y = k x + b
        /// </summary>
        /// <param name="x">tacs x-coodrinates</param>
        /// <param name="y">tacs y-coodrinates</param>
        /// <returns>parameterNames {k,b}</returns>
		public static double[] ApproximateParameters(double[] x, double[] y)
        {
            if (x.Length != y.Length) throw new TPCException("x and y length must be the same");
            int ndata = x.Length;

            int i;
			double sx = 0.0f, sy = 0.0f, st2 = 0.0f, t, sxoss;
			double k = 0;
            for (i = 0; i < ndata; i++)
            {
                sx += x[i];
                sy += y[i];
            }
            sxoss = sx / ndata;
            for (i = 0; i < ndata; i++)
            {
                t = x[i] - sxoss;
                st2 += t * t;
                k += t * y[i];
            }
            k = k / st2;
			double b = (sy - sx * k) / ndata;
			return new double[] { k, b };
        }

        /// <summary>
        /// Interpolates at tacs points.
        /// </summary>
        /// <param name="x">tacs points</param>
        /// <param name="p">parameterNames {k,a}</param>
        /// <returns>y-values evaluated at locations x</returns>
		public static double[] Interpolate(double[] x, double[] p)
        {
            if (p.Length != 2) throw new TPCException("parameterNames for line must be {k,a}");
			double[] y = new double[x.Length];
            for (int i = 0; i < x.Length; i++) {
                y[i] = p[0] * x[i] + p[1];
            }
            return y;
        }
        /// <summary>
        /// Calculate slope and intercept of a line and Pearson's correlation coefficient.
        /// {slope, slope deviation, intercept, intercept deviation, Pearson's correlation coefficient, Root Mean Squared Error}
        /// </summary>
        /// <param name="x">tacs x-coordinates</param>
        /// <param name="y">tacs y-coordinates</param>
        /// <returns>parameterNames {b_0, b_0SD, b_1, b_1SD, r, MSE}</returns>
		public static double[] Pearson(double[] x, double[] y)
        {
            //sums of x and y
			double sum_x = 0.0f, sum_y = 0.0f;
            //sums of multiplies
			double sum_xx = 0.0f, sum_xy = 0.0f, sum_yy = 0.0f;
            //vairances of x and y, and covariance
            //these are used without division operation since it is unnecessary for 
            //linear regression
			double sumsdx = 0.0f, sumsdy = 0.0f, sumdxdy = 0.0f;
            //temporary variables, and the mean of x and the mean of y
			double f, g, meanx, meany;
            //Sum of Squared Error
			double SSE = 0.0f;
            //slope and intercept
			double b_0, b_1;
            //Pearson's correlation coefficient
			double r;
            //Standard errors for slope and intercept
			double b_0_SD, b_1_SD;
            //Root Mean Squared Error
			double RMSE;

            //Check that there is some tacs
            if (x == null || y == null)
                throw new TPCException("x- and/or y-values was null");
            if (x.Length == 0)
                throw new TPCException("x-values length was zero");
            if (x.Length != y.Length)
                throw new TPCException("x- and/or y-values length must be the same");
            //handle special case when there are only one point
            if(x.Length < 1)
				return new double[] { Single.NaN, Single.NaN, 0, 0, 1, 0 };
            //handle special case when there are only two points
            if (x.Length == 2) {
                f = x[1] - x[0];
                if (Math.Abs(f) < 1.0E-50) throw new TPCException("Error using 2 datapoints");
                b_0 = (y[1] - y[0]) / f;
                b_1 = y[0] - b_0 * x[0];
				return new double[] { b_0, 0, b_1, 0, 1, 0 };
            }
            //Calculate (x,y) sums and means
            for (int i = 0; i < x.Length; i++)
            {
                sum_x += x[i];
                sum_y += y[i];
                sum_xx += x[i] * x[i];
                sum_yy += y[i] * y[i];
                sum_xy += x[i] * y[i];
            }
			meanx = sum_x / (double)x.Length;
			meany = sum_y / (double)x.Length;
            //and then based on means
            for (int i = 0; i < x.Length; i++)
            {
                f = x[i] - meanx;
                g = y[i] - meany;
                sumsdx += f * f;
                sumsdy += g * g;
                sumdxdy += f * g;
            }
            //return analytical solutions
            if (sumsdx < 1.0e-50)
				return new double[] { double.PositiveInfinity, 0.0f, double.NaN, double.NaN, 1.0f, 0.0f };
            if (sumsdy < 1.0e-50)
				return new double[] { 0.0f, 0.0f, meany, 0.0f, 1.0f, 0.0f };
            //Regression coefficient (slope of line)
            b_1 = sumdxdy / sumsdx;
            //Intercept with y axis
            b_0 = meany-b_1*meanx;
            //Errors
            for (int i = 0; i < x.Length; i++)
            {
                f = b_1 * x[i] + b_0 - y[i];
                SSE += f * f;
            }
            //Calculate Mean Squared Error
            if (SSE <= 1.0e-12) RMSE = 0.0f;
			else RMSE = (double)Math.Sqrt(SSE / (x.Length - 2));
            //SD of slope and intercept
			b_1_SD = RMSE / (double)Math.Sqrt(sumsdx);
			b_0_SD = RMSE * (double)Math.Sqrt(1.0 / x.Length + (meanx * meanx) / sumsdx);
            //Pearson's correlation coefficient
			r = sumdxdy / (double)(Math.Sqrt(sumsdx) * Math.Sqrt(sumsdy));
            return new double[] { b_1, b_1_SD, b_0, b_0_SD, r, RMSE };

        } //pearson

        /// <summary>
        /// Calculate slope and intercept of a line and Pearson's correlation coefficient.
        /// Data points may contain NA's.
        /// {slope, slope deviation, intercept, intercept deviation, Pearson's correlation coefficient, deviation along y}
        /// </summary>
        /// <param name="x">tacs x-coordinates</param>
        /// <param name="y">tacs y-coordinates</param>
        /// <returns>parameterNames {k, kSD, b, bSD, r, ySD}</returns>
		public static double[] Pearson3(double[] x, double[] y)
        {
			List<double> new_x = new List<double>();
			List<double> new_y = new List<double>();
            
            //Copy tacs to new arrays
            int j = 0;
            for (int i = 0; i < x.Length; i++)
            {
				if (x[i] != double.MinValue && y[i] != double.MinValue)
				{
                    new_x.Add(x[i]);
                    new_y.Add(y[i]);
                    j++;
                }
            }
			double[] nx = new_x.ToArray();
			double[] ny = new_y.ToArray();
            //Use pearson()
            return Pearson(nx, ny);

        }
    }
}

