/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

namespace TPClib.Modeling.Integrals
{
    /// <summary>
    /// Linear interpolation dataFunctions
    /// </summary>
    public class LinearInterpolation : InterpolationMethod
    {
		/// <summary>
		/// Interpolates at tacs points.
		/// </summary>
		/// <param name="x">interpolation locations</param>
		/// <param name="data_x">x-values of tacs</param>
		/// <param name="data_y">y-values of tacs</param>
		/// <param name="m">Extrapolation method</param>
		/// <returns>y-values evaluated at locations x</returns>
        public override double[] Interpolate(double[] x, double[] data_x, double[] data_y, ExternalValueHandlingMethod m)
        {
			// Y-values evaluated at locations x
            double[] result = new double[x.Length];

			// slopes between two tacs value points
            double[] slopes = new double[data_x.Length - 1];

            // Counting slopes 
            double x1;
            double y1;
            double x2;
            double y2;
            for (int k = 0; k < data_x.Length - 2; k++)
            {
                x1 = data_x[k];
                y1 = data_y[k];
                x2 = data_x[k + 1];
                y2 = data_y[k + 1];
                slopes[k] = (y2 - y1) / (x2 - x1);
            }
            int i, j;
            
            
            switch (m)
            {
                case ExternalValueHandlingMethod.EXTRAPOLATION:
                    // evaluation of y-values
                    i = 0;
                    j = 0;
                    // x-values start before data_x-values
                    while (x[i] <= data_x[j])
                    {
                        result[i] = slopes[j] * (x[i] - data_x[j]) + data_y[j];
                        i++;
                    }
                    j++;
                    // x-values are between data_x[0] - data_x[data_x.Length]
                    while (i < x.Length - 1 && j < data_x.Length)
                    {
                        if (x[i] <= data_x[j])
                        {
                            result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                            i++;
                        }
                        else
                        {
                            while (x[i] > data_x[j] && j < data_x.Length) j++;

                            result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                            i++;

                        }
                    }
                    // data_x-values end before x-values end
                    while (i < x.Length)
                    {
                        result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                        i++;
                    }    
                break;                
                case ExternalValueHandlingMethod.ZERO_VALUES:
                    // evaluation of y-values
                    i = 0;
                    j = 0;
                    // Turning result values, which x-coordinates are located before data_x[0], to zero.
                    while (x[i] <= data_x[j])
                    {
                        result[i] = 0;
                        i++;
                    }
                    j++;
                    // x-values are between data_x[0] - data_x[data_x.Length]
                    while (i < x.Length - 1 && j < data_x.Length)
                    {
                        if (x[i] <= data_x[j])
                        {
                            result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                            i++;
                        }
                    else
                        {
                        while (x[i] > data_x[j] && j < data_x.Length) j++;

                        result[i] = slopes[j-1] * (x[i] - data_x[j]) + data_y[j];
                        i++;

                        }
                    }
                    // Turning result values, which x-coordinates are located after data_x[data_x.Length], to zero.
                    while (i < x.Length)
                    {
                        result[i] = 0;
                        i++;
                    }  
                break;
                case ExternalValueHandlingMethod.PATLAK:
                // evaluation of y-values
                i = 0;
                j = 0;
                // Turning result values, which x-coordinates are located before data_x[0], to zero.
                if (data_x[1] - data_x[0] < data_x[0] - x[0])
                {
                    while (x[i] <= data_x[j])
                    {
                        result[i] = 0;
                        i++;
                    }
                }
                j++;
                // x-values are between data_x[0] - data_x[data_x.Length]
                while (i < x.Length - 1 && j < data_x.Length)
                {
                    if (x[i] <= data_x[j])
                    {
                        result[i] = slopes[j - 1] * (x[i] - data_x[j]) + data_y[j];
                        i++;
                    }
                    else
                    {
                        while (x[i] > data_x[j] && j < data_x.Length) j++;

                        result[i] = slopes[j - 1] * (x[i] - data_x[j]) + data_y[j];
                        i++;

                    }
                }
                // data_x-values end before x-values end
                while (i < x.Length)
                {
                    result[i] = slopes[j - 1] * (x[i] - data_x[j]) + data_y[j];
                    i++;
                }  
                break;
                default: break;
            } 
            return result;
        }
        
        /// <summary>
        /// Linear interpolation and integration.
        /// </summary>
        /// <remarks>
        /// Both original and new interpolated tacs represent the actual values
        /// at specified time points, not from framed tacs.
        /// Then output can be integrated dot-by-dot.
        /// PRE: 
        /// Original tacs and new x values must be sorted by ascending x.
        /// Subsequent x values can have equal values, enabling the use of step dataFunctions.
        /// Negative x (time) values can be processed.
        /// The output tacs must've been allocated.
        /// POST:
        /// If output y is null, this doesn't calculate the first and second integral values.
        /// If necessary, the tacs is extrapolated assuming that: 
        ///     1) y[infinity]=y[y.Length-1] and 
        ///     2) if x[0] greater than 0, y[0] greater than 0 and x[0] is less or equal to x[1]-x[0], 
        ///        an imaginary line is drawn from (0,0) to (x[0],y[0]). 
        /// </remarks>
        /// <param name="x">input x-values</param>
        /// <param name="y">input y-values</param>
        /// <param name="newx">interpolation x-locations</param>
        /// <param name="newy">interpolated y-values; null can be given</param>
        /// <param name="int1st">Integral; null can be given</param>
        /// <param name="int2nd">Second Integral; null can be given</param>
        public static void Interpolate(
			double[] x, double[] y,
			double[] newx, out double[] newy,
			out double[] int1st, out double[] int2nd)
        {
			newy = new double[newx.Length];
			int1st = new double[newx.Length];
			int2nd = new double[newx.Length];

            // Check for tacs
            if (newx.Length != newy.Length)
                throw new TPCException("the x and y input tacs sizes don't match.");

            if (x.Length != y.Length)
                throw new TPCException("the x and y input tacs sizes don't match.");

            if (newy == null && int1st == null && int2nd == null)
                throw new TPCException("The output tacs must've been allocated.");

			double ty, tyi, tyii, ly, li, lii, lil;

            // Set output values before the input tacs has started
            // If newx is also negative, set values to zero
            int j;
            for (j = 0; j < newx.Length; j++)
            {
                if (newx[j] >= x[0] || newx[j] >= 0.0)
                    break;

                ty = tyi = tyii = 0.0f;

                if (newy != null)
                    newy[j] = ty;
                if (int1st != null)
                    int1st[j] = tyi;
                if (int2nd != null)
                    int2nd[j] = tyii;
            }
            // Now newx[j]>=x[0] or newx[j]>=0 
            // If newx is still smaller than x[0],
            // but positive, then interpolate
            // between (0,0) and (x[0], y[0]) 
            if (newx[j] >= 0.0)
            {
                for (; j < newx.Length; j++)
                {
                    if (newx[j] > x[0]) break;

                    // Second condition was added 14.6.
                    if (x[0] == 0.0 || (x.Length > 1 && x[0] > x[1] - x[0]))
                    {
                        ty = tyi = tyii = 0.0f;
                        if (newx[j] == x[0])
                            ty = y[0];
                    }
                    else
                    {
                        ty = newx[j] * y[0] / x[0];
                        tyi = 0.5f * ty * newx[j];
                        tyii = 0.5f * tyi * newx[j];
                    }

                    if (newy != null)
                        newy[j] = ty;
                    if (int1st != null)
                        int1st[j] = tyi;
                    if (int2nd != null)
                        int2nd[j] = tyii;
                }
            }

            if (j == newx.Length)
            {
                // Normal Return
                return;
            }

            // Now newx[j]>=x[0]
            if (newx[j] < x[0])
                throw new TPCException("3");

            // Calculate input tacs values at x[0]
            ly = y[0];

            if (x[0] <= 0.0 || x[0] > x[1] - x[0])
            {
                li = lii = lil = 0.0f;
            }
            else
            {
                lil = li = 0.5f * x[0] * ly;
                lii = 0.5f * x[0] * li;
            }

            // Calculate input tacs values at x[i]
            int i;
            for (i = 1; i < x.Length && j < newx.Length; i++)
            {
                // calculate output values between x[i-1] and x[i]
                for (; j < newx.Length; j++)
                {
                    if (newx[j] > x[i])
                        break;

                    ty = ((y[i] - y[i - 1]) / (x[i] - x[i - 1]))
                        * (newx[j] - x[i - 1]) + y[i - 1];

                    if (newy != null)
                        newy[j] = ty;

                    tyi = li + 0.5f * (ty + y[i - 1]) * (newx[j] - x[i - 1]);

                    if (int1st != null)
                        int1st[j] = tyi;
                    if (int2nd != null)
                        int2nd[j] = tyii = lii + 0.5f * (tyi + li) * (newx[j]
                            - x[i - 1]);
                }
                // calculate integrals of original tacs at x[i]
                li += 0.5f * (y[i] + y[i - 1]) * (x[i] - x[i - 1]);
                lii += 0.5f * (li + lil) * (x[i] - x[i - 1]);
                lil = li;
            }

            // new values after the last x
            for (; j < newx.Length; j++)
            {
                ty = y[x.Length - 1];
                if (newy != null)
                    newy[j] = ty;
                tyi = li + ty * (newx[j] - x[i - 1]);
                if (int1st != null)
                    int1st[j] = tyi;
                if (int2nd != null)
                    int2nd[j] = tyii = lii + 0.5f * (tyi + li) * (newx[j] - x[i - 1]);
            }
        }

        /// <summary>
        /// Interpolate and integrate PET TAC tacs to frame end tacs.
        /// </summary>
        /// <remarks>
        /// PRE: Any of output arrays may be set to null if not needed. 
        /// Frames must be in ascending time order.
        /// Gaps and small overlap are allowed.
        /// The output must've been allocated
        /// POST:
        /// </remarks>
        /// <param name="x1">frame start tacs</param>
        /// <param name="x2">frame end tacs</param>
        /// <param name="y">avg value during frame</param>
        /// <param name="e">number of frames</param>
        /// <param name="ie">integrals at frame end time</param>
        /// <param name="iie">2nd integrals at frame end time</param>
        public void InterpolateToFrameEndTimes(
            double[] x1, double[] x2,
            double[] y, double[] e, double[] ie, double[] iie)
        {
            if (e == null && ie == null && iie == null)
                throw new TPCException("The output must've been allocated");

            int nr = x1.Length;
            double x;
            int i;
            double last_x = 0.0, last_x2 = 0.0, last_y = 0.0,
                last_integral = 0.0, value = 0.0, integral = 0.0,
                integral2 = 0.0, frame_len = 0.0, xdist = 0.0,
                s = 0.0;

            for (i = 0; i < nr; i++)
            {
                frame_len = x2[i] - x1[i];
                if (frame_len < 0.0)
                    throw new TPCException("5");

                x = 0.5 * (x1[i] + x2[i]);
                xdist = x - last_x;
                if (last_x > 0.0 && xdist <= 0.0)
                    throw new TPCException("6");

                if (x < 0)
                {
                    if (e != null)
                        e[i] = value;
                    if (ie != null)
                        ie[i] = integral;
                    if (iie != null)
                        iie[i] = integral2;
                    continue;
                }

                // slope between x[i-1] and x[i]
                s = (y[i] - last_y) / xdist;
                // If there is a big gap in the
                // beginning, it is eliminated
                if (i == 0)
                {
                    if (x1[0] > x2[0] - x1[0])
                    {
                        last_x2 = last_x = x1[0];
                    }
                }

                // gap
                integral += (x1[i] - last_x2) * (last_y +
                    s * ((last_x2 + x1[i]) / 2.0 - last_x));
                integral += frame_len * y[i];
                integral2 += (x2[i] - last_x2) *
                    (integral + last_integral) * 0.5;

                if (e != null && i > 0)
                {
                    // value at previous frame end
                    value = last_y + s * (last_x2 - last_x);
                    e[i - 1] = value;
                }
                if (ie != null)
                    ie[i] = integral;
                if (iie != null)
                    iie[i] = integral2;
                last_x = x;
                last_x2 = x2[i];
                last_y = y[i];
                last_integral = integral;
            }

            if (e != null)
            {
                // Value for the last frame
                value = last_y + s * (last_x2 - last_x);
                e[i - 1] = value;
            }
        }

        /// <summary>
        /// Interpolate and integrate TAC to PET frames.
        /// 
        /// It is assumed, that original tacs is not from framed tacs, but that
        /// the values represent the actual value at specified time point, which
        /// allows the integration to be calculated dot-by-dot.
        /// </summary>
        /// <remarks>
        /// PRE:
        /// Original tacs and new x values must be sorted by ascending x.
        /// Subsequent x values can have equal values, enabling the use of step dataFunctions.
        /// The OutputX2values must always be higher or equal than OutputX1values.
        /// PET frames can overlap, but interpolation may then be slower. Negative x (time) values can be processed.
        /// The  output tacs must've been allocated.
        /// POST:
        /// If output y is null, the first and second integral values are not calculated.
        /// If necessary, the tacs is extrapolated assuming that 
        ///     1) y[inf]=y[nr-1] and 
        ///     2) if x[0]>0 and y[0]>0, an imaginary line is drawn from (0,0) to (x[0],y[0]). 
        /// </remarks>
        /// <param name="x">Times of original tacs</param>
        /// <param name="y">Values of original tacs</param>
        /// <param name="pet_frame_start">PET frame start tacs; frames may overlap</param>
        /// <param name="pet_frame_end">PET frame end tacs; frames may overlap</param>
        /// <param name="interpolated_y">Interpolated mean values in PET frames</param>
        /// <param name="integral1st">Integral at frame mid time</param>
        /// <param name="integral2nd">2nd integral at frame mid time</param>
		public static void InterpolateToPETFrames(double[] x, double[] y,
			double[] pet_frame_start, double[] pet_frame_end,
			out double[] interpolated_y, out double[] integral1st, out double[] integral2nd)
        {

			interpolated_y = new double[pet_frame_start.Length];
			integral1st = new double[pet_frame_start.Length];
			integral2nd = new double[pet_frame_start.Length];
            
            
           
            
            int overlap=0;
            int zeroframe=0;
			double[] petx = new double[3];
			double[] pety = new double[3];
			double[] petyi = new double[3];
			double[] petyii = new double[3];
            
            /* Check for tacs */
            if (x.Length < 1 || pet_frame_start.Length < 1) throw new TPCException("Frame length was <= 1");

            // Check that PET input tacs is not totally outside the original input tacs
            if (pet_frame_end[pet_frame_start.Length - 1] <= x[0] || pet_frame_start[0] >= x[x.Length - 1])
                throw new TPCException("PET tacs are totally outside the original input tacs");

            /* Check frame lengths, also for overlap and zero length frames */
			double fdur;
            for (int fi = 0; fi < pet_frame_start.Length; fi++) {
                /* Calculate frame length */
                fdur = pet_frame_end[fi] - pet_frame_start[fi];
                /* If frame length is <0, that is an error */
                if (fdur < 0.0) throw new TPCException("Frame length was negative");
                if (fdur == 0.0) zeroframe++;
                /* Overlap? */
                if( fi > 0 && pet_frame_end[fi-1] > pet_frame_start[fi] ) overlap++;
            }
            
            /* Interpolate and integrate one frame at a time, if there is:
            -overlap in frames
            -frames of zero length
            -only few interpolated frames
            -if only integrals are needed
            -if neither of integrals is needed, then there is no temp memory to
            do otherwise
            */
            if (overlap > 0 || zeroframe > 0 || pet_frame_start.Length <= 3)
            {
                for (int fi=0; fi < pet_frame_start.Length; fi++) 
                {
                    /* Set frame start, middle and end tacs */
                    petx[0]=pet_frame_start[fi]; 
                    petx[2]=pet_frame_end[fi]; 
                    petx[1]=0.5f*(petx[0]+petx[2]);
                    /* Calculate frame length */
                    fdur=petx[2]-petx[0];
                    /* If frame length is <0, that is an error */
                    if (fdur < 0.0) throw new TPCException("Frame length was negative");
                    /* If frame length is 0, then use direct interpolation */
                    if (fdur==0.0) 
                    {
                        Interpolate(x, y, petx, out pety, out petyi, out petyii);
                        interpolated_y[fi]=petyi[0];
                        integral1st[fi]=petyi[0];
                        integral2nd[fi]=petyii[0];
                        continue;
                    }
                    /* Calculate integrals at frame start, middle, and end */
					double[] not_needed;
                    Interpolate(x, y, petx, out not_needed, out petyi, out petyii);
                    /* Set output integrals, if required */
                    integral1st[fi]=petyi[1];
                    integral2nd[fi]=petyii[1];
                    /* Calculate frame mean, if required */
                    interpolated_y[fi] = (petyi[2] - petyi[0]) / fdur;
                } // next frame

            } 
            else 
            {
                /* Set temp array */
				double[] tp; 
                tp = integral2nd; 
                
                /* Integrate at frame start tacs */
				double[] not_needed;
				double[] not_needed_2;
                Interpolate(x, y, pet_frame_start, out not_needed, out tp, out not_needed_2);
                /* Integrate at frame end tacs */
                Interpolate(x, y, pet_frame_end, out not_needed, out interpolated_y, out not_needed_2);
                /* Calculate average frame value */
                for (int fi = 0; fi < pet_frame_start.Length; fi++)
                    interpolated_y[fi] = (interpolated_y[fi] - tp[fi]) / (pet_frame_end[fi] - pet_frame_start[fi]);
                /* Calculate integrals */
                
                for (int fi = 0; fi < pet_frame_start.Length; fi++) 
                    pet_frame_start[fi] += 0.5f * (pet_frame_end[fi] - pet_frame_start[fi]);
                Interpolate(x, y, pet_frame_start, out not_needed, out integral1st, out integral2nd);
                for (int fi = 0; fi < pet_frame_start.Length; fi++) 
                    pet_frame_start[fi] -= (pet_frame_end[fi] - pet_frame_start[fi]);
                
            }
        }
    }
}
