/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;

namespace TPClib.Modeling
{
    /// <summary>
    /// Iterative method for linear least-squares fit with errors in both
    /// coordinates. This function is fully based on article [3].
    /// 
    /// For n tacs-point pairs (x[i], y[i]) each point has its own weighting factors
    /// in (wx[i], wy[i]). This routine finds the values of the parameters m (slope)
    /// and c (intercept, ic) that yield the "best-fit" of the model equation
    /// Y = mX + c to the tacs, where X and Y are the predicted or calculated values
    /// of the tacs points.
    /// 
    /// Weighting factors wx and wy must be assigned as the inverses of the variances
    /// or squares of the measurement uncertainties (SDs), i.e. w[i]=1/(sd[i])^2
    /// 
    /// If true weights are unknown but yet the relative weights are correct,
    /// the slope, intercept and residuals (WSS) will be correct.
    /// The applied term S/(N-2) makes also the estimate of sigma (sd) of slope
    /// less dependent on the scaling of weights. The sigmas are not exact, since
    /// only the lowest-order terms in Taylor-series expansion are incorporated;
    /// anyhow sigmas are more accurate than the ones based on York algorithm.
    /// 
    /// One or more tacs points can be excluded from the fit by setting either x or y
    /// weight to 0.
    /// 
    /// References:
    /// 1. York, D. Linear-squares fitting of a straight line.
    ///    Can. J. Phys. 1966;44:1079-1086.
    /// 2. Lybanon, M. A better least squares method when both variables have
    ///    uncertainties. Am. J. Phys. 1984;52:22-26 and 276-278.
    /// 3. Reed BC. Linear least-squares fits with errors in both coordinates. II:
    ///    Comments on parameter variances. Am. J. Phys. 1992;60:59-62.
    /// </summary>
    public static class LinearLeastSquares
    {
        /// <summary>
        /// LineatLeastSquares calculation
        /// </summary>
        /// <param name="x">coordinates of tacs points (of dimension n)</param>
        /// <param name="y">coordinates of tacs points (of dimension n)</param>
        /// <param name="wx">weighting factors in x</param>
        /// <param name="wy">weighting factors in y</param>
        /// <param name="tol">allowed tolerance in slope estimation</param>
        /// <param name="out_w">Effective weights</param>
        /// <param name="ic">Estimated intercept</param>
        /// <param name="slope">Estimated slope</param>
        /// <param name="nwss">sqrt(WSS)/wsum of the residuals</param>
        /// <param name="sic">expected sd of intercept at calculated points</param>
        /// <param name="sslope">Expected sd of slope at calculated points</param>
        /// <param name="cx">Estimated tacs points (X,y)</param>
        /// <param name="cy">Estimated tacs points (x,Y)</param>
        public static void Calculate(double[] x, double[] y, double[] wx, double[] wy,
                              double tol, out double[] out_w, out double ic, out double slope, out double nwss,
                              out double sic, out double sslope, out double[] cx, out double[] cy)
        {
            int niter = 0, nn;
            double c, m, f, xb = 0.0, yb = 0.0, qa = 0.0, qb = 0.0, qc, ss = 0.0, s1, s2, wsum = 0.0;
            double xsum = 0.0, x2sum = 0.0, ysum = 0.0, xysum = 0.0, delta, discr, sqdis;
            double m_1, m_2, bcont, m2 = 0.0, w2;
            double u, v, AA, BB, CC, DD, EE, FF, GG, HH, JJ;
            double varc, varm, dmx, dmy, dcx, dcy;

            cx = new double[x.Length];
            cy = new double[x.Length];
            out_w = new double[x.Length];

            /*
            *  Lets look what we got for arguments
            */

            // Check that there is some tacs
            if (x.Length < 2) throw new TPCException("Error: Input-tacs is too short");

            if (tol < 1.0e-100) throw new TPCException("Error: Tolerance must be > 0");

            /* If only 2 datapoints */
            if (x.Length == 2)
            {
                f = x[1] - x[0];
                if (f == 0.0)
                {
                    slope = ic = sic = sslope = nwss = out_w[0] = out_w[1] = 0.0;
                    return;
                }
                slope = (y[1] - y[0]) / f;
                ic = y[0] - slope * x[0];
                sic = sslope = 0;
                out_w[0] = 1.0;
                out_w[1] = 1.0;
                nwss = 0.0;
                return;
            }

            /*
            *  Fit the LLSQ line
            */

            /* First estimation of the slope and intercept by unweighted regression */
            for (int i = 0; i < x.Length; i++)
            {
                xsum += x[i];
                ysum += y[i];
                x2sum += x[i] * x[i];
                xysum += x[i] * y[i];
            }
            delta = (double)x.Length * x2sum - xsum * xsum;
            /* Initial guesses of the slope and intercept */
            if (delta == 0.0)
            {
                slope = ic = sic = sslope = nwss = out_w[0] = out_w[1] = 0.0;
                return;
            }
            m = ((double)x.Length * xysum - xsum * ysum) / delta;
            c = (x2sum * ysum - xsum * xysum) / delta;


            /* Begin the iterations */
            bcont = m + 2.0 * tol;
            while (Math.Abs(m - bcont) > tol && niter < 20)
            {

                bcont = m;
                niter++;
                /* Compute the weighting factors */
                m2 = m * m;
                nn = 0;
                for (int i = 0; i < x.Length; i++)
                {
                    if (wx[i] <= 0.0 || wy[i] <= 0.0) out_w[i] = 0.0;
                    else
                    {
                        out_w[i] = wx[i] * wy[i] / (m2 * wy[i] + wx[i]);
                        nn++;
                    }

                }
                if (nn < 2)
                {

                    slope = ic = sic = sslope = nwss = 0.0;
                    return;
                }
                /* Compute the barycentre coordinates */
                xb = yb = wsum = 0.0;
                for (int i = 0; i < x.Length; i++)
                {
                    xb += out_w[i] * x[i];
                    yb += out_w[i] * y[i];
                    wsum += out_w[i];
                }
                if (wsum <= 0.0) throw new TPCException("All weights are 0");
                xb /= wsum;
                yb /= wsum;

                /* Estimate the slope as either of the roots of the quadratic */
                /* compute the coefficients of the quadratic */
                qa = qb = qc = 0.0;
                for (int i = 0; i < x.Length; i++)
                    if (out_w[i] > 0.0)
                    {
                        u = x[i] - xb;
                        v = y[i] - yb;
                        w2 = out_w[i] * out_w[i];
                        qa += w2 * u * v / wx[i];
                        qb += w2 * (u * u / wy[i] - v * v / wx[i]);
                        qc += -w2 * u * v / wy[i];
                    }

                if (qa == 0.0)
                {
                    m = 0.0;
                    /* Calculate WSS */
                    ss = 0.0;
                    for (int i = 0; i < x.Length; i++)
                    {
                        f = v = y[i] - yb;
                        ss += out_w[i] * f * f;
                    }
                }
                else
                    if (qa == 1.0)
                    {
                        /* check if quadratic reduces to a linear form */
                        m = -qc / qb;
                        /* Calculate WSS */
                        ss = 0.0;
                        for (int i = 0; i < x.Length; i++)
                        {
                            u = x[i] - xb;
                            v = y[i] - yb;
                            f = v - m * u;
                            ss += out_w[i] * f * f;
                        }
                    }
                    else
                    {
                        /* discriminant of quadratic */
                        discr = qb * qb - 4.0 * qa * qc;

                        if (discr <= 0.0) sqdis = 0.0;
                        else sqdis = Math.Sqrt(discr);
                        /* compute the two solutions of quadratic */
                        m_1 = (-qb + sqdis) / (2.0 * qa);
                        m_2 = (-qb - sqdis) / (2.0 * qa);
                        /* Calculate WSS for both solutions */
                        s1 = s2 = 0.0;
                        for (int i = 0; i < x.Length; i++)
                        {
                            u = x[i] - xb;
                            v = y[i] - yb;
                            f = v - m_1 * u;
                            s1 += out_w[i] * f * f;
                            f = v - m_2 * u;
                            s2 += out_w[i] * f * f;
                        }
                        /* choose the solution with lower WSS */
                        if (s1 <= s2)
                        {
                            m = m_1;
                            ss = s1;
                        }
                        else
                        {
                            m = m_2;
                            ss = s2;
                        }
                    }
                /* Calculate the intercept */
                c = yb - m * xb;
            } // end of iteration loop
            ic = c;
            slope = m;
            nwss = Math.Sqrt(ss) / wsum;
            /* Set function return values */


            /*
            *  Calculate the fitted line
            */
            for (int i = 0; i < x.Length; i++)
            {
                if (out_w[i] > 0.0)
                {
                    f = out_w[i] * (c + m * x[i] - y[i]); /* Lagrangian multiplier */
                    cx[i] = x[i] - f * m / wx[i];
                    cy[i] = y[i] + f / wy[i];
                }
                else
                {
                    cx[i] = x[i];
                    cy[i] = y[i];
                }
            }

            /*
            *  Estimate the variances of the parameters (Reed 1992)
            */
            /* varm = var of slope ; varc = var of intercept  */

            /* Use the true nr of tacs points, i.e. exclude the ones with zero weight */
            nn = 0;
            for (int i = 0; i < x.Length; i++)
                if (out_w[i] > 0.0) nn++;
            if (nn < 3)
            {
                sslope = sic = 0.0;
                return;
            }

            /* Compute the barycentre coordinates again from computed tacs points */
            xb = yb = wsum = 0.0;
            for (int i = 0; i < x.Length; i++)
            {
                xb += out_w[i] * cx[i];
                yb += out_w[i] * cy[i];
                wsum += out_w[i];
            }
            if (wsum <= 0.0) throw new TPCException("All weights are 0");
            xb /= wsum;
            yb /= wsum;


            /* common factors */
            HH = JJ = qa = qb = qc = 0.0;
            for (int i = 0; i < x.Length; i++)
                if (out_w[i] > 0.0)
                {
                    u = cx[i] - xb;
                    v = cy[i] - yb;
                    w2 = out_w[i] * out_w[i];
                    qa += w2 * u * v / wx[i];
                    qb += w2 * (u * u / wy[i] - v * v / wx[i]);
                    qc += -w2 * u * v / wy[i];
                    HH += out_w[i] * out_w[i] * v / wx[i];
                    JJ += out_w[i] * out_w[i] * u / wx[i];
                }
            HH *= -2.0 * m / wsum;
            JJ *= -2.0 * m / wsum;

            AA = BB = CC = 0.0;
            for (int i = 0; i < x.Length; i++)
                if (out_w[i] > 0.0)
                {
                    u = cx[i] - xb; v = cy[i] - yb;
                    w2 = out_w[i] * out_w[i];
                    AA += out_w[i] * out_w[i] * out_w[i] * u * v / (wx[i] * wx[i]);
                    BB -= w2 * (4.0 * m * (out_w[i] / wx[i]) * (u * u / wy[i] - v * v / wx[i])
                                - 2.0 * v * HH / wx[i] + 2.0 * u * JJ / wy[i]);
                    CC -= (w2 / wy[i]) * (4.0 * m * out_w[i] * u * v / wx[i] + v * JJ + u * HH);
                }
            if (m != 0) AA = 4.0 * m * AA - wsum * HH * JJ / m;
            else AA = 0.0;

            /* sigmas */
            varc = varm = 0.0;
            for (int j = 0; j < x.Length; j++)
                if (out_w[j] > 0.0)
                {
                    /* factors for this j */
                    DD = EE = FF = GG = 0.0;
                    for (int i = 0; i < x.Length; i++)
                        if (out_w[i] > 0.0)
                        {
                            u = cx[i] - xb;
                            v = cy[i] - yb;
                            w2 = out_w[i] * out_w[i];
                            if (i == j) delta = 1.0;
                            else delta = 0.0; /* Kronecker delta */
                            f = delta - out_w[j] / wsum;
                            DD += (w2 * v / wx[i]) * f;
                            EE += (w2 * u / wy[i]) * f;
                            FF += (w2 * v / wy[i]) * f;
                            GG += (w2 * u / wx[i]) * f;
                        }
                    EE *= 2.0;
                    /* derivatives at jth tacs point */
                    f = (2.0 * m * qa + qb - AA * m2 + BB * m - CC);
                    m2 = m * m;
                    dmx = -(m2 * DD + m * EE - FF) / f;
                    dmy = -(m2 * GG - 2.0 * m * DD - EE / 2.0) / f;
                    dcx = (HH - m * JJ - xb) * dmx - m * out_w[j] / wsum;
                    dcy = (HH - m * JJ - xb) * dmy + out_w[j] / wsum;
                    /* sum terms for sigmas */
                    varm += dmy * dmy / wy[j] + dmx * dmx / wx[j];
                    varc += dcy * dcy / wy[j] + dcx * dcx / wx[j];

                }
            varm *= ss / (double)(nn - 2);
            varc *= ss / (double)(nn - 2);

            sslope = Math.Sqrt(varm);
            sic = Math.Sqrt(varc);


        }

        /// <summary>
        /// Simple non-iterative perpendicular line fitting. This function is fully based on the article (1).
        /// 
        /// References:
        /// 1. Varga J and Szabo Z. Modified regression model for the Logan plot.
        ///    J Cereb Blood Flow Metab 2002; 22:240-244.
        /// </summary>
        /// <param name="x">coordinates of tacs points</param>
        /// <param name="y">coordinates of tacs points</param>
        /// <param name="slope">Estimated slope</param>
        /// <param name="ic">Estimated intercept</param>
        /// <param name="ssd">Sum of squared distances / x.Length</param>
        /// <returns>rue or false, depending LlsqPerp succeeds or not</returns>
        public static bool LlsqPerp(double[] x, double[] y, out double slope, out double ic, out double ssd)
        {
            double a, b, c, d, m1, m2;
            slope = ic = ssd = double.NaN;
            
            /* Check the tacs */
            if (x.Length < 2 || x == null || y == null)
            {
                Console.WriteLine("Input tacs is too short or empty.");
                return false;
            }
            if (x.Length != y.Length)
            {
                Console.WriteLine("x and y must have same lengths");
                return false;
            }
            /* Calculate the means */
            double mx, my;
            mx = my = 0;
            for (int i = 0; i < x.Length; i++) 
            { 
                mx += x[i]; 
                my += y[i]; 
            }
            mx /= x.Length; 
            my /= x.Length;
            
            /* Calculate the Q's */
            double qxx, qyy, qxy;
            qxx = qyy = qxy = 0;
            for (int i = 0; i < x.Length; i++)
            {
                a = x[i] - mx; 
                b = y[i] - my; 
                qxx += a * a; 
                qyy += b * b; 
                qxy += a * b;
            }
            if (qxx < 1.0E-100 || qyy < 1.0E-100)
            {
                //Calculation of Q's failed.
                return false;
            }

            /* Calculate the slope as real roots of quadratic equation */
            a = qxy; 
            b = qxx - qyy; 
            c = -qxy;
            int rnr = Quadratic(a, b, c, out m1, out m2);
            
            if (rnr == 0)
            {
                Console.WriteLine("No roots found in quandratic equation.");
                return false;
            }

            /* Calculate the sum of squared distances for the first root */
            a = m1; 
            b = -1; 
            c = my - m1 * mx;
            double ssd1 = 0;
            double ssd2;
            for (int i = 0; i < x.Length; i++)
            {
                /* calculate the distance from point (x[i],y[i]) to line ax+by+c=0 */
                d = (a * x[i] + b * y[i] + c) / Math.Sqrt(a * a + b * b);
                ssd1 += d * d;
            }

            // Also to the 2nd root, if there is one
            if (rnr == 2)
            {
                a = m2; 
                b = -1; 
                c = my - m2 * mx;
                ssd2 = 0;
                for (int i = 0; i < x.Length; i++)
                {
                    /* calculate the distance from point (x[i],y[i]) to line ax+by+c=0 */
                    d = (a * x[i] + b * y[i] + c) / Math.Sqrt(a * a + b * b);
                    ssd2 += d * d;
                }
            }
            else ssd2 = ssd1;
            
            /* If there were 2 roots, select the one with smaller ssd */
            if (rnr == 2 && ssd2 < ssd1) 
            { 
                ssd1 = ssd2; 
                m1 = m2; 
            }
            
            /* Set the slope and intercept */
            slope = m1; 
            ic = my - m1 * mx; 
            ssd = ssd1 / x.Length;
            return true;
        }

        /// <summary>
        /// Simple non-iterative perpendicular line fitting.
        /// This function is fully based on the article [1].
        /// 
        /// References:
        /// 1. Varga J and Szabo Z. Modified regression model for the Logan plot.
        ///    J Cereb Blood Flow Metab 2002; 22:240-244.
        /// </summary>
        /// <param name="x">coordinates of tacs points</param>
        /// <param name="y">coordinates of tacs points</param>
        /// <param name="from">Coordinate index, where calculation starts</param>
        /// <param name="nr">The count of tacs points included in calculation</param>
        /// <param name="slope">Estimated slope</param>
        /// <param name="ic">Estimated intercept</param>
        /// <param name="ssd">Sum of squared distances / nr</param>
        /// <returns>True or false, depending LlsqPerp succeeds or not</returns>
        public static bool LlsqPerp_2(double[] x, double[] y, int from, int nr, out double slope, out double ic, out double ssd)
        {
            double a, b, c, d, m1, m2;
            slope = ic = ssd = double.NaN;
            
            /* Check the tacs */
            if (nr+from < 2 || x == null || y == null)
            {
                Console.WriteLine("Input tacs is too short or empty.");
                return false;
            }
            if (x.Length != y.Length)
            {
                Console.WriteLine("x and y must have same lengths");
                return false;
            }
            /* Calculate the means */
            double mx, my;
            mx = my = 0;
            for (int i = from; i < nr+from; i++)
            {
                mx += x[i];
                my += y[i];
            }
            mx /= nr;
            my /= nr;

            /* Calculate the Q's */
            double qxx, qyy, qxy;
            qxx = qyy = qxy = 0;
            for (int i = from; i < nr+from; i++)
            {
                a = x[i] - mx;
                b = y[i] - my;
                qxx += a * a;
                qyy += b * b;
                qxy += a * b;
            }
            if (qxx < 1.0E-100 || qyy < 1.0E-100)
            {
                //Calculation of Q's failed.
                return false;
            }

            /* Calculate the slope as real roots of quadratic equation */
            a = qxy;
            b = qxx - qyy;
            c = -qxy;
            int rnr = Quadratic(a, b, c, out m1, out m2);

            if (rnr == 0)
            {
                Console.WriteLine("No roots found in quandratic equation.");
                return false;
            }

            /* Calculate the sum of squared distances for the first root */
            a = m1;
            b = -1;
            c = my - m1 * mx;
            double ssd1 = 0;
            double ssd2;
            for (int i = from; i < nr+from; i++)
            {
                /* calculate the distance from point (x[i],y[i]) to line ax+by+c=0 */
                d = (a * x[i] + b * y[i] + c) / Math.Sqrt(a * a + b * b);
                ssd1 += d * d;
            }

            // Also to the 2nd root, if there is one
            if (rnr == 2)
            {
                a = m2;
                b = -1;
                c = my - m2 * mx;
                ssd2 = 0;
                for (int i = from; i < nr+from; i++)
                {
                    /* calculate the distance from point (x[i],y[i]) to line ax+by+c=0 */
                    d = (a * x[i] + b * y[i] + c) / Math.Sqrt(a * a + b * b);
                    ssd2 += d * d;
                }
            }
            else ssd2 = ssd1;

            /* If there were 2 roots, select the one with smaller ssd */
            if (rnr == 2 && ssd2 < ssd1)
            {
                ssd1 = ssd2;
                m1 = m2;
            }

            /* Set the slope and intercept */
            slope = m1;
            ic = my - m1 * mx;
            ssd = ssd1 / nr;
            return true;
            
        }
        
        /// <summary>
        /// Finds the real roots of a*x^2 + b*x + c = 0.
        /// </summary>
        /// <param name="a">multipliers of x^2</param>
        /// <param name="b">multipliers of x</param>
        /// <param name="c">constant term</param>
        /// <param name="m1">root 1</param>
        /// <param name="m2">root 2</param>
        /// <returns>Returns the nr of roots, and the roots in m1 and m2.</returns>
        private static int Quadratic(double a, double b, double c, out double m1, out double m2)
        {
            double discriminant, r, r1, r2, sgnb, temp;
            m1 = m2 = double.NaN;

            if (a == 0) 
            { 
                if (b == 0) return (0); 
                else 
                { 
                    m1 = m2 = -c / b; 
                    return (1); 
                } 
            }
            discriminant = b * b - 4 * a * c;
            if (discriminant > 0)
            {
                if (b == 0) 
                { 
                    r = Math.Abs(0.5 * Math.Sqrt(discriminant) / a); 
                    m1 = -r; 
                    m2 = r; 
                }
                else
                {
                    sgnb = (b > 0 ? 1 : -1); 
                    temp = -0.5 * (b + sgnb * Math.Sqrt(discriminant));
                    r1 = temp / a; 
                    r2 = c / temp; 
                    if (r1 < r2) 
                    { 
                        m1 = r1; 
                        m2 = r2; 
                    } 
                    else 
                    { 
                        m1 = r2; 
                        m2 = r1; 
                    }
                }
                return (2);
            }
            else if (discriminant == 0)
            {
                m1 = -0.5 * b / a; 
                m2 = -0.5 * b / a;
                return (2);
            }
            else
            {
                return (0);
            }
        }
    }
}

