/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using TPClib.Common;

namespace TPClib.ROI
{
    /// <summary>
    /// Cirle-shaped 2D Region Of Interest (ROI) reference point is located at the center of the circle.
    /// The circle is currently defined only in transaxial plane.
    /// </summary>
    public class CircleROI : ROI, IGeometricalROI
    {
        /// <summary>
        /// Circle radius in units
        /// </summary>
        /// <seealso cref="ROI.Unit"/>
        public double Radius;

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        /// <param name="x">x-coordinate</param>
        /// <param name="y">y-coordinate</param>
        /// <param name="z">z-coordinate</param>
        /// <param name="radius">radius in units</param>
        /// <seealso cref="ROI.Unit"/>
        public CircleROI(double x, double y, double z, double radius)
            : base(ROI_Type.Circle)
        {
            this.Location = new Point(x,y,z);
            this.Radius = radius;
        }

        /// <summary>
        /// Creates circle ROI
        /// </summary>
        public CircleROI()
            : base(ROI_Type.Circle)
        {
        }

        /// <summary>
        /// Creates deep copy of this object
        /// </summary>
        public override object Clone()
        {
            CircleROI r = new CircleROI();
            r.Radius = this.Radius;
            r.ROIName = this.ROIName;
            r.Unit = this.Unit;
            return r;
        }

        /// <summary>
        /// Fills maskable item with 1's into given size mask image. 
        /// Origo is considered to be located at bottom left low corner of mask. 
        /// Fill is done with midpoint circle algorithm (or Bresenham's circle algorithm).
        /// </summary>
        /// <param name="mask">target mask image</param>
        /// <param name="method">filling method</param>
        public override void Fill(ref TPClib.Image.MaskImage mask, Fill_Method method)
        {
            int f = 1 - (int)Radius;
            int ddF_x = 1;
            int ddF_y = -2 * (int)Radius;
            int x = 0;
            int y = (int)Radius;
            int i = 0;
            int loc_x = (int)Location.X;
            int loc_y = (int)Location.Y;
            int loc_z = (int)Location.Z;

            mask[loc_x, loc_y + (int)Radius, loc_z] = 1;
            mask[loc_x, loc_y - (int)Radius, loc_z] = 1;
            mask[loc_x + (int)Radius, loc_y, loc_z] = 1;
            mask[loc_x - (int)Radius, loc_y, loc_z] = 1;
            for (i = loc_x - (int)Radius; i < loc_x + (int)Radius; i++) mask[i, loc_y, loc_z] = 1;

            //proceed along edge of first octant (zero angle from x-axis) of circle
            while (x < y) {
                // ddF_x == 2 * x + 1;
                // ddF_y == -2 * y;
                // f == x*x + y*y - radius*radius + 2*x - y + 1;
                if (f >= 0) {
                    y--;
                    ddF_y += 2;
                    f += ddF_y;
                }
                x++;
                ddF_x += 2;
                f += ddF_x;
                //mask all octants 
                mask[loc_x + x, loc_y + y, loc_z] = 1;
                mask[loc_x - x, loc_y + y, loc_z] = 1;
                mask[loc_x + x, loc_y - y, loc_z] = 1;
                mask[loc_x - x, loc_y - y, loc_z] = 1;
                mask[loc_x + y, loc_y + x, loc_z] = 1;
                mask[loc_x - y, loc_y + x, loc_z] = 1;
                mask[loc_x + y, loc_y - x, loc_z] = 1;
                mask[loc_x - y, loc_y - x, loc_z] = 1;
                //fill
                for (i = loc_x - x; i < loc_x + x; i++) mask[i, loc_y + y, loc_z] = 1;
                for (i = loc_x - x; i < loc_x + x; i++) mask[i, loc_y - y, loc_z] = 1;
                for (i = loc_x - y; i < loc_x + y; i++) mask[i, loc_y + x, loc_z] = 1;
                for (i = loc_x - y; i < loc_x + y; i++) mask[i, loc_y - x, loc_z] = 1;
            }
        }
        /// <summary>
        /// Gets bounding box of circle
        /// </summary>
        /// <returns>bounds of this ROI</returns>
        public override Limits GetBoundingBox()
        {
            return new Limits(new double[] { this.Location.X - Radius, this.Location.Y - Radius, this.Location.Z },
                              new double[] { this.Location.X + Radius, this.Location.Y + Radius, this.Location.Z });
        }
        /// <summary>
        /// Width of geometrical 2D shape. Note that the orientation may affect actual ROI presentation.
        /// </summary>
        public double Width
        {
            get { return 2*Radius; }
        }
        /// <summary>
        /// Height of geometrical 2D shape. Note that the orientation may affect actual ROI presentation.
        /// </summary>
        public double Height
        {
            get { return 2 * Radius; }
        }
    }
}
