/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using TPClib.Common;

namespace TPClib.ROI 
{
    /// <summary>
    /// Class that reads ROI -files of Imadeus
    /// </summary>
    public class ImadeusROIFile : ROIFile
    {
        /// <summary>
        /// Combination of ROIs in Imadeus ROI file.
        /// </summary>
        public class ROICombination
        {
            /// <summary>
            /// Name of combinated roiset
            /// </summary>
            public String Name;
            /// <summary>
            /// Roi names which are part of combination
            /// </summary>
            public List<String> ROIs;
            /// <summary>
            /// Default constructor
            /// </summary>
            public ROICombination()
            {
                Name = ".";
                ROIs = new List<string>();
            }
        }

        /// <summary>
        /// Imadeus specific header information.
        /// </summary>
        public ImadeusHeader Header = new ImadeusHeader();

        /// <summary>
        /// Combinations of Imadeus ROIs, or empty list if no combinations
        /// </summary>
        public List<ROICombination> Combinations = new List<ROICombination>();

        /// <summary>
        /// Imadeus file format specific header information
        /// </summary>
        public class ImadeusHeader
        {
            /// <summary>
            /// Number of regions in file
            /// </summary>
            public int Regions = 0;
            /// <summary>
            /// Related image filename
            /// </summary>
            public string Image = "";
            /// <summary>
            /// Related raw data filename
            /// </summary>
            public string RawImage = "";
            /// <summary>
            /// ROI set name
            /// </summary>
            public string Name = "";
            /// <summary>
            /// Related image orientation 
            /// </summary>
            public string Orientation = "";
            /// <summary>
            /// Image orientation as Imadeus number code
            /// </summary>
            public int Orient = 0;
            /// <summary>
            /// Related image x-dimension
            /// </summary>
            public int XResolution = 0;
            /// <summary>
            /// Related image y-dimension
            /// </summary>
            public int YResolution = 0;
            /// <summary>
            /// Related image z-resolution
            /// </summary>
            public int ZResolution = 0;
            /// <summary>
            /// Related image voxel x-dimension in mm
            /// </summary>
            public float XVoxelDim = 0.0f;
            /// <summary>
            /// Related image voxel y-deimesion in mm
            /// </summary>
            public float YVoxelDim = 0.0f;
            /// <summary>
            /// Related image voxel z-dimension in mm
            /// </summary>
            public float ZVoxelDim = 0.0f;
            /// <summary>
            /// Flag for image flip in X-axis direction
            /// </summary>
            public bool XFlip = false;
            /// <summary>
            /// Flag for image flip in Y-axis direction
            /// </summary>
            public bool YFlip = false;
            /// <summary>
            /// Flag for image flip in Z-axis direction
            /// </summary>
            public bool ZFlip = false;
            /// <summary>
            /// Origo x-coordinate, note that this may change between Imadeus software versions.
            /// </summary>
            public int XOri = 0;
            /// <summary>
            /// Origo x-coordinate, note that this may change between Imadeus software versions.
            /// </summary>
            public int YOri = 0;
            /// <summary>
            /// Origo x-coordinate, note that this may change between Imadeus software versions.
            /// </summary>
            public int ZOri = 0;
            /// <summary>
            /// Free comment text.
            /// </summary>
            public string Comment = "";
            /// <summary>
            /// Gets [Creator] field as a string
            /// </summary>
            /// <returns>string representation of '[Creator]' field at the bottom of the file</returns>
            public String GetDefinitionField()
            {
                String output =
                    "[Definition]\r\n"+
                    "Regions=" + Regions + "\r\n" +
                    "Image=" + Image + "\r\n" +
                    "RawImage=" + RawImage + "\r\n" +
                    "Name=" + Name + "\r\n" +
                    "Orientation=" + Orientation+"\r\n" +
                    "Orient=" + Orient + "\r\n" +
                    "XResolution=" + XResolution + "\r\n" +
                    "YResolution=" + YResolution + "\r\n" +
                    "ZResolution=" + ZResolution + "\r\n" +
                    "XVoxelDim=" + XVoxelDim + "\r\n" +
                    "YVoxelDim=" + YVoxelDim + "\r\n" +
                    "ZVoxelDim=" + ZVoxelDim + "\r\n" +
                    "XFlip=" + XFlip + "\r\n" +
                    "YFlip=" + YFlip + "\r\n" +
                    "ZFlip=" + ZFlip + "\r\n" +
                    "XOri=" + XOri + "\r\n" +
                    "YOri=" + YOri + "\r\n" +
                    "ZOri=" + ZOri + "\r\n" +
                    "Comment=" + Comment + "\r\n";
                return output;
            }
            /// <summary>
            /// (Under [Creator] field) Creating software product name, default == 'TPClib'
            /// </summary>
            public String ProductName = "TPClib";
            /// <summary>
            /// (Under [Creator] field) Creator version, default == version information from library
            /// </summary>
            /// <see cref="TPClib.Version.GetVersionInformation()"/>
            public String Version = TPClib.Version.GetVersionInformation();
            /// <summary>
            /// (Under [Creator] field) Copyright notice if any, or empty string
            /// </summary>
            public String Copyright = "";
            /// <summary>
            /// Gets [Creator] field as a string
            /// </summary>
            /// <returns>string representation of '[Creator]' field at the bottom of the file</returns>
            public String GetCreatorField()
            {
                String output = 
                    "[Creator]\r\nProductName="+ProductName+"\r\nVersion="+Version+
                    "\r\nProductName="+ProductName+"\r\nCopyright="+Copyright+"\r\n";
                return output;
            }
        }

        /// <summary>
        /// Format how doubles are stored in ImadeusROIFile
        /// </summary>
        private static System.Globalization.CultureInfo doubleformat = new System.Globalization.CultureInfo("en-GB");

        /// <summary>
        /// Creates an Imadeus ROI file object
        /// </summary>
        /// <param name="filename">ROI filename</param>
        public ImadeusROIFile(String filename) : base( filename )
        {
            Filetype = FileType.Imadeus;
        }

        /// <summary>
        /// Checks if the file is valid Imadeus ROI file.
        /// </summary>
        /// <returns>True if the file is valid</returns>
		public override bool CheckFormat()
		{
			if (filename == null) throw new TPCROIFileException("You have not given filename");
			try
			{
				string text = File.ReadAllText(filename, new ASCIIEncoding());
				if (!text.Contains("[Definition]")) return false;
				if (!text.Contains("XVoxelDim=")) return false;
				if (!text.Contains("XOri=")) return false;
			}
			catch { return false; }
			return true;
		}

        /// <summary>
        /// Writes current data with VOIs into file
        /// </summary>
        /// <param name="vois">VOIs that are written</param>
        public override void WriteFile(ref List<VOI> vois) {
        
            if (filename == null) throw new TPCROIFileException("You have not given filename");

			using (StreamWriter fileWriter = new StreamWriter(new FileStream(filename, FileMode.Create, FileAccess.Write), Encoding.GetEncoding(1252)))
			{
				fileWriter.AutoFlush = true;
				fileWriter.Write(WriteHeader());
				fileWriter.Write(WriteData());
				fileWriter.Write(WriteCombinations());
				fileWriter.Write(WriteCreator());
			}
		}

        /// <summary>
        /// Reads Imadeus ROI files.
        /// </summary>
        public override void ReadFile()
        {
            if (!CheckFormat())
                throw new TPCROIFileException("ReadFile: The file " + filename + " was not in correct Imadeus ROI format.");

            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
			string text = File.ReadAllText(filename, Encoding.GetEncoding(1252));
			String HeaderPart = String.Empty;
			String DataPart = String.Empty;

            try
            {
                HeaderPart = text.Substring(0, text.IndexOf("\n[ROI1]"));
                DataPart = text.Substring(text.IndexOf("\n[ROI1]"));
            }
            catch (Exception)
            {
                throw new TPCROIFileException("ReadFile: The file " + filename + " was not in correct Imadeus ROI format.");
            }

            ReadHeader(HeaderPart);
            ReadData(DataPart);
            ReadCombinations(text);
            ReadCreator(text);
        }

        /// <summary>
        /// Writes the contents of ImadeusROIFile to string
        /// </summary>
        /// <returns>String containing the contents of ImadeusFile</returns>
        public string GetContentsAsString()
        {
            return WriteHeader() + WriteData() + WriteCombinations() + WriteCreator();
        }

        #region private members

        /// <summary>
        /// Writes current header information as string
        /// </summary>
        /// <returns>Header data as string</returns>
        private String WriteHeader()
        {
            StringBuilder str = new StringBuilder();

            str.Append("[Definition]\r\n");
            str.Append("Regions=" + VOIs.Count + "\r\n");
            str.Append("Image=" + Header.Image + "\r\n");
            str.Append("RawImage=" + Header.RawImage + "\r\n");
            str.Append("Name=" + Header.Name + "\r\n");
            str.Append("Orientation=" + Header.Orientation + "\r\n");
            str.Append("Orient=" + Header.Orient + "\r\n");
            str.Append("XResolution=" + Header.XResolution + "\r\n");
            str.Append("YResolution=" + Header.YResolution + "\r\n");
            str.Append("ZResolution=" + Header.ZResolution + "\r\n");
            str.Append("XVoxelDim=" + Header.XVoxelDim + "\r\n");
            str.Append("YVoxelDim=" + Header.YVoxelDim + "\r\n");
            str.Append("ZVoxelDim=" + Header.ZVoxelDim + "\r\n");
            str.Append("XFlip=" + HelpFunctions.Print1_0(Header.XFlip) + "\r\n");
            str.Append("YFlip=" + HelpFunctions.Print1_0(Header.YFlip) + "\r\n");
            str.Append("ZFlip=" + HelpFunctions.Print1_0(Header.ZFlip) + "\r\n");
            str.Append("XOri=" + Header.XOri + "\r\n");
            str.Append("YOri=" + Header.YOri + "\r\n");
            str.Append("ZOri=" + Header.ZOri + "\r\n");
            str.Append("Comment=" + Header.Comment + "\r\n\r\n");

            return str.ToString();
        }

        /// <summary>
        /// Reads header data from string
        /// </summary>
        /// <param name="text">Imadeus file ROI header information in string format</param>
        private void ReadHeader( String text )
        {
            //culture used to remove all culture specific behavior in order to force the same behavior in all cultures
            System.Globalization.CultureInfo culture = new System.Globalization.CultureInfo("fi-FI");

            Header.Regions = Convert.ToInt32(HelpFunctions.GetParameterFromText("Regions=", ref text));
            Header.Image = HelpFunctions.GetParameterFromText("Image=", ref text);
            Header.RawImage = HelpFunctions.GetParameterFromText("RawImage=", ref text);
            Header.Name = HelpFunctions.GetParameterFromText("Name=", ref text);
            Header.Orientation = HelpFunctions.GetParameterFromText("Orientation=", ref text);
            Header.Orient = Convert.ToInt32(HelpFunctions.GetParameterFromText("Orient=", ref text));
            Header.XResolution = Convert.ToInt32(HelpFunctions.GetParameterFromText("XResolution=", ref text));
            Header.YResolution = Convert.ToInt32(HelpFunctions.GetParameterFromText("YResolution=", ref text));
            Header.ZResolution = Convert.ToInt32(HelpFunctions.GetParameterFromText("ZResolution=", ref text));
            //commas are changed to dots which is according to format specification (in all cultures)
            Header.XVoxelDim = (float)Convert.ToDouble(HelpFunctions.GetParameterFromText("XVoxelDim=", ref text).Replace('.', ','), culture);
            Header.YVoxelDim = (float)Convert.ToDouble(HelpFunctions.GetParameterFromText("YVoxelDim=", ref text).Replace('.', ','), culture);
            Header.ZVoxelDim = (float)Convert.ToDouble(HelpFunctions.GetParameterFromText("ZVoxelDim=", ref text).Replace('.', ','), culture);
            Header.XFlip = Convert.ToBoolean(HelpFunctions.GetParameterFromText("XFlip=", ref text).Equals("1"));
            Header.YFlip = Convert.ToBoolean(HelpFunctions.GetParameterFromText("YFlip=", ref text).Equals("1"));
            Header.ZFlip = Convert.ToBoolean(HelpFunctions.GetParameterFromText("ZFlip=", ref text).Equals("1"));
            Header.XOri = Convert.ToInt32(HelpFunctions.GetParameterFromText("XOri=", ref text));
            Header.YOri = Convert.ToInt32(HelpFunctions.GetParameterFromText("YOri=", ref text));
            Header.ZOri = Convert.ToInt32(HelpFunctions.GetParameterFromText("ZOri=", ref text));
            Header.Comment = HelpFunctions.GetParameterFromText("Comment=", ref text);
        }

        /// <summary>
        /// Reads ROI data from string
        /// </summary>
        /// <param name="text">Imadeus file ROI data as string</param>
        private void ReadData(String text)
        {
            if (Header.Regions == 0) return;

            char[] separators = new char[] { ' ', ',' };
            
            StringReader sr = new StringReader(text);
            string str = "";
            String[] words;

            for (int i = 0; i < Header.Regions; i++)
            {
                // [ROI..n..]
                words = ReadWordsFromLine( ref sr );
                if (words.Length>0 && !words[0].Contains("[ROI" + (i+1) + "]")) 
                    throw new TPCROIFileException("ReadData: File was not in correct format.");

                ROIStack voi = new ROIStack();
                
                // Information for one ROI. (Contains many regions)
                str = sr.ReadLine();
                voi.Name = HelpFunctions.GetParameterFromText("Name=", str);
                str = sr.ReadLine();
                int num = Convert.ToInt32(HelpFunctions.GetParameterFromText("nRegion=", str));
                //ignore field Color= which seems to be usually zero
                str = sr.ReadLine();
                //there seems to be difference in color coding between Imadeus versions
                //1.20.0.[393] and 1.20.0.[417]
                //using 32-bit unsigned coding works for both
                str = sr.ReadLine();
                int colorcode = Convert.ToInt32(HelpFunctions.GetParameterFromText("col=", str));
                byte[] colorcode_bytes = BitConverter.GetBytes(colorcode);
                voi.Color = new byte[] { colorcode_bytes[0], colorcode_bytes[1], colorcode_bytes[2] };

                // Lets read all the regions for the ROI
                for (int region = 1; region <= num; region++)
                {
                    String line = HelpFunctions.ReadNextLine( ref sr );
                    line = line.Substring( line.IndexOf("=")+1 );
                    words = line.Split(separators, StringSplitOptions.RemoveEmptyEntries);
                    int numPoints = Convert.ToInt32(words[2]);
                    if (words.Length > 0)
                    {
                        int Z = Convert.ToInt32(words[0]);
                        if (Header.ZFlip) Z = Header.ZResolution - Z;
                        TraceROI t_roi = new TraceROI( 0, 0, Z );
                        t_roi.Unit = DataUnit.Pixel;
                        t_roi.ROIName = "Region" + region;                  

                        for (int w = 0; w < numPoints; w++ )
                        {
                            // flipped points
                            double xp, yp; 

                            // point x position
                            if (Header.XFlip) xp = HelpFunctions.StrToDouble(words[w * 2 + 3]);
                            else xp = (Header.XResolution - (HelpFunctions.StrToDouble(words[w * 2 + 3])) - 1.0d);

                            // point y position
                            if (!Header.YFlip) yp = (Header.YResolution - (HelpFunctions.StrToDouble(words[w * 2 + 4])));
                            else yp = HelpFunctions.StrToDouble(words[w * 2 + 4])-1.0d;

                            t_roi.AddPoint(xp, yp);
                        }
                        
                        voi.Add(t_roi);
                    }
                }
                VOIs.Add(voi);
            }         
        }


        /// <summary>
        /// Writes VOI data into string
        /// </summary>
        /// <returns>VOI data as string in Imadeus ASCII format</returns>
        private String WriteData()
        {
            StringBuilder str = new StringBuilder();
            int num = 1;
            foreach (VOI voi in VOIs)
            {
                if (!(voi is ROIStack))
                    throw new TPCROIFileException("Cannot write data because Imadeus file format does not support other VOI types than ROIStack.");
                str.Append("[ROI" + num + "]\r\n");
                str.Append( VOIToString( (voi as ROIStack) ) );
                num++;
            }
            return str.ToString();
        }

        /// <summary>
        /// Writes the contents of VOI to string
        /// </summary>
        /// <returns>String containing the VOI information.</returns>
        private string VOIToString( ROIStack voi )
        {
            StringBuilder str = new StringBuilder();
            str.Append("Name=" + voi.Name + "\r\n");
            str.Append("nRegion=" + voi.Count + "\r\n");
            str.Append("Color=" + BitConverter.ToInt32(new byte[] { voi.Color[0], voi.Color[1], voi.Color[2], 0 }, 0).ToString() + "\r\n");
            //write allways zero value here, because Imadeus format seems to ignore the field
            str.Append("col=" + 0 + "\r\n\r\n");

            int num = 1;

            foreach (ROI roi in voi)
            {
                if (!(roi is TraceROI))
                    throw new TPCROIFileException("Cannot write data because Imadeus file format does not support other ROI types than TraceROI.");
                str.Append("Region" + num + "=" + (int)(roi.Location.Z) + ",2," + (roi as TraceROI).Points.Count + ", ");
                foreach (Point p in (roi as TraceROI).Points)
                {
                    // flipped points
                    double xp, yp; 

                    // point x position
                    if (Header.XFlip) xp = p.X;
                    else xp = (Header.XResolution - p.X - 1.0d);

                    // point y position
                    if (!Header.YFlip)
                        yp = (Header.YResolution - p.Y);
                    else yp = p.Y - 1.0d;

                    str.Append( xp.ToString("f2", doubleformat ) + ", " + yp.ToString("f2", doubleformat ) + ", " );
                }
                // removes the last ", "
                str.Remove(str.Length - 2, 2); 
                str.Append("\r\n\r\n");
                num++;
            }

            return str.ToString();
        }

        /// <summary>
        /// Reads all combinations from file if those exist
        /// </summary>
        /// <param name="text">File contents in string</param>
        private void ReadCombinations(String text)
        {
            if (text.Contains("[Combinations]"))
            {
                String Comb = text.Substring( text.IndexOf("[Combinations]") );
                StringReader reader = new StringReader(Comb);

                reader.ReadLine(); // [Combinations]
                bool End=false;
                int num = 0;
                while (!End)
                {
                    String line = reader.ReadLine(); // This should be Comb0  ...etc...

                    if (line == null || line.Length <= 4) break;
                    String Param = HelpFunctions.GetParameterFromText("Comb"+num+"=", line);

                    // If there is no meaningful line, we add empty combination line
                    if (Param.Equals(".")) break;//Combinations.Add( new ROICombination() );

                    ROICombination combinat = new ROICombination();
                    String[] words = Param.Split( new char[] {' ' }, StringSplitOptions.RemoveEmptyEntries );
                    if (words.Length <= 3) continue;

                    combinat.Name = words[0];
                    for (int i = 3; i < words.Length; i++) combinat.ROIs.Add(words[i]);

                    Combinations.Add(combinat);

                    num++;
                }
            }
        }

        /// <summary>
        /// Reads the Creator part from Imadeus file
        /// </summary>
        /// <param name="text">File contents in String that have '[Creator]' tag data</param>
        private void ReadCreator(String text)
        {
            if (text.Contains("[Creator]"))
            {
                String Comb = text.Substring(text.IndexOf("[Creator]"));
                StringReader reader = new StringReader(Comb);
                // The "[Creator]" line
                reader.ReadLine();
                Header.ProductName = HelpFunctions.GetParameterFromText("ProductName=", reader.ReadLine());
                Header.Version = HelpFunctions.GetParameterFromText("Version=", reader.ReadLine());
                // The Product name is second time here, so the line is useless
                reader.ReadLine();
                Header.Copyright = HelpFunctions.GetParameterFromText("Copyright=", reader.ReadLine());
            }
        }

        /// <summary>
        /// Writes the combinations to disk
        /// </summary>
        private String WriteCombinations()
        {
            if (Combinations.Count > 0)
            {
                StringBuilder str = new StringBuilder();
                str.Append("[Combinations]\r\n");

                int num=0;
                foreach (ROICombination rc in Combinations)
                {
                    // Main part of combination line "Comb0=name 0 2 "
                    str.Append("Comb"+num+"="+rc.Name+" 0 "+rc.ROIs.Count+" ");
                    foreach (String name in rc.ROIs)
                    {
                        // Names of rois that are part of combination
                        str.Append(name+" ");
                    }
                    str.Append("\r\n");
                    num++;
                }
                str.Append("\r\n");
                return str.ToString();
            }
            return "";
        }

        /// <summary>
        /// Writes the Creator part of Imadeus file to string
        /// </summary>
        /// <returns>The '[Creator]' tag part as String</returns>
        private String WriteCreator()
        {
            return Header.GetCreatorField() + "\r\n";
        }

        /// <summary>
        /// This function reads all the words from next line of the file and returns then as String list.
        /// All possible Comment lines are added to Comments list and empty lines are ignored. Returns null if
        /// file has been read to the end.
        /// </summary>
        /// <param name="reader">Reader object that reads the string.</param>
        /// <returns>List of all words in the next line of reader. Null if the string has been readed to the end.</returns>
        protected String[] ReadWordsFromLine(ref System.IO.StringReader reader)
        {
            string line = "";
            bool correct_line = true;

            // space and tab will divide strings
            char[] separators = new char[] { ' ', '\t' };

            // every loop reads one line from file. Empty lines are ignored.
            do
            {
                line = reader.ReadLine();

                // null line means that the string has been read to the end
                if (line == null) return null;

                line.Trim();

                // All comments are added to Comments list
                if (line.Length <= 1) // empty lines are ignored
                {
                    correct_line = false;
                }
                else correct_line = true;
            }
            while (!correct_line);

            //Console.WriteLine(line);

            return line.Split(separators, StringSplitOptions.RemoveEmptyEntries);
        }

#endregion
    }
}
