/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// This class fills matrix with ImageTool masking algorithm
    /// </summary>
    static class ImageToolMask
    {
        /// <summary>
        /// Fills the matrix with ImageTool masking algorithm
        /// </summary>
        /// <param name="matrix">Matrix that will be filled.</param>
        /// <param name="points">Trace points of ROI.</param>
        /// <param name="center_x">ROI center x-coord</param>
        /// <param name="center_y">ROI center x-coord</param>
        public static void Fill(ROI_Matrix matrix, List<Point> points, double center_x, double center_y)
        {
            // If there is not enough points to fill anything
            if (points.Count < 3) return;

            // Converting points to ImageTool:
            List<ROIPoint> p = new List<ROIPoint>();

            //double dx = 0;   // 30 errors
            //double dy = 0;

            //double dx = center_x * factor - (double)((int)center_x) * factor;
            //double dy = center_y * factor - (double)((int)center_y) * factor;

            //double dx = (center_x/factor - (int)(center_x /factor)) * factor; // HelpFunctions.LowRound(center_x);
            //double dy = (center_y/factor - (int)(center_y /factor)) * factor; // HelpFunctions.LowRound(center_y);

            double dx = center_x - (int)center_x;  
            double dy = center_y - (int)center_y;


            for (int i = 0; i < points.Count; i++)
            {
                /*if (fill_method == Fill_Method.ImageTool) return (x_axisX * xx + y_axisX * yy) + (double)((int)x) * factorX;*/
                p.Add(new ROIPoint(points[i].x-dx, points[i].y-dy));
            }

            Edge[] edges = calculateEdges(ref p);
            byte bit = 0;

            for (int y = 0; y < matrix.Height; y++)
            {
                bit = 0;
                int[] crossed = get_CrossingEdges(y, ref edges);
                if ( crossed.Length == 0 && bit==0 ) continue;

                for (int x = 0; x < matrix.Width; x++)
                {
                    matrix[x, y] |= bit;
                    foreach (int cross in crossed)
                    {
                        if (cross == x)
                        {
                            // if there is only one cross(triangle tip) the bit is not flipped
                            if (crossed.Length > 1) bit ^= 0x1;

                            matrix[x, y] = 1;
                        }
                    }
                }
            }
        }

        private static int[] get_CrossingEdges( int y, ref Edge[] edges )
        {
            List<int> crossed = new List<int>();
            int newX=0;

            for (int i = 0; i < edges.Length; i++ )
            {

                //if (((edges[i].start.y <= y) && (y < edges[i].end.y)) || ((edges[i].end.y <= y) && (y < edges[i].start.y)))
                if ( (HelpFunctions.temp_roundf( edges[i].MinY ) <= y) && 
                     (y < HelpFunctions.temp_roundf(edges[i].MaxY)) )
                {
                    newX = HelpFunctions.temp_roundf(edges[i].m * ((double)y - HelpFunctions.temp_roundf(edges[i].MinY)) + (double)edges[i].x);
                    if (newX < 0) newX = 0;
                    crossed.Add( newX );
                }
            }

            return crossed.ToArray();
        }

        /// <summary>
        /// This method calculates the edges from point list. 
        /// </summary>
        /// <param name="points"></param>
        /// <returns></returns>
        private static Edge[] calculateEdges(ref List<Point> points)
        {
            List<Edge> edges = new List<Edge>();

            for (int i = 0; i < points.Count; i++)
            {
                int n = i + 1;
                if (n == points.Count) n = 0; // last point will be connected to first

                if (points[i].y != points[n].y)
                {
                    edges.Add( new Edge(points[i], points[n]) );
                }
            }

            // Last point is connected to first
            //edges[points.Count-1] = new Edge(points[points.Count-1], points[0] );

            return edges.ToArray();
        }


    }

    /// <summary>
    /// This struct is used by Imadeus and ImageTool masking algorithms
    /// </summary>
    struct Edge
    {
        /// <summary>
        /// Maximum y-value of edge points
        /// </summary>
        public double MaxY;
        /// <summary>
        /// Minimum y-value of edge points
        /// </summary>
        public double MinY;
        /// <summary>
        /// x-cordinate of minimum(y) edge point
        /// </summary>
        public double x;
        /// <summary>
        /// x-cordinate of maximum(y) edge point
        /// </summary>
        public double x_end;
        /// <summary>
        /// Slope of edge (delta x / delta y)
        /// </summary>
        public Double m;

        /// <summary>
        /// Constructs new edge. Max/Min values are calculated. Y-cordinates can't be same.
        /// </summary>
        /// <param name="first">First point of edge.</param>
        /// <param name="second">Second point of edge.</param>
        public Edge(Point first, Point second)
        {
            double maxx = 0;

            if (first.y < second.y)
            {
                //edges[edge].minY = roiy[p] * f + pos_y;
                //edges[edge].x = roix[p] * f + pos_x;
                //edges[edge].maxY = roiy[n] * f + pos_y;
                //maxX = roix[n] * f + pos_x;

                this.MinY = first.y;
                this.x = first.x;
                this.x_end = second.x; // ??

                this.MaxY = second.y;
                maxx = second.x;
            }
            else if (first.y > second.y)
            {
                //edges[edge].minY = roiy[n] * f + pos_y;
                //edges[edge].x = roix[n] * f + pos_x;
                //edges[edge].maxY = roiy[p] * f + pos_y;
                //maxX = roix[p] * f + pos_x;

                this.MinY = second.y;
                this.x = second.x;
                this.x_end = first.x; // ??
                this.MaxY = first.y;
                maxx = first.x;
            }
            else throw new TPCROIException("Edge: Y coordinates cannot be same.");

            this.m = (maxx - x) / (MaxY - MinY);

            if (MaxY < MinY) throw new TPCROIException("Edge: Y-maximum cannot be less than minimum.");
        }
    }
}
