/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Text;
using System.IO;
using TPClib.Common;

namespace TPClib.ROI
{
    /// <summary>
    /// Class that writes/reads ImageTool ROI files. This format is produced by Yait and MunichHeart software
    /// </summary>
    public class ImageToolROIFile : ROIFile
    {
        /// <summary>
        /// Imadeus specific header information.
        /// </summary>
        public ImageToolHeader ImagetoolHeader = new ImageToolHeader();

        /// <summary>
        /// Default constructor
        /// </summary>
        public ImageToolROIFile()
        {
        }
        /// <summary>
        /// Creates an ImageTools ROI file object
        /// </summary>
        public ImageToolROIFile(String filename)
        {
            this.filename = filename;
        }

        /// <summary>
        /// Writes the ROI information to ImageToolROIfile
        /// </summary>
        public override void WriteFile(ref System.Collections.Generic.List<TPClib.ROI.VOI> vois)
        {
            if (filename == null) throw new TPCROIFileException("Filename is null");

            FileStream filestream = new FileStream(filename, FileMode.Create, FileAccess.Write);
            StreamWriter fileWriter = new StreamWriter(filestream, new ASCIIEncoding());

            fileWriter.AutoFlush = true;

            fileWriter.Write(WriteData());

            filestream.Close();
        }


        /// <summary>
        /// Reads ImageTool ROI files.
        /// </summary>
        public override void ReadFile()
        {
            if (!CheckFormat()) 
                throw new TPCROIFileException("ReadFile: The file " + filename + " was not in correct ImageTool ROI format.");

            this.Filetype = FileType.ImageTool;

            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());
            string text = reader.ReadToEnd();
            filestream.Close();
            ReadData(text);
        }

        /// <summary>
        /// Checks if the file is valid ImageTool ROI file.
        /// </summary>
        /// <returns>True if the file is valid</returns>
        public override bool CheckFormat()
        {
            if (filename == null) throw new TPCROIFileException("You have not given filename");

            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());
            string text = reader.ReadToEnd();
            filestream.Close();

            StringReader strReader = new StringReader(text);
            String line = HelpFunctions.ReadNextLine(ref strReader);
            
            // If the file is empty
            if (line == null) return false;
            
            // Imagetool roi file starts always with *
            if (line[0] != '*') return false;

            String[] words = line.Split(separators, StringSplitOptions.RemoveEmptyEntries );
            if (words.Length < 13) return false;
            strReader.Close();

            return true;
        }

        /// <summary>
        /// Returns a String that represents the current Object.
        /// </summary>
        /// <returns>A String that represents the current Object.</returns>
        public override string ToString()
        {
            return "ImageToolROIFile[filename="+filename+"]";
        }

        #region private members

        /// <summary>
        /// Reads ROI data from string
        /// </summary>
        /// <param name="text">string containing ROI data in ImageTool format</param>
        private void ReadData(String text)
        {            
            StringReader sr = new StringReader(text);
            bool end = false;
            int numRegions=0;
            ROIStack stack = new ROIStack();
            while (!end)
            {
                String line = HelpFunctions.ReadNextLine(ref sr);
                if (line != null)
                {
                    if (line[0] != '*') throw new TPCROIFileException("The ImageTool file was not in correct format.");

                    String[] words = line.Split(separators);

                    if (words.Length < 13) throw new TPCROIFileException("The ImageTool file was not in correct format.");

                    ImagetoolHeader.imageFile = words[0].Substring(1);
                    ImagetoolHeader.zoomfactor = HelpFunctions.StrToDouble(words[1]);

                    // unused reconstruction zoom
                    ImagetoolHeader.recZoom = HelpFunctions.StrToDouble(words[2]);

                    ImagetoolHeader.MatrixNumber = Convert.ToInt32(words[3]);
                    ImagetoolHeader.ROItype = (ROI_Type)Convert.ToInt32(words[4]);
                    ImagetoolHeader.OriginX = Convert.ToInt32(words[6]);
                    ImagetoolHeader.OriginY = Convert.ToInt32(words[7]);
                    ImagetoolHeader.OriginZ = new Image.Ecat7File.Ecat7_Matval(ImagetoolHeader.MatrixNumber).plane - 1;

                    ImagetoolHeader.width = Convert.ToInt32(words[8]);
                    ImagetoolHeader.height = Convert.ToInt32(words[9]);
                    ImagetoolHeader.ROInumber = Convert.ToInt32(words[11]);

                    ImagetoolHeader.numPoints = Convert.ToInt32(words[words.Length - 1]);

                    // ROI name can contain spaces
                    String ROIName = "";
                    for (int i = 12; i < words.Length - 1; i++)
                    {
                        if (i != 12) ROIName += " ";
                        ROIName += words[i];
                    }
                    
                    ROI roi;

                    switch (ImagetoolHeader.ROItype)
                    {
                        case ROI_Type.Rectangle:
                            roi = new RectangleROI();
                            (roi as RectangleROI).Width = (double)ImagetoolHeader.width / ImagetoolHeader.zoomfactor;
                            (roi as RectangleROI).Height = (double)ImagetoolHeader.height / ImagetoolHeader.zoomfactor;
                            break;
                        case ROI_Type.Circle:
                            roi = new CircleROI(0,0,0,0.5*(double)ImagetoolHeader.width / ImagetoolHeader.zoomfactor);
                            break;
                        case ROI_Type.Ellipse:
                            roi = new EllipseROI(0, 0, 0, (double)ImagetoolHeader.width / ImagetoolHeader.zoomfactor, (double)ImagetoolHeader.height / ImagetoolHeader.zoomfactor);
                            break;
                        case ROI_Type.Trace:
                            roi = new TraceROI();
                            break;
                        default:
                            throw new TPCROIFileException("The ImageTool file was not in correct format.");
                    }
                    roi.ROIName = ROIName.Substring(0, ROIName.IndexOf("///0"));

                    roi.Location.X = (double)ImagetoolHeader.OriginX / ImagetoolHeader.zoomfactor;
                    roi.Location.Y = (double)ImagetoolHeader.OriginY / ImagetoolHeader.zoomfactor;
                    roi.Location.Z = ImagetoolHeader.OriginZ;
                    roi.Unit = DataUnit.Pixel;
                    
                    // If ROI type is Trace, next line will contain the trace point data.
                    if (ImagetoolHeader.ROItype == TPClib.ROI.ROI_Type.Trace)
                    {
                        line = HelpFunctions.ReadNextLine(ref sr);
                        words = line.Split(separators, StringSplitOptions.RemoveEmptyEntries);

                        // ROI points must be same amount as the file claims
                        if ((words.Length / 2) != ImagetoolHeader.numPoints)
                            throw new TPCROIFileException("The ImageTool file was not in correct format.");

                        for (int i = 0; i < ImagetoolHeader.numPoints; i++)
                        {
                            (roi as TraceROI).AddPoint(
                                (HelpFunctions.StrToDouble(words[i * 2]) / ImagetoolHeader.zoomfactor),
                                (HelpFunctions.StrToDouble(words[i * 2 + 1]) / ImagetoolHeader.zoomfactor)
                                );
                        }
                    }
                    stack.Add(roi);

                    numRegions++;
                }
                else
                { 
                    end = true;
                }
            }

            this.VOIs.Add( stack );
        }

        /// <summary>
        /// Writes current data into string
        /// </summary>
        /// <returns>data in file as string</returns>
        private String WriteData()
        {
            StringBuilder str = new StringBuilder();
            int num = 1;
            // normally always only one
            foreach (VOI voi in VOIs)
            {
                if (!(voi is ROIStack)) throw new TPCROIFileException("Cannot write other than ROI stacks with ImageTool file format.");
                ROIStack stack = (voi as ROIStack);
                foreach( ROI roi in stack )
                {
                    str.Append("*" + ImagetoolHeader.imageFile);
                    str.Append(" " + ImagetoolHeader.zoomfactor.ToString("F6", new System.Globalization.CultureInfo("en-GB")));
                    str.Append(" " + ImagetoolHeader.recZoom.ToString("F6", new System.Globalization.CultureInfo("en-GB"))); // reconstruction zoom
                    Image.Ecat7File.Ecat7_Matval matval = new TPClib.Image.Ecat7File.Ecat7_Matval(ImagetoolHeader.MatrixNumber);
                    str.Append(" " + new Image.Ecat7File.Ecat7_Matval(matval.frame, (int)roi.Location.Z+1, matval.gate, matval.data, matval.bed).matrix_id);

                    str.Append(" " + (int)roi.RoiType);
                    str.Append(" 1" ); // status code
                    str.Append(" " + (int)(roi.Location.X * ImagetoolHeader.zoomfactor) );
                    str.Append(" " + (int)(roi.Location.Y * ImagetoolHeader.zoomfactor));
                    if (roi is IGeometricalROI)
                    {
                        str.Append(" " + (int)((roi as IGeometricalROI).Width * ImagetoolHeader.zoomfactor));
                        str.Append(" " + (int)((roi as IGeometricalROI).Height * ImagetoolHeader.zoomfactor));
                    }
                    else {
                        str.Append(" " + (int)(0));
                        str.Append(" " + (int)(0));                    
                    }
                    str.Append(" 0"); 
                    str.Append(" " + num );
                    str.Append(" " + roi.ROIName+"///0");

                    int npoints = 0;
                    if (roi.RoiType == ROI_Type.Trace) npoints = (roi as TraceROI).Count;
                    str.Append(" " + npoints);

                    str.Append( "\r\n" );

                    if (roi.RoiType == ROI_Type.Trace)
                    {
                        foreach (Point rp in (roi as TraceROI).Points)
                        {
                            str.Append((int)(rp.X * ImagetoolHeader.zoomfactor) + " " + (int)(rp.Y * ImagetoolHeader.zoomfactor) + " ");
                        }
                        str.Append("\r\n");
                    }

                }
                num++;
            }
            return str.ToString();
        }

        /// <summary>
        /// This function reads all the words from next line of the file and returns then as String list.
        /// All possible Comment lines are added to Comments list and empty lines are ignored. Returns null if
        /// file has been read to the end.
        /// </summary>
        /// <param name="reader">Reader object that reads the string.</param>
        /// <returns>List of all words in the next line of reader. Null if the string has been readed to the end.</returns>
        protected String[] ReadWordsFromLine(ref System.IO.StringReader reader)
        {
            string line = "";
            bool correct_line = true;

            // space and tab will divide strings
            char[] separators = new char[] { ' ', '\t' };

            // every loop reads one line from file. Empty lines are ignored.
            do
            {
                line = reader.ReadLine();

                // null line means that the string has been read to the end
                if (line == null) return null;

                line.Trim();

                // All comments are added to Comments list
                if (line.Length <= 1) // empty lines are ignored
                {
                    correct_line = false;
                }
                else correct_line = true;
            }
            while (!correct_line);

            //Console.WriteLine(line);

            return line.Split(separators, StringSplitOptions.RemoveEmptyEntries);
        }

        private char[] separators = new char[] { ' ', '\r', '\n' };

#endregion

        /// <summary>
        /// ImageTool format specific header information
        /// </summary>
        public class ImageToolHeader {
            /// <summary>
            /// Reference image file name that was used when drawing this ROI
            /// </summary>
            public String imageFile;
            /// <summary>
            /// Zooming factor of underlaying image when drawing this ROI
            /// </summary>
            public double zoomfactor;
            /// <summary>
            /// Reconstruction zoom
            /// </summary>
            public double recZoom;
            /// <summary>
            /// Matrix number that has coded ROI information
            /// </summary>
            public int MatrixNumber;
            /// <summary>
            /// ROI type code number
            /// </summary>
            public TPClib.ROI.ROI_Type ROItype;
            /// <summary>
            /// ROI location x-coordinate
            /// </summary>
            public int OriginX;
            /// <summary>
            /// ROI location y-coodrdinate
            /// </summary>
            public int OriginY;
            /// <summary>
            /// ROI location z-coordinate
            /// </summary>
            public int OriginZ;
            /// <summary>
            /// ROI width for geometrical shape ROI
            /// </summary>
            public int width;
            /// <summary>
            /// ROI height for geometrical shape ROI
            /// </summary>
            public int height;
            /// <summary>
            /// ROI number
            /// </summary>
            public int ROInumber;
            /// <summary>
            /// Number of points for trace ROI
            /// </summary>
            public int numPoints;
        }
    }
}
