/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Common;

namespace TPClib.ROI
{
    /// <summary>
    /// Region Of Interest (ROI), which is defined by a 2D boundary.
    /// </summary>
    public abstract class ROI: Maskable, System.ICloneable
    {
        /// <summary>
        /// Length unit of ROI vertex coordinates.
        /// </summary>
        public DataUnit Unit = DataUnit.Unknown;
        /// <summary>
        /// ROI name
        /// </summary>
        public String ROIName;
        /// <summary>
        /// Type of ROI definition
        /// </summary>
        public readonly ROI_Type RoiType;

        /// <summary>
        /// Reference location of ROI. All ROI coordinates are related to this location
        /// </summary>
        public Point Location = new Point();
        /// <summary>
        /// Default constructor
        /// </summary>
        public ROI() : base()
        {
            this.ROIName = ".";
        }
        /// <summary>
        /// Constructor with values
        /// </summary>
        /// <param name="roiType">roi type</param>
        public ROI(ROI_Type roiType)
            : base()
        {
            this.RoiType = roiType;
        }
        /// <summary>
        /// Constructor with values
        /// </summary>
        /// <param name="name">ROI name</param>
        /// <param name="roiType"></param>
        /// <param name="x">ROI reference point location x-coordinate</param>
        /// <param name="y">ROI reference point location y-coordinate</param>
        /// <param name="z">ROI reference point location z-coordinate</param>
        public ROI(String name, ROI_Type roiType,
                    Double x, Double y, Double z) : base()
        {
            ROIName = name;
            this.RoiType = roiType;
        }

        /// <summary>
        /// Returns a String that represents the current Object.
        /// </summary>
        /// <returns>A String that represents the current Object.</returns>
        public override string ToString()
        {
            StringBuilder str = new StringBuilder();
            str.Append("ROI[name=");
            str.Append(this.ROIName);
            str.Append(" type=");
            str.Append(this.RoiType);
            str.Append(" loc=");
            str.Append(this.Location);
            str.Append(" unit=");
            str.Append(this.Unit);
            str.Append("]");
            return str.ToString();
        }
        /// <summary>
        /// Creates deep copy of this object
        /// </summary>
        /// <returns>cloned object</returns>
        public abstract Object Clone();

        /// <summary>
        /// Coordinates surrounding 3x3x3 cube
        /// </summary>
        private static readonly int[,] surroundings = new int[,] {
                                                                {-1, 1, 0},   { 1, 0, 0},   {-1,-1, 0}, 
                                                                { 0, 1, 0},   { 1,-1, 0},   {-1, 0, 0}, 
                                                                { 1, 1, 0},   { 0,-1, 0}, //{ 0, 0, 0},
                                                                { 0,-1,-1},   { 0, 0,-1},   { 0, 1,-1}, 
                                                                {-1,-1,-1},   {-1, 0,-1},   {-1, 1,-1}, 
                                                                { 1,-1,-1},   { 1, 0,-1},   { 1, 1,-1}, 

                                                                { 0,-1, 1},   { 0, 0, 1},   { 0, 1, 1}, 
                                                                {-1,-1, 1},   {-1, 0, 1},   {-1, 1, 1}, 
                                                                { 1,-1, 1},   { 1, 0, 1},   { 1, 1, 1}
                                                          };


        /// <summary>
        /// Inversion Boundary Filling Algorithm. The mask will have 1's inside region. 
        /// The region should not have holes in it.
        /// </summary>
        /// <param name="mask">mask image having 1's at edge pixels</param>
        /// <param name="plane_i">plane index</param>
        public void BoundaryFillRegionGrow(ref TPClib.Image.MaskImage mask, int plane_i) {
            //queue for growing points
            Queue<int[]> p = new Queue<int[]>(1024);
            int[] p_C = new int[2];
            int[] p_Ng = new int[2];
            int planeposition = mask.Planelength*plane_i;
            int si = 0;
            System.Collections.BitArray used = new System.Collections.BitArray(mask.DataLength);
            used.SetAll(false);
            //go through all possible starting points at image borders
            for (int y = 0; y < mask.DimY; y++)
            {
                for (int x = 0; x < mask.DimX; x++)
                {
                    //find next starting point outside polygon
                    if((x > 0 && x < mask.DimX-1) && (y > 0 && y < mask.DimY-1)) 
                        continue;
                    if (used[planeposition + y * mask.DimX + x]) 
                        continue;
                    if (mask[planeposition + y * mask.DimX + x] != 0) 
                        continue;
                    //start region grow from starting point
                    used[planeposition + y * mask.DimX + x] = true;
                    p.Enqueue(new int[] { x, y });
                    //region grow while there are unused points in queue
                    while (p.Count != 0)
                    {
                        //take next point from unused points
                        p_C = p.Dequeue();
                        for (si = 0; si < 8; si++)
                        {
                            p_Ng[0] = p_C[0] + surroundings[si, 0];
                            p_Ng[1] = p_C[1] + surroundings[si, 1];
                            if (p_Ng[0] < 0) continue;
                            if (p_Ng[1] < 0) continue;
                            if (p_Ng[0] >= mask.DimX) continue;
                            if (p_Ng[1] >= mask.DimY) continue;
                            //break if polygon border was reached  
                            if (mask[planeposition + p_Ng[1] * mask.DimX + p_Ng[0]] == 1) break;
                        }
                        //do not grow if region border was found among neighbours
                        if (si < 8)
                            continue;
                        for (si = 0; si < 26; si++)
                        {
                            p_Ng[0] = p_C[0] + surroundings[si, 0];
                            p_Ng[1] = p_C[1] + surroundings[si, 1];
                            if (p_Ng[0] < 0) continue;
                            if (p_Ng[1] < 0) continue;
                            if (p_Ng[0] >= mask.DimX) continue;
                            if (p_Ng[1] >= mask.DimY) continue;
                            //continue if neighbour is already used
                            if (!used[planeposition + p_Ng[1] * mask.DimX + p_Ng[0]])
                            {
                                used[planeposition + p_Ng[1] * mask.DimX + p_Ng[0]] = true;
                                p.Enqueue(new int[] { p_Ng[0], p_Ng[1] });
                            } 
                        }
                    }
                }
            }
            //label masked region
            for (int i = 0; i < mask.DataLength; i++) {
                if (!used[i]) mask[i] = 1;
            }
        }
    }

    /// <summary>
    /// Type of ROI shape. (rectangle, circle, ellipse, trace)
    /// </summary>
    public enum ROI_Type
    {
        /// <summary>
        /// Rectangle ROI
        /// </summary>
        Rectangle = 0,
        /// <summary>
        /// Circle ROI
        /// </summary>
        Circle = 1,
        /// <summary>
        /// Ellipse ROI
        /// </summary>
        Ellipse = 2,
        /// <summary>
        /// Trace ROI, consisting of series of points
        /// </summary>
        Trace = 3
    }
    /// <summary>
    /// Common interface for geometrically defined 2D boundaries.
    /// </summary>
    public interface IGeometricalROI {
        /// <summary>
        /// Width of geometrical 2D shape. Note that the orientation may affect actual ROI presentation.
        /// </summary>
        double Width { get; }
        /// <summary>
        /// Height of geometrical 2D shape. Note that the orientation may affect actual ROI presentation.
        /// </summary>
        double Height { get; }
    }
}
