/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Image;

namespace TPClib.ROI
{
    /// <summary>
    /// Calculates covariance from pixel values
    /// </summary>
    public class ROICalculator_Covariance : ROICalculator
    {
        private double[] sum;
        private double[] sqrsum;
        private int num;

        /// <summary>
        /// Constructs min calculator object with length of 1. All other frames are ignored
        /// </summary>
        public ROICalculator_Covariance()
        {
            SetLength(1);
        }
        /// <summary>
        /// Constructor that defines number of ROIs.
        /// </summary>
        /// <param name="length">number of ROIs</param>
        public ROICalculator_Covariance(int length)
        {
            SetLength(length);
        }
        /// <summary>
        /// Sets number of ROIs in calculations
        /// </summary>
        /// <param name="length">new number of ROIs</param>
        public override void SetLength(int length)
        {
            Length = length;
            sum = new double[Length];
            sqrsum = new double[Length];
            num = 0;
        }
        /// <summary>
        /// Adds values to all ROIs.
        /// </summary>
        /// <param name="value">added values</param>
        public override void AddValue(float[] value)
        {
            for (int i = 0; i < Length; i++)
            {
                sum[i] += value[i];
                sqrsum[i] += value[i] * value[i];
            }
            num++;
        }
        /// <summary>
        /// Gets all calculation results
        /// </summary>
        /// <returns>results as double values</returns>
        public override Double[] GetResult()
        {
            double[] res = new double[Length];
            for (int i = 0; i < Length; i++)
            {
                res[i] = sum[i] / (double)num;
                if (res[i] < 0.0d) res[i] = 0.0d;
                double div = (sqrsum[i] / (double)num) - (res[i] * res[i]);
                if (div > 0.0d) res[i] = Math.Sqrt(div);
            }
            return res;
        }

    }

}
