/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.ROI;
using TPClib;
using System.IO;

namespace TPClib.ROI
{
    /// <summary>
    /// Class representing ROI file.
    /// </summary>
    public abstract class ROIFile: TPCFile
    {
        /// <summary>
        /// Supported ROI file types
        /// </summary>
        public enum FileType
        {
            /// <summary>
            /// Vinci software ROI format. Max-Planck-Institut f�r neurologische Forschung, K�ln (Cologne, Germany).
            /// </summary>
            Vinci,
            /// <summary>
            /// Imadeus software ROI format. Forima.
            /// </summary>
            Imadeus,
            /// <summary>
            /// Dicom-RT Struct Set
            /// </summary>
            DICOM_RTStruct,
            /// <summary>
            /// ImageTool ROI format. ECAT.
            /// </summary>
            ImageTool,
            /// <summary>
            /// Indicates that ROI format is unknown.
            /// </summary>
            Unknown
        }

        /// <summary>
        /// Gets list of supported ROI formats
        /// </summary>
        /// <returns>List of supported ROI formats</returns>
        public static List<FileType> GetSupportedFormats()
        {
            List<FileType> result = new List<FileType>();
            result.Add(FileType.ImageTool);
            result.Add(FileType.Imadeus);
            result.Add(FileType.DICOM_RTStruct);
            return result;
        }

        /// <summary>
        /// VOI and ROI data of file.
        /// </summary>
        public List<VOI> VOIs = new List<VOI>();
        /// <summary>
        /// ROIfile format type. default == FileType.Unknown
        /// </summary>
        public FileType Filetype = FileType.Unknown;

        /// <summary>
        /// Constructs ROI file.
        /// </summary>
        public ROIFile() 
        {
            this.filename = null;
        }

        /// <summary>
        /// Constructs ROI file.
        /// </summary>
        /// <param name="filename">full path to file.</param>
        public ROIFile(string filename)
        {
            this.filename = filename;
        }

        /// <summary>
        /// Implementation of TPCFile interface. Writes this file.
        /// </summary>
        public override void WriteFile()
        {
            WriteFile(ref VOIs);
        }

        /// <summary>
        /// Writes a file.
        /// </summary>
        /// <param name="vois">VOIs that are written</param>
        public abstract void WriteFile(ref List<VOI> vois);

        /// <summary>
        /// Writes the ROI information to ImageToolROIfile
        /// </summary>
        /// <param name="fileName">Name of file</param>
        public void WriteFile(String fileName)
        {
            this.filename = fileName;
            WriteFile();
        }
        /// <summary>
        /// Writes image file. Note that actual image file type is read from 
        /// image's filetype field.
        /// </summary>
        /// <param name="roifile">roi data that is written</param>
        /// <exception cref="Exception">if writing of file failed</exception>
        public static void WriteFile(ROIFile roifile)
        {
            //write image tacs depending on tacs type
            switch (roifile.Filetype)
            {
                case FileType.DICOM_RTStruct:
                    //use input object straight if possible
                    if (roifile is RTStructROIFile)
                    {
                        (roifile as RTStructROIFile).WriteFile();
                    }
                    else
                    {
                        RTStructROIFile file = new RTStructROIFile(roifile.filename);
                        file.VOIs = roifile.VOIs;
                        file.WriteFile();
                    }
                    break;
                default:
                    throw new TPCException("Cannot write ROIs with this method since type " + roifile.Filetype + " is not currently supported.");
            }
        }
        /// <summary>
        /// Reads ROI file from defined file
        /// <seealso cref="TPCFile.filename"/>
        /// </summary>
        /// <param name="filename">full path to read file</param>
        public static ROIFile ReadFile(string filename)
        {
            FileType filetype;
            try
            {
                filetype = ROIFile.ResolveFormat(filename);
            }
            catch (Exception)
            {
                throw new TPCException("Failed to recognize file " + filename);
            }
            switch (filetype)
            {
                case FileType.DICOM_RTStruct:
                    RTStructROIFile rtstructfile = new RTStructROIFile(filename);
                    rtstructfile.ReadFile();
                    return rtstructfile;
                case FileType.Imadeus:
                    ImadeusROIFile imadeusfile = new ImadeusROIFile(filename);
                    imadeusfile.ReadFile();
                    return imadeusfile;
                case FileType.ImageTool:
                    ImageToolROIFile imagetoolfile = new ImageToolROIFile(filename);
                    imagetoolfile.ReadFile();
                    return imagetoolfile;
                default:
                    throw new TPCException("Cannot read ROIs with this method since type " + filetype + " is not currently supported.");
            }
        }

        /// <summary>
        /// Checks the given file format. If the file is in some supported format, true will
        /// be returned
        /// </summary>
        /// <returns>True, if the given file is in some supported format.</returns>
        public virtual bool CheckFormat()
        {
            if (ROIFile.ResolveFormat( filename ) == FileType.Unknown) return false;
            else return true;
        }

        /// <summary>
        /// Checks the given file format and returns the format if it is supported.
        /// </summary>
        /// <returns>Format of the given file</returns>
        public static FileType ResolveFormat( String filename )
        {
            if (filename == null) throw new TPCROIFileException("You have not given filename");
            if (!File.Exists(filename)) return FileType.Unknown;

            FileType result = FileType.Unknown;

            // Next we have to find out the file type 
            ROIFile file;
            if (result == FileType.Unknown)
            {
                file = new RTStructROIFile(filename);
                if (file.CheckFormat()) { result = FileType.DICOM_RTStruct; }
            }
            if (result == FileType.Unknown)
            {
                file = new ImadeusROIFile(filename);
                if (file.CheckFormat()) { result = FileType.Imadeus; }
            }
            if (result == FileType.Unknown)
            {
                file = new ImageToolROIFile(filename);
                if (file.CheckFormat()) { result = FileType.ImageTool; }
            }
            if (result != FileType.Unknown)
            {
                return result;
            }

            return FileType.Unknown;
        }
    }
}
