/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.ROI
{
    /// <summary>
    /// VOI consisting of stack of ROIs. 
    /// </summary>
    public class ROIStack: VOI
    {
        /// <summary>
        /// List of ROIs in stack
        /// </summary>
        protected List<ROI> ROIs = new List<ROI>();
        /// <summary>
        /// Gets the number of ROIs in this stack.
        /// </summary>
        public int Count {
            get { return ROIs.Count; }
        }
        /// <summary>
        /// Adds new ROI to VOI.
        /// </summary>
        /// <param name="roi">ROI to add</param>
        public void Add(ROI roi)
        {
            ROIs.Add(roi);
        }
        /// <summary>
        /// Replaces one ROI by another
        /// </summary>
        /// <param name="i">zero-based number of ROI to replace</param>
        /// <param name="roi">ROI which will replace the ROI at given index</param>
        public void SetRoi(int i, ROI roi)
        {
            if (i < 0 || i >= ROIs.Count) throw new TPCROIException("Remove: Index out of bounds.");
            ROIs.Insert(i, roi);
        }
        /// <summary>
        /// Gets one ROI from VOI
        /// </summary>
        /// <param name="ROInum">zero-based number of ROI</param>
        /// <returns>The ROI with given index</returns>
        public ROI GetRoi(int ROInum)
        {
            if (ROInum < 0 || ROInum >= ROIs.Count) throw new TPCROIException("Remove: Index out of bounds.");
            return ROIs[ROInum];
        }
        /// <summary>
        /// Removes one ROI from VOI
        /// </summary>
        /// <param name="ROInum">zero based ROI number</param>
        public void Remove(int ROInum)
        {
            if (ROInum < 0 || ROInum >= ROIs.Count) throw new TPCROIException("Remove: Index out of bounds.");
            ROIs.RemoveAt(ROInum);
        }
        /// <summary>
        /// Index operator over stacked rois
        /// </summary>
        /// <param name="i">index of ROI</param>
        /// <returns>ROI at index i</returns>
        public ROI this[int i] {
            get
            {
                return ROIs[i];
            }
            set {
                ROIs[i] = (ROI)value.Clone();
            }
        }
        /// <summary>
        /// Iterator over ROIs.
        /// </summary>
        /// <returns>Iterator object</returns>
        public IEnumerator<ROI> GetEnumerator()
        {
            for (int i = 0; i < this.ROIs.Count; i++)
            {
                yield return ROIs[i];
            }
        }
        /// <summary>
        /// Returns the rois which are on given z slice.
        /// </summary>
        /// <param name="z">coordinate of slice</param>
        /// <returns>An array of ROIs which belong to the slice.</returns>
        public virtual ROI[] GetRois(int z)
        {
            List<ROI> return_list = new List<ROI>();

            foreach (ROI r in ROIs)
            {
                if (r.Location.Z == z) 
                    return_list.Add(r);
            }
            return return_list.ToArray();
        }
        /// <summary>
        /// Catenates another ROI stact to this VOI
        /// </summary>
        /// <param name="stack">catenated ROI stack</param>
        public void Catenate(ROIStack stack)
        {
            if (stack == null) throw new TPCROIException("Catenate: Cannot catenate null VOI.");
            ROIs.AddRange(stack.ROIs);
        }
        /// <summary>
        /// Fills maskable item with 1's into given size mask image.
        /// </summary>
        /// <param name="mask">Mask to fill</param>
		/// <param name="method">Fill method</param>
		public override void Fill(ref TPClib.Image.MaskImage mask, Fill_Method method)
        {
            foreach(ROI r in ROIs) {
                r.Fill(ref mask, method);
            }
        }
        /// <summary>
        /// Gets bounding box of stack of ROIs
        /// </summary>
        /// <returns>bounds of this VOI</returns>
        public override Limits GetBoundingBox()
        {
            Limits bounds = new Limits(new double[] { double.MaxValue, double.MaxValue, double.MaxValue },
                                       new double[] { double.MinValue, double.MinValue, double.MinValue });
            Limits ROI_bounds;
            foreach (ROI r in ROIs)
            {
                ROI_bounds = r.GetBoundingBox();
                //update bounding box values
                if (ROI_bounds.GetLimit(0, Limits.Limit.LOW) < bounds.GetLimit(0, Limits.Limit.LOW))
                    bounds.SetLimit(0, Limits.Limit.LOW, ROI_bounds.GetLimit(0, Limits.Limit.LOW));
                if (ROI_bounds.GetLimit(1, Limits.Limit.LOW) < bounds.GetLimit(1, Limits.Limit.LOW))
                    bounds.SetLimit(1, Limits.Limit.LOW, ROI_bounds.GetLimit(1, Limits.Limit.LOW));
                if (r.Location[2] < bounds.GetLimit(2, Limits.Limit.LOW)) bounds.SetLimit(2, Limits.Limit.LOW, r.Location[2]);
                if (ROI_bounds.GetLimit(0, Limits.Limit.HIGH) > bounds.GetLimit(0, Limits.Limit.HIGH))
                    bounds.SetLimit(0, Limits.Limit.HIGH, ROI_bounds.GetLimit(0, Limits.Limit.HIGH));
                if (ROI_bounds.GetLimit(1, Limits.Limit.HIGH) > bounds.GetLimit(1, Limits.Limit.HIGH))
                    bounds.SetLimit(1, Limits.Limit.HIGH, ROI_bounds.GetLimit(1, Limits.Limit.HIGH));
                if (r.Location[2] > bounds.GetLimit(2, Limits.Limit.HIGH)) bounds.SetLimit(2, Limits.Limit.HIGH, r.Location[2]);
            }
            return bounds;
        }
    }
}
