/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using TPClib.Image;
using TPCDicom.DataStructure;
using TPCDicom.DataStructure.DataSet;
using TPCDicom.Registry;
using TPClib.Common;

namespace TPClib.ROI 
{
    /// <summary>
    /// DICOM-RT Struct Set decoding program.
    /// Input argument is a valid DICOM-RT Struct set file.
    /// Output is an array of TPClib.ROI.VOI. This VOI coordinate is in unit of mm, and centered to (0,0,0) coordinate.
    /// </summary>
    public class RTStructROIFile: ROIFile
    {
        /// <summary>
        /// Event that is sent while I/O task proceeds
        /// </summary>
        protected IOProcessEventHandler pevent = null;

        /// <summary>
        /// Event that is sent when reading of file has progressed.
        /// </summary>
        public static event IOProcessEventHandler IOProgress;

        /// <summary>
        /// RTSTRUCT related header information
        /// </summary>
        public RTROIHeader Header = new RTROIHeader();

        /// <summary>
        /// Construct RTStructReader. Argument is valid file name, which is DICOM-RT struct set file.
        /// </summary>
        /// <param name="filename">DICOM filename for ROI file</param>
        public RTStructROIFile(string filename): base(filename)
        {
            Filetype = FileType.DICOM_RTStruct;
        }

        /// <summary>
        /// Checks if the file is valid Imadeus ROI file.
        /// </summary>
        /// <returns>True if the file is valid</returns>
        public override bool CheckFormat()
        {
            if (filename == null) throw new TPCROIFileException("You have not given filename");

            try
            {
                if (TPCDicom.File.DicomFile.IsDicomFile(filename))
                    return true;
                else
                    return false;
            }
            catch (Exception)
            {
                return false;
            }
        }

        /// <summary>
        /// Writes a file.
        /// </summary>
        /// <param name="vois">VOIs that are written</param>
        public override void WriteFile(ref List<VOI> vois)
        {
            try
            {
                FileStream stream = new FileStream(filename, FileMode.Create);
                Write0002(stream);
                WriteROIHeader(stream);
                WriteReferencedFrameOfReferenceSequence(stream);
                WriteStructureSetROISequence(stream);
                WriteROIContourSequence(stream);
                WriteRTROIObservationSequence(stream);
                stream.Close();
            }
            catch (Exception dicomFileException)
            {
                throw new TPCROIFileException("Problems processing RTSTRUCT DICOM file:\n" + dicomFileException);
            }
        }
        /// <summary>
        /// Returns a String that represents the current Object.
        /// </summary>
        /// <returns>A String that represents the current Object.</returns>
        public override string ToString()
        {
            return "RTStructROIFile[filename="+filename+" "+VOIs+"]";
        }

        /// <summary>
        /// Writes group 0x0002 of DICOM header
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected void Write0002(Stream stream)
        {
            byte[] buf;
            long length_pos = 0;
            long start_pos = 0;

            // write the empty preamable 
            buf = new byte[128];
            stream.Write(buf, 0, buf.Length);

            // write the signature 
            buf = TPClib.Image.DicomFile.ToBytesArray("DICM".ToCharArray());
            stream.Write(buf, 0, 4);

            // group 0x0002 
            TPClib.Image.DicomFile.WriteElement(stream, 0x0002, 0x0000, new byte[] { 0x00, 0x00, 0x00, 0x00 });
            length_pos = stream.Position - 4;
            start_pos = stream.Position;

            buf = new byte[] { 0x00, 0x01 };
            TPClib.Image.DicomFile.WriteElement(stream, 0x0002, 0x0001, buf);
            //MediaStorageSOPClassUID
            buf = TPClib.Image.DicomFile.ToBytesArray("1.2.840.10008.5.1.4.1.1.481.3".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0002, 0x0002, buf);
            //MediaStorageSOPInstanceUID, often same as file basename
            buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.276.0.20.1.4.4.92.4068.1252498973.60").ToCharArray());
            //buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.276.0.20.1.4.4." + (Math.Abs(header.PatientID.GetHashCode()) % 100) + ".4068.1252498973." + (Math.Abs(header.InstanceCreationTime.GetHashCode()) % 100)).ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0002, 0x0003, buf);

            // transfer syntax: Explicit VR Little Endian
            buf = TPClib.Image.DicomFile.ToBytesArray("1.2.840.10008.1.2.1".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0002, 0x0010, buf);
            //ImplementationClassUID
            buf = TPClib.Image.DicomFile.ToBytesArray("1.2.276.0.20.1.1.4.1.5.0.4".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0002, 0x0012, buf);
            //ImplementationVersionName
            buf = TPClib.Image.DicomFile.ToBytesArray("".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0002, 0x0013, buf);
            //SourceApplicationEntityTitle
            buf = TPClib.Image.DicomFile.ToBytesArray("".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0002, 0x0016, buf);

            TPClib.Image.DicomFile.WriteLengthToStream(stream, length_pos, (uint)(stream.Position - start_pos));
        }

        /// <summary>
        /// Writes ROI header tags
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected void WriteROIHeader(Stream stream)
        {
            string str;
            byte[] buf; 
            //(0008,0005) CS [ISO_IR 100]                             #  10, 1 SpecificCharacterSet
            buf = TPClib.Image.DicomFile.ToBytesArray(string.Format("ISO_IR 100").ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0005, buf);
            //(0008,0012) DA [20090909]                               #   8, 1 InstanceCreationDate
            str = Header.InstanceCreationTime.ToString("yyyyMMdd");
            buf = TPClib.Image.DicomFile.ToBytesArray(str.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0012, buf);
            //(0008,0013) TM [152253.000000]                          #  14, 1 InstanceCreationTime
            str = Header.InstanceCreationTime.ToString("HHmmss.ffffff") + " ";
            buf = TPClib.Image.DicomFile.ToBytesArray(str.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0013, buf);
            //(0008,0016) UI =RTStructureSetStorage                   #  30, 1 SOPClassUID
            buf = TPClib.Image.DicomFile.ToBytesArray(string.Format("1.2.840.10008.5.1.4.1.1.481.3").ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0016, buf);
            //(0008,0018) UI [1.2.276.0.20.1.4.4.964242475771.4068.1252498973.921875] #  54, 1 SOPInstanceUID
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.SOPInstanceUID.ToString().ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0018, buf);
            //(0008,0020) DA [20090909]                               #   8, 1 StudyDate
            str = Header.StudyTime.ToString("yyyyMMdd");
            buf = TPClib.Image.DicomFile.ToBytesArray(str.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0020, buf);
            //(0008,0030) TM (no value available)                     #   0, 0 StudyTime
            str = Header.StudyTime.ToString("HHmmss.ffffff") + " ";
            buf = TPClib.Image.DicomFile.ToBytesArray(str.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0030, buf);
            //(0008,0050) SH (no value available)                     #   0, 0 AccessionNumber
            buf = TPClib.Image.DicomFile.ToBytesArray("0".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0050, buf);
            //(0008,0060) CS [RTSTRUCT]                               #   8, 1 Modality
            buf = TPClib.Image.DicomFile.ToBytesArray(string.Format("RTSTRUCT").ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0060, buf);
            //(0008,0070) LO [BrainLAB]                               #   8, 1 Manufacturer
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.Manufacturer.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0070, buf);
            //(0008,0090) PN (no value available)                     #   0, 0 ReferringPhysiciansName
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.ReferringPhysiciansName.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x0090, buf);
            //(0008,1010) SH [IPLANRT1]                               #   8, 1 StationName
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.StationName.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x1010, buf);
            //(0008,1030) LO [d2890]                                  #   6, 1 StudyDescription
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.StudyDescription.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x1030, buf);
            //(0008,103e) LO [RT Struct Series]                       #  16, 1 SeriesDescription
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.SeriesDescription.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x103e, buf);
            //(0008,1090) LO [PatXferRT]                              #  10, 1 ManufacturersModelName
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.ManufacturersModelName.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0008, 0x1090, buf);
            //(0009,0010) LO [BrainLAB_Conversion]                    #  20, 1 PrivateCreator
            buf = TPClib.Image.DicomFile.ToBytesArray("".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0009, 0x0010, buf);
            //(0009,1001) LO [Varian Vision 7.3.10 SP3]               #  24, 1 Unknown Tag & Data
            buf = TPClib.Image.DicomFile.ToBytesArray("".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0009, 0x1001, buf);
            //(0010,0010) PN [AAAAA^BBBBB^CCCCCCCCCCC]                #  24, 1 PatientsName
            buf = TPClib.Image.DicomFile.ToBytesArray((Header.PatientsName + " ").ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0010, 0x0010, buf);
            //(0010,0020) LO [211271-1402]                            #  12, 1 PatientID
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.PatientID.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0010, 0x0020, buf);
            //(0010,0030) DA [19711221]                               #   8, 1 PatientsBirthDate
            str = Header.PatientsBirthDate.ToString("yyyyMMdd");
            buf = TPClib.Image.DicomFile.ToBytesArray(str.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0010, 0x0030, buf);
            //(0010,0040) CS [F]                                      #   2, 1 PatientsSex
            switch (Header.PatientsSex)
            {
                case ImageHeader.Patient_sex_e._FEMALE: str = "F"; break;
                case ImageHeader.Patient_sex_e.SEX_MALE: str = "M"; break;
                default: str = "U"; break;
            }
            buf = TPClib.Image.DicomFile.ToBytesArray(string.Format(str).ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0010, 0x0040, buf);
            //(0010,1020) DS [1.88]                                   #   4, 1 PatientsSize
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.PatientsSize.ToString("0.00").Replace(',', '.').ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0010, 0x1020, buf);
            //(0010,1030) DS [86]                                     #   2, 1 PatientsWeight
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.PatientsWeight.ToString("0.00").Replace(',', '.').ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0010, 0x1030, buf);
            //(0018,1020) LO [1.5.0 Build 118064]                     #  18, 1 SoftwareVersions
            if (Header.Contains("SoftwareVersions"))
                buf = TPClib.Image.DicomFile.ToBytesArray(Header["SoftwareVersions"].ToString().ToCharArray());
            else
                buf = TPClib.Image.DicomFile.ToBytesArray("".ToString().ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0018, 0x1020, buf);
            //(0020,000d) UI [1.2.276.0.20.1.2.4.964242475771.4068.1252498939.93750] #  54, 1 StudyInstanceUID
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.StudyInstanceUID.ToString().ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0020, 0x000d, buf);
            //(0020,000e) UI [1.2.276.0.20.1.3.4.964242475771.4068.1252498973.906250] #  54, 1 SeriesInstanceUID
            buf = TPClib.Image.DicomFile.ToBytesArray(Header.SeriesInstanceUID.ToString().ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0020, 0x000e, buf);
            //(0020,0010) SH [3634049, 4896]                          #  14, 1 StudyID
            buf = TPClib.Image.DicomFile.ToBytesArray((Header.StudyID + " ").ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0020, 0x0010, buf);
            //(0020,0011) IS [1]                                      #   2, 1 SeriesNumber
            buf = TPClib.Image.DicomFile.ToBytesArray("1".ToString().ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x0020, 0x0011, buf);
            //(0020,0013) IS [1]                                      #   2, 1 InstanceNumber
            TPClib.Image.DicomFile.WriteElement(stream, 0x0020, 0x0013, buf);
            //(3006,0002) SH [1]                                      #   2, 1 StructureSetLabel
            buf = TPClib.Image.DicomFile.ToBytesArray("1".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x3006, 0x0002, buf);
            //(3006,0004) LO [2]                                      #   2, 1 StructureSetName
            buf = TPClib.Image.DicomFile.ToBytesArray("2".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x3006, 0x0004, buf);
            //(3006,0006) ST [Exported by 'iPlan RT Image 3.0.1 build 111921(RELEASE)' to PatXfer RT.] #  72, 1 StructureSetDescription
            buf = TPClib.Image.DicomFile.ToBytesArray("".ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x3006, 0x0006, buf);
            //(3006,0008) DA [20090909]                               #   8, 1 StructureSetDate
            str = Header.StructureSetCreationTime.ToString("yyyyMMdd");
            buf = TPClib.Image.DicomFile.ToBytesArray(str.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x3006, 0x0008, buf);
            //(3006,0009) TM [152131.000000]                          #  14, 1 StructureSetTime
            str = Header.StructureSetCreationTime.ToString("HHmmss.ffffff") + " ";
            buf = TPClib.Image.DicomFile.ToBytesArray(str.ToCharArray());
            TPClib.Image.DicomFile.WriteElement(stream, 0x3006, 0x0009, buf);
        }

        /// <summary>
        /// Writes ReferencedFrameOfReferenceSequence starting from tag(3006,0010) 
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected void WriteReferencedFrameOfReferenceSequence(Stream stream)
        {
            byte[] buf;
            TPClib.Image.DicomFile.SequenceWriteData[] sequence = new TPClib.Image.DicomFile.SequenceWriteData[4];
            TPClib.Image.DicomFile.NestedDataSetWriteData[] nested = new TPClib.Image.DicomFile.NestedDataSetWriteData[4];
            int No_rois = GetTotalNumberOfROIs();

            //ReferencedFrameOfReferenceSequence
            sequence[0] = new TPClib.Image.DicomFile.SequenceWriteData();
            sequence[0].SetTag(0x3006, 0x0010);
                nested[0] = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.276.0.20.1.4.4.0.4068.1252498973.0").ToCharArray());
                nested[0].Add(TPClib.Image.DicomFile.CreateDataElement(0x0020, 0x0052, buf));
                //RTReferencedStudySequence
                sequence[1] = new TPClib.Image.DicomFile.SequenceWriteData();
                sequence[1].SetTag(0x3006, 0x0012);
                    nested[1] = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                    buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.840.10008.3.1.2.5.5").ToCharArray());
                    nested[1].Add(TPClib.Image.DicomFile.CreateDataElement(0x0008, 0x1150, buf));
                    buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.276.0.20.1.2.4.0.4068.1252498973.0").ToCharArray());
                    nested[1].Add(TPClib.Image.DicomFile.CreateDataElement(0x0008, 0x1155, buf));
                    //RTReferencedSeriesSequence
                    sequence[2] = new TPClib.Image.DicomFile.SequenceWriteData();
                    sequence[2].SetTag(0x3006, 0x0014);
                        nested[2] = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                        buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.276.0.20.1.4.4.0.4068.1252498973.0").ToCharArray());
                        nested[2].Add(TPClib.Image.DicomFile.CreateDataElement(0x0020, 0x000e, buf));
                        //ContourImageSequence information for reference image data (image where ROI was drawn)
                        sequence[3] = new TPClib.Image.DicomFile.SequenceWriteData();
                        sequence[3].SetTag(0x3006, 0x0016);
                        for (int i = 0; i < VOIs.Count; i++)
                        {
                            nested[3] = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                            buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.840.10008.5.1.4.1.1.2").ToCharArray());
                            nested[3].Add(TPClib.Image.DicomFile.CreateDataElement(0x0008, 0x1150, buf));
                            buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.276.0.20.1.4.4.0.4068.1252498973."+i).ToCharArray());
                            nested[3].Add(TPClib.Image.DicomFile.CreateDataElement(0x0008, 0x1155, buf));
                            sequence[3].Add(nested[3]);                  
                        }
                        nested[2].Add(sequence[3]);
                    sequence[2].Add(nested[2]);
                    nested[1].Add(sequence[2]);
                sequence[1].Add(nested[1]);
                nested[0].Add(sequence[1]);
            sequence[0].Add(nested[0]);
            TPClib.Image.DicomFile.WriteSequenceRecursive(stream, sequence[0]);
        }

        /// <summary>
        /// Writes ROI 
        /// </summary>
        /// <param name="stream">writable stream</param>
        protected void WriteStructureSetROISequence(Stream stream)
        {
            byte[] buf;
            TPClib.Image.DicomFile.SequenceWriteData sequence;
            TPClib.Image.DicomFile.NestedDataSetWriteData nested;

            sequence = new TPClib.Image.DicomFile.SequenceWriteData();
            sequence.SetTag(0x3006, 0x0020);
            for (int i = 0; i < VOIs.Count; i++)
            {
                nested = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                //ROINumber
                buf = TPClib.Image.DicomFile.ToBytesArray((i.ToString()).ToCharArray());
                nested.Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0022, buf));
                //ReferencedFrameOfReferenceUID
                buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.276.0.20.1.30.4.0.4068.0.0").ToCharArray());
                nested.Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0024, buf));
                //ROIName
                buf = TPClib.Image.DicomFile.ToBytesArray((VOIs[i].Name).ToCharArray());
                nested.Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0026, buf));
                //ROIGenerationAlgorithm
                buf = TPClib.Image.DicomFile.ToBytesArray(("unknown").ToCharArray());
                nested.Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0036, buf));
                sequence.Add(nested);
            }
            TPClib.Image.DicomFile.WriteSequenceRecursive(stream, sequence);

        }

        /// <summary>
        /// Reads ROI header tags until (3006,0010)
        /// </summary>
        /// <param name="sequence">read sequence</param>
        /// <param name="di">data element index</param>
        protected void ReadROIHeader(ref Sequence sequence, ref int di)
        {
            //initialize time fields
            Header.StudyTime = new System.DateTime(0);
            Header.PatientsBirthDate = new System.DateTime(0);
            Header.InstanceCreationTime = new System.DateTime(0);
            Header.StructureSetCreationTime = new System.DateTime(0);
            //read until ReferencedFrameOfReferenceSequence
            while (!sequence[di].Tag.Equals("(3006,0010)") && di < sequence.Count)
            {
                //add field into general header
                try
                {
                    if (sequence[di] is DataElement && sequence[di].Value is Value && sequence[di].Tag is Tag && TPClib.Image.DicomFile.dataElementDictionary.GetDictionaryEntry(sequence[di].Tag) is DataElementDictionaryEntry)
                        Header.Add(TPClib.Image.DicomFile.dataElementDictionary.GetDictionaryEntry(sequence[di].Tag).Description, (sequence[di] as DataElement));
                }
                catch (TPCDicom.DicomException)
                {
                    //do not add null reference_times if it does exist
                    //throw new TPCDicomFileError("TPCDicom.DicomException");
                }
                catch (TPCGeneralHeaderException)
                {
                    //ignore multiple fields, just saving the first occurrence
                }
                //(0008,0012) DA [20090909]                               #   8, 1 InstanceCreationDate
                if (sequence[di].Tag.Equals("(0008,0012)"))
                {
                    System.DateTime span = (System.DateTime)sequence[di].Value.ToArray()[0];
                    Header.InstanceCreationTime = Header.InstanceCreationTime.AddDays(span.Day - 1);
                    Header.InstanceCreationTime = Header.InstanceCreationTime.AddMonths(span.Month - 1);
                    Header.InstanceCreationTime = Header.InstanceCreationTime.AddYears(span.Year - 1);
                }
                //(0008,0013) TM [152253.000000]                          #  14, 1 InstanceCreationTime
                if (sequence[di].Tag.Equals("(0008,0013)"))
                {
                    System.TimeSpan span = (System.TimeSpan)sequence[di].Value.ToArray()[0];
                    Header.InstanceCreationTime = Header.InstanceCreationTime.AddHours(span.Hours);
                    Header.InstanceCreationTime = Header.InstanceCreationTime.AddMinutes(span.Minutes);
                    Header.InstanceCreationTime = Header.InstanceCreationTime.AddSeconds(span.Seconds);
                    Header.InstanceCreationTime = Header.InstanceCreationTime.AddMilliseconds(span.Milliseconds);
                }
                //(0008,0018) UI [1.2.276.0.20.1.4.4.964242475771.4068.1252498973.921875] #  54, 1 SOPInstanceUID
                if (sequence[di].Tag.Equals("(0008,0018)"))
                    Header.SOPInstanceUID = (Uid)sequence[di].Value.ToArray()[0];
                //(0008,0020) DA [20090909]                               #   8, 1 StudyDate
                if (sequence[di].Tag.Equals("(0008,0020)"))
                {
                    System.DateTime span = (System.DateTime)sequence[di].Value.ToArray()[0];
                    Header.StudyTime = Header.StudyTime.AddDays(span.Day - 1);
                    Header.StudyTime = Header.StudyTime.AddMonths(span.Month - 1);
                    Header.StudyTime = Header.StudyTime.AddYears(span.Year - 1);
                }
                //(0008,0030) TM (no value available)                     #   0, 0 StudyTime
                if (sequence[di].Tag.Equals("(0008,0030)"))
                {
                    System.TimeSpan span = (System.TimeSpan)sequence[di].Value.ToArray()[0];
                    Header.StudyTime = Header.StudyTime.AddHours(span.Hours);
                    Header.StudyTime = Header.StudyTime.AddMinutes(span.Minutes);
                    Header.StudyTime = Header.StudyTime.AddSeconds(span.Seconds);
                    Header.StudyTime = Header.StudyTime.AddMilliseconds(span.Milliseconds);
                }
                //(0008,0070) LO [BrainLAB]                               #   8, 1 Manufacturer
                if (sequence[di].Tag.Equals("(0008,0070)")) Header.Manufacturer = sequence[di].Value.ToArray()[0].ToString();
                //(0008,0090) PN (no value available)                     #   0, 0 ReferringPhysiciansName
                if (sequence[di].Tag.Equals("(0008,0090)")) Header.ReferringPhysiciansName = sequence[di].Value.ToArray()[0].ToString();
                //(0008,1010) SH [IPLANRT1]                               #   8, 1 StationName
                if (sequence[di].Tag.Equals("(0008,1010)")) Header.StationName = sequence[di].Value.ToArray()[0].ToString();
                //(0008,1030) LO [d2890]                                  #   6, 1 StudyDescription
                if (sequence[di].Tag.Equals("(0008,1030)")) Header.StudyDescription = sequence[di].Value.ToArray()[0].ToString();
                //(0008,103e) LO [RT Struct Series]                       #  16, 1 SeriesDescription
                if (sequence[di].Tag.Equals("(0008,103e)")) Header.SeriesDescription = sequence[di].Value.ToArray()[0].ToString();
                //(0008,1090) LO [PatXferRT]                              #  10, 1 ManufacturersModelName
                if (sequence[di].Tag.Equals("(0008,1090)")) Header.ManufacturersModelName = sequence[di].Value.ToArray()[0].ToString();
                //(0010,0010) PN [NAME^SURNAME]                #  24, 1 PatientsName
                if (sequence[di].Tag.Equals("(0010,0010)"))
                    Header.PatientsName = sequence[di].Value.ToArray()[0].ToString();
                //(0010,0020) LO [211271-1402]                            #  12, 1 PatientID
                if (sequence[di].Tag.Equals("(0010,0020)")) Header.PatientID = sequence[di].Value.ToArray()[0].ToString();
                //(0010,0030) DA [19711221]                               #   8, 1 PatientsBirthDate
                if (sequence[di].Tag.Equals("(0010,0030)"))
                {
                    System.DateTime date;
                    System.DateTime.TryParse(sequence[di].Value.ToArray()[0].ToString(), out date);
                    Header.PatientsBirthDate = date;
                }
                //(0010,0040) CS [F]                                      #   2, 1 PatientsSex
                if (sequence[di].Tag.Equals("(0010,0040)"))
                {
                    string str = sequence[di].Value.ToArray()[0].ToString().Trim().ToUpperInvariant();
                    if (str.StartsWith("F")) Header.PatientsSex = ImageHeader.Patient_sex_e._FEMALE;
                    else if (str.StartsWith("M")) Header.PatientsSex = ImageHeader.Patient_sex_e.SEX_MALE;
                    else Header.PatientsSex = ImageHeader.Patient_sex_e._UNKNOWN;
                }
                //(0010,1020) DS [1.88]                                   #   4, 1 PatientsSize
                if (sequence[di].Tag.Equals("(0010,1020)")) Header.PatientsSize = (float)((Decimal)sequence[di].Value.ToArray()[0]);
                //(0010,1030) DS [86]                                     #   2, 1 PatientsWeight
                if (sequence[di].Tag.Equals("(0010,1030)")) Header.PatientsWeight = (float)((Decimal)sequence[di].Value.ToArray()[0]);
                //(0018,1020) LO [1.5.0 Build 118064]                     #  18, 1 SoftwareVersions
                if (sequence[di].Tag.Equals("(0018,1020)")) Header.SoftwareVersions = sequence[di].Value.ToArray()[0].ToString();
                //(0020,000d) UI [1.2.276.0.20.1.2.4.964242475771.4068.1252498939.93750] #  54, 1 StudyInstanceUID
                if (sequence[di].Tag.Equals("(0020,000d)")) Header.StudyInstanceUID = (Uid)sequence[di].Value.ToArray()[0];
                //(0020,000e) UI [1.2.276.0.20.1.3.4.964242475771.4068.1252498973.906250] #  54, 1 SeriesInstanceUID
                if (sequence[di].Tag.Equals("(0020,000e)")) Header.SeriesInstanceUID = (Uid)sequence[di].Value.ToArray()[0];
                //(0020,0010) SH [3634049, 4896]                          #  14, 1 StudyID
                if (sequence[di].Tag.Equals("(0020,0010)")) Header.StudyID = sequence[di].Value.ToArray()[0].ToString();
                
                //(3006,0008) 
                if (sequence[di].Tag.Equals("(3006,0008)"))
                {
                    System.DateTime span = (System.DateTime)sequence[di].Value.ToArray()[0];
                    Header.StructureSetCreationTime = Header.StructureSetCreationTime.AddDays(span.Day - 1);
                    Header.StructureSetCreationTime = Header.StructureSetCreationTime.AddMonths(span.Month - 1);
                    Header.StructureSetCreationTime = Header.StructureSetCreationTime.AddYears(span.Year - 1);
                }
                //(3006,0009) 
                if (sequence[di].Tag.Equals("(3006,0009)"))
                {
                    System.TimeSpan span = (System.TimeSpan)sequence[di].Value.ToArray()[0];
                    Header.StructureSetCreationTime = Header.StructureSetCreationTime.AddHours(span.Hours);
                    Header.StructureSetCreationTime = Header.StructureSetCreationTime.AddMinutes(span.Minutes);
                    Header.StructureSetCreationTime = Header.StructureSetCreationTime.AddSeconds(span.Seconds);
                    Header.StructureSetCreationTime = Header.StructureSetCreationTime.AddMilliseconds(span.Milliseconds);
                }

                //go to next data element
                di++;
            }
        }

        /// <summary>
        /// Writes a ROIContourSequence, tag(3006,0080)
        /// </summary>
        /// <param name="stream">output stream</param>
        protected void WriteRTROIObservationSequence(Stream stream)
        {
            byte[] buf;
            TPClib.Image.DicomFile.SequenceWriteData[] sequence = new TPClib.Image.DicomFile.SequenceWriteData[1];
            TPClib.Image.DicomFile.NestedDataSetWriteData[] nested = new TPClib.Image.DicomFile.NestedDataSetWriteData[1];
            int No_rois = GetTotalNumberOfROIs();
            StringBuilder points_str = new StringBuilder();
            Point p = new Point();

            //ReferencedFrameOfReferenceSequence
            sequence[0] = new TPClib.Image.DicomFile.SequenceWriteData();
            sequence[0].SetTag(0x3006, 0x080);
            //write item for each VOI
            for (int i = 0; i < VOIs.Count; i++)
            {
                nested[0] = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                if (VOIs[i] is RTSTRCT_ROIStack)
                    buf = TPClib.Image.DicomFile.ToBytesArray(((VOIs[i] as RTSTRCT_ROIStack).ROINumber.ToString()).ToCharArray());
                else
                    buf = TPClib.Image.DicomFile.ToBytesArray((i.ToString()).ToCharArray());
                nested[0].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0082, buf));
                if (VOIs[i] is RTSTRCT_ROIStack)
                    buf = TPClib.Image.DicomFile.ToBytesArray(((VOIs[i] as RTSTRCT_ROIStack).ObservationNumber.ToString()).ToCharArray());
                else
                    buf = TPClib.Image.DicomFile.ToBytesArray((i.ToString()).ToCharArray());
                nested[0].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0084, buf));
                if (VOIs[i] is RTSTRCT_ROIStack)
                    buf = TPClib.Image.DicomFile.ToBytesArray(((VOIs[i] as RTSTRCT_ROIStack).RTROIInterpretedType).ToCharArray());
                else
                    buf = TPClib.Image.DicomFile.ToBytesArray(("").ToCharArray());
                nested[0].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x00a4, buf));
                if (VOIs[i] is RTSTRCT_ROIStack)
                    buf = TPClib.Image.DicomFile.ToBytesArray(((VOIs[i] as RTSTRCT_ROIStack).ROIInterpreter).ToCharArray());
                else
                    buf = TPClib.Image.DicomFile.ToBytesArray(("").ToCharArray());
                nested[0].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x00a6, buf));
                sequence[0].Add(nested[0]);            
            }             
            TPClib.Image.DicomFile.WriteSequenceRecursive(stream, sequence[0]);
        }

        /// <summary>
        /// Writes a ROIContourSequence, tag(3006,0039)
        /// </summary>
        /// <param name="stream">output stream</param>
        protected void WriteROIContourSequence(Stream stream)
        {
            byte[] buf;
            TPClib.Image.DicomFile.SequenceWriteData[] sequence = new TPClib.Image.DicomFile.SequenceWriteData[3];
            TPClib.Image.DicomFile.NestedDataSetWriteData[] nested = new TPClib.Image.DicomFile.NestedDataSetWriteData[3];
            int No_rois = GetTotalNumberOfROIs();
            StringBuilder points_str = new StringBuilder();
            Point p = new Point();
            TraceROI roi = null;
            System.IFormatProvider formatprovider = System.Globalization.CultureInfo.CreateSpecificCulture("en-GB");

            //ReferencedFrameOfReferenceSequence
            sequence[0] = new TPClib.Image.DicomFile.SequenceWriteData();
            sequence[0].SetTag(0x3006, 0x039);
            //write item for each VOI
            for (int i = 0; i < VOIs.Count; i++)
            {
                nested[0] = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                buf = TPClib.Image.DicomFile.ToBytesArray((VOIs[i].Color[0] + "\\" + VOIs[i].Color[1] + "\\" + VOIs[i].Color[2]).ToCharArray());
                nested[0].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x002a, buf));
                if (VOIs[i] is ROIStack)
                {
                    //ContourSequence
                    sequence[1] = new TPClib.Image.DicomFile.SequenceWriteData();
                    sequence[1].SetTag(0x3006, 0x0040);
                    for (int j = 0; j < (VOIs[i] as ROIStack).Count; j++)
                    {
                        if (!((VOIs[i] as ROIStack)[j] is TraceROI)) continue;
                        roi = ((VOIs[i] as ROIStack)[j] as TraceROI);
                        nested[1] = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                        //ContourImageSequence
                        sequence[2] = new TPClib.Image.DicomFile.SequenceWriteData();
                        sequence[2].SetTag(0x3006, 0x0016);
                            nested[2] = new TPClib.Image.DicomFile.NestedDataSetWriteData();
                            buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.840.10008.5.1.4.1.1.2").ToCharArray());
                            nested[2].Add(TPClib.Image.DicomFile.CreateDataElement(0x0008, 0x1150, buf));
                            if(roi is RTSTRUCT_ROI)
                                buf = TPClib.Image.DicomFile.ToBytesArray(((roi as RTSTRUCT_ROI).ReferencedSOPInstanceUID.ToString()).ToCharArray());
                            else
                                buf = TPClib.Image.DicomFile.ToBytesArray(("1.2.276.0.20.1.4.4.0.4068.1252498973." + i).ToCharArray());
                            nested[2].Add(TPClib.Image.DicomFile.CreateDataElement(0x0008, 0x1155, buf));
                            sequence[2].Add(nested[2]);
                        nested[1].Add(sequence[2]);
                        //ContourGeometricType
                        buf = TPClib.Image.DicomFile.ToBytesArray(("CLOSED_PLANAR").ToCharArray());
                        nested[1].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0042, buf));
                        //NumberOfContourPoints
                        buf = TPClib.Image.DicomFile.ToBytesArray((roi.Count.ToString()).ToCharArray());
                        nested[1].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0046, buf));
                        //write contour vertex coordinates in groups of three: x_1\y_1\z_1\x_2\y_2\z_2...
                        points_str = new StringBuilder();
                        for (int k = 0; k < roi.Count; k++)
                        {
                            p = roi[k];
                            points_str.Append(p.X.ToString("##.####", formatprovider));
                            points_str.Append("\\");
                            points_str.Append(p.Y.ToString("##.####", formatprovider));
                            points_str.Append("\\");
                            points_str.Append(p.Z.ToString("##.####", formatprovider));
                            if(k < roi.Count-1) points_str.Append("\\");
                        }

                        buf = TPClib.Image.DicomFile.ToBytesArray((points_str.ToString()).ToCharArray());
                        nested[1].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0050, buf));
                        sequence[1].Add(nested[1]);
                    }
                    nested[0].Add(sequence[1]);

                }
                //write essential reference number 
                if (VOIs[i] is RTSTRCT_ROIStack)
                {
                    //ReferencedROINumber
                    buf = TPClib.Image.DicomFile.ToBytesArray(((VOIs[i] as RTSTRCT_ROIStack).ROINumber.ToString()).ToCharArray());
                    nested[0].Add(TPClib.Image.DicomFile.CreateDataElement(0x3006, 0x0084, buf));
                }
                sequence[0].Add(nested[0]);
            }
            TPClib.Image.DicomFile.WriteSequenceRecursive(stream, sequence[0]);
        }

        /// <summary>
        /// Reads a StructureSetROISequence, tag(3006,0020), until tag(3006,0039)
        /// </summary>
        /// <param name="sequence">read sequence</param>
        /// <param name="di">current sequence index at tag(3006,0020)</param>
        /// <param name="vois">vois where data is gathered</param>
        /// <returns>true if all data was read successfully, false otherwise</returns>
        protected static void ReadStructureSetROISequence(ref Sequence sequence, ref int di, ref List<VOI> vois) {
            if (!sequence[di].Tag.Equals("(3006,0020)"))
                throw new ArgumentException("tag must be (3006,0020)");
            Uid ReferencedFrameOfReferenceUID = new Uid("0");
            string ROIName = "";
            string ROIGenerationAlgorithm = "";
            int roinumber = 0;
            int i = 0;
            //read until sequence delimitation item
            while (++di < sequence.Count)
            {
                if (sequence[di].Tag.Equals("(fffe,e000)")) {
                    //read to end of item
                    while (++di < sequence.Count)
                    {
                        //(3006,0022) IS [0]                                      #   2, 1 ROINumber
                        if (sequence[di].Tag.Equals("(3006,0022)"))
                        {
                            roinumber = (int)((long)sequence[di].Value.ToArray()[0]);
                        }
                        //(3006,0024) UI [1.2.276.0.20.1.30.4.964242475771.4068.1252498939.125000] #  56, 1 ReferencedFrameOfReferenceUID
                        else if (sequence[di].Tag.Equals("(3006,0024)"))
                        {
                            ReferencedFrameOfReferenceUID = (Uid)sequence[di].Value.ToArray()[0];
                        }
                        //(3006,0026) LO [Brainstem]                              #  10, 1 ROIName
                        else if (sequence[di].Tag.Equals("(3006,0026)"))
                        {
                            ROIName = sequence[di].Value.ToArray()[0].ToString();
                        }
                        //(3006,0036) CS [SEMIAUTOMATIC]                          #  14, 1 ROIGenerationAlgorithm
                        else if (sequence[di].Tag.Equals("(3006,0036)"))
                        {
                            ROIGenerationAlgorithm = sequence[di].Value.ToArray()[0].ToString();

                            //add new data if corresponding reference number was found
                            for (i = 0; i < vois.Count; i++)
                            {
                                if ((vois[i] is RTSTRCT_ROIStack) && (vois[i] as RTSTRCT_ROIStack).ROINumber == roinumber)
                                {
                                    (vois[i] as RTSTRCT_ROIStack).ReferencedFrameOfReferenceUID = ReferencedFrameOfReferenceUID;
                                    (vois[i] as RTSTRCT_ROIStack).Name = ROIName;
                                    (vois[i] as RTSTRCT_ROIStack).RoigenerationAlgorithm = ROIGenerationAlgorithm;
                                    break;
                                }
                            }
                            //add new item into list if it was not found
                            if (i == vois.Count)
                            {
                                RTSTRCT_ROIStack voi = new RTSTRCT_ROIStack();
                                voi.ReferencedFrameOfReferenceUID = ReferencedFrameOfReferenceUID;
                                voi.Name = ROIName;
                                voi.RoigenerationAlgorithm = ROIGenerationAlgorithm;
                                voi.ROINumber = roinumber;
                                vois.Add(voi);
                            }
                            break;
                        }
                    }
                }
                //end tag is found
                else if (sequence[di].Tag.Equals("(3006,0039)")) 
                    break;
            }       
        }

        /// <summary>
        /// Reads a ROIContourSequence, tag(3006,0039), until tag(3006,0080)
        /// </summary>
        /// <param name="sequence">read sequence</param>
        /// <param name="di">current sequence index at tag(3006,0039)</param>
        /// <param name="vois">vois where data is gathered</param>
        /// <returns>true if all data was read successfully, false otherwise</returns>
        protected static void ReadROIContourSequence(ref Sequence sequence, ref int di, ref List<VOI> vois)
        {
            if (!sequence[di].Tag.Equals("(3006,0039)"))
                throw new ArgumentException("tag must be (3006,0039)");
            int[] ROIDisplayColor = new int[3];
            //read until sequence delimitation item
            while (++di < sequence.Count) {
                //(3006,002a) IS [81\232\67]                              #  10, 3 ROIDisplayColor
                if (sequence[di].Tag.Equals("(3006,002a)"))
                {
                    ROIDisplayColor[0] = (int)((long)sequence[di].Value.ToArray()[0]);
                    ROIDisplayColor[1] = (int)((long)sequence[di].Value.ToArray()[1]);
                    ROIDisplayColor[2] = (int)((long)sequence[di].Value.ToArray()[2]);
                }
                //exit if ending tag is found
                else if (sequence[di].Tag.Equals("(3006,0040)"))
                {
                    //read new list of contours
                    RTSTRCT_ROIStack stack = new RTSTRCT_ROIStack();
                    ReadContourSequenceItem(ref sequence, ref di, ref stack);
                    if (di < sequence.Count && sequence[di].Tag.Equals("(3006,0084)")) 
                        stack.ROINumber = (int)((long)sequence[di].Value.ToArray()[0]);

                    //add new data if corresponding reference number was found
                    int i = 0;
                    for (i = 0; i < vois.Count; i++)
                    {
                        if ((vois[i] is RTSTRCT_ROIStack) && (vois[i] as RTSTRCT_ROIStack).ROINumber == stack.ROINumber)
                        {
                            (vois[i] as RTSTRCT_ROIStack).Color[0] = (byte)(ROIDisplayColor[0] % 256);
                            (vois[i] as RTSTRCT_ROIStack).Color[1] = (byte)(ROIDisplayColor[1] % 256);
                            (vois[i] as RTSTRCT_ROIStack).Color[2] = (byte)(ROIDisplayColor[2] % 256);
                            (vois[i] as RTSTRCT_ROIStack).Catenate(stack);
                            break;
                        }
                    }
                    //add new item into list if it was not found
                    if (i == vois.Count)
                    {
                        stack.Color[0] = (byte)(ROIDisplayColor[0] % 255);
                        stack.Color[1] = (byte)(ROIDisplayColor[1] % 255);
                        stack.Color[2] = (byte)(ROIDisplayColor[2] % 255);
                        vois.Add(stack);
                    }
                }
                //end tag is found
                else if (sequence[di].Tag.Equals("(3006,0080)"))
                    break;
            }
        }

        /// <summary>
        /// Reads an item in ContourSequence, tag(3006,0040), until tag(3006,0084)
        /// </summary>
        /// <param name="sequence">read sequence</param>
        /// <param name="di">current sequence index at tag(3006,0040)</param>
        /// <param name="stack">stack of contours where data is gathered</param>
        /// <returns>true if all data was read successfully, false otherwise</returns>
        protected static void ReadContourSequenceItem(ref Sequence sequence, ref int di, ref RTSTRCT_ROIStack stack)
        {
            if (!sequence[di].Tag.Equals("(3006,0040)"))
                throw new ArgumentException("tag must be (3006,0040)");
            RTSTRUCT_ROI roi = new RTSTRUCT_ROI();
            //read until sequence delimitation item
            while (++di < sequence.Count)
            {
                //ReferencedSOPClassUID
                if(sequence[di].Tag.Equals("(0008,1150)"))
                    roi.ReferencedSOPClassUID = (Uid)sequence[di].Value.ToArray()[0];
                //ReferencedSOPInstanceUID
                if (sequence[di].Tag.Equals("(0008,1155)"))
                    roi.ReferencedSOPInstanceUID = (Uid)sequence[di].Value.ToArray()[0];
                //ContourGeometricType
                else if (sequence[di].Tag.Equals("(3006,0042)"))
                    roi.ContourGeometricType = sequence[di].Value.ToArray()[0].ToString();
                //ContourData
                else if (sequence[di].Tag.Equals("(3006,0050)"))
                {
                    RTSTRUCT_ROI newroi = new RTSTRUCT_ROI();
                    object[] values = sequence[di].Value.ToArray();
                    //values should be 3D space coordinates
                    if (values.Length % 3 != 0)
                        throw new TPCDicomFileException("tag(3006,0050) number of items must be dividable by 3");
                    for (int i = 0; i < values.Length; i += 3)
                    {
                        newroi.AddPoint(new Point((double)(Decimal)values[i], (double)(Decimal)values[i + 1], (double)(Decimal)values[i + 2]));
                    }
                    //add new data item into stack
                    newroi.ContourGeometricType = roi.ContourGeometricType;
                    newroi.ReferencedSOPClassUID = new Uid(roi.ReferencedSOPClassUID.ToString());
                    newroi.ReferencedSOPInstanceUID = new Uid(roi.ReferencedSOPInstanceUID.ToString());
                    stack.Add(newroi);
                }
                //exit if ending tag is found
                else if (sequence[di].Tag.Equals("(3006,0084)")) {
                    break;
                }
            }
        }   

        /// <summary>
        /// Reads a RTROIObservationsSequence, tag(3006,0080)
        /// </summary>
        /// <param name="sequence">read sequence</param>
        /// <param name="di">current sequence index at tag(3006,0080)</param>
        /// <param name="vois">vois where data is gathered</param>
        /// <returns>true if all data was read successfully, false otherwise</returns>
        protected static void ReadRTROIObservationsSequence(ref Sequence sequence, ref int di, ref List<VOI> vois)
        {
            if (!sequence[di].Tag.Equals("(3006,0080)"))
                throw new ArgumentException("tag must be (3006,0080)");
            int observationNumber= -1;
            int referencedROINumber = -1;
            string interpretedtype = "";
            string interpreter = "";
            //read until sequence delimitation item
            while (++di < sequence.Count)
            {
                if (sequence[di].Tag.Equals("(fffe,e000)")) {
                    //read to end of item
                    while (++di < sequence.Count)
                    {
                        //    (3006,0082) IS [0]                                      #   2, 1 ObservationNumber
                        if (sequence[di].Tag.Equals("(3006,0082)"))
                        {
                            observationNumber = (int)((long)sequence[di].Value.ToArray()[0]);
                        }
                        //    (3006,0084) IS [0]                                      #   2, 1 ReferencedROINumber
                        else if (sequence[di].Tag.Equals("(3006,0084)"))
                        {
                            referencedROINumber = (int)((long)sequence[di].Value.ToArray()[0]);
                        }
                        //    (3006,00a4) CS [ORGAN]                                  #   6, 1 RTROIInterpretedType
                        else if (sequence[di].Tag.Equals("(3006,00a4)"))
                        {
                            interpretedtype = sequence[di].Value.ToArray()[0].ToString();
                        }
                        //    (3006,00a6) PN (no value available)                     #   0, 0 ROIInterpreter
                        else if (sequence[di].Tag.Equals("(3006,00a6)"))
                        {
                            interpreter = sequence[di].Value.ToArray()[0].ToString();
                        }
                        //exit if ending tag is found
                        else if (sequence[di].Tag.Equals("(fffe,e00d)"))
                        {
                            //add new data if corresponding reference number was found
                            int i = 0;
                            for (i = 0; i < vois.Count; i++)
                            {
                                if ((vois[i] is RTSTRCT_ROIStack) && (vois[i] as RTSTRCT_ROIStack).ROINumber == referencedROINumber)
                                {
                                    (vois[i] as RTSTRCT_ROIStack).ObservationNumber = observationNumber;
                                    (vois[i] as RTSTRCT_ROIStack).RTROIInterpretedType = interpretedtype;
                                    (vois[i] as RTSTRCT_ROIStack).ROIInterpreter = interpreter;
                                    break;
                                }
                            }
                            //add new item into list if it was not found
                            if (i == vois.Count)
                            {
                                RTSTRCT_ROIStack voi = new RTSTRCT_ROIStack();
                                voi.ObservationNumber = observationNumber;
                                voi.ROINumber = referencedROINumber;
                                voi.RTROIInterpretedType = interpretedtype;
                                voi.ROIInterpreter = interpreter;
                                vois.Add(voi);
                            }
                            break;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Read file into variables
        /// </summary>
        public override void ReadFile()
        {
            TPCDicom.File.AcrNemaFile file = null;
            //set end condition tag as pixel data to skip reading it
            Sequence.EndConditionTag = new Tag("(7FE0,0010)");
            //call static constructor in order to load DICOM dictionaries
            new TPClib.Image.DicomFile("");
            try
            {
                if (TPCDicom.File.DicomFile.IsDicomFile(filename))
                    file = new TPCDicom.File.DicomFile(filename, false);
                else if (TPCDicom.File.AcrNemaFile.IsAcrNemaFile(filename))
                    file = new TPCDicom.File.AcrNemaFile(filename, false);
                else
                    Console.Error.WriteLine("File " + filename + " is wether a DICOM nor an ACR-NEMA file.");
            }
            catch (Exception dicomFileException)
            {
                throw new TPCDicomFileException("Failed to read DICOM file " + filename + ":" + dicomFileException);
            }
            finally
            {
                //take of end condition
                Sequence.EndConditionTag = null;
            }

            Sequence sequence = file.GetJointDataSets().GetJointSubsequences();
            //read all elements
            VOIs.Clear();
            List<RTSTRCT_ROIStack> number_name_pairs = new List<RTSTRCT_ROIStack>();
            int di = 0;
            //read header information
            ReadROIHeader(ref sequence, ref di);
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(0.0f, System.Reflection.MethodBase.GetCurrentMethod(), "Reading DICOM tags."));          
            while (++di < sequence.Count)
            {
                //StructureSetROISequence
                if (sequence[di].Tag.Equals("(3006,0020)"))
                    ReadStructureSetROISequence(ref sequence, ref di, ref VOIs);
                pevent = IOProgress;
                if (pevent != null)
                    pevent.Invoke(this, new IOProgressEventArgs(100.0f*((float)di/(float)sequence.Count), System.Reflection.MethodBase.GetCurrentMethod(), "Reading DICOM tags."));
                //ROIContourSequence
                if (di < sequence.Count && sequence[di].Tag.Equals("(3006,0039)"))
                    ReadROIContourSequence(ref sequence, ref di, ref VOIs);
                pevent = IOProgress;
                if (pevent != null)
                    pevent.Invoke(this, new IOProgressEventArgs(100.0f * ((float)di / (float)sequence.Count), System.Reflection.MethodBase.GetCurrentMethod(), "Reading DICOM tags."));
                //RTROIObservationsSequence
                if (di < sequence.Count && sequence[di].Tag.Equals("(3006,0080)"))
                    ReadRTROIObservationsSequence(ref sequence, ref di, ref VOIs);
                pevent = IOProgress;
                if (pevent != null)
                    pevent.Invoke(this, new IOProgressEventArgs(100.0f * ((float)di / (float)sequence.Count), System.Reflection.MethodBase.GetCurrentMethod(), "Reading DICOM tags."));
            }
            pevent = IOProgress;
            if (pevent != null)
                pevent.Invoke(this, new IOProgressEventArgs(100.0f, System.Reflection.MethodBase.GetCurrentMethod(), "Reading DICOM tags."));
        }

        /// <summary>
        /// Calculates total number of ROIs in all VOIs
        /// </summary>
        /// <returns>total number of ROIs in VOIs</returns>
        public int GetTotalNumberOfROIs() {
            int r = 0;
            for (int vi = 0; vi < VOIs.Count; vi++)
                if (VOIs[vi] is ROIStack) r += (VOIs[vi] as ROIStack).Count;
            return r;
        }

        /// <summary>
        /// Subclass for definition of ROIs
        /// </summary>
        public class RTSTRUCT_ROI : TraceROI
        {
            /// <summary>
            /// ReferencedSOPClassUID
            /// </summary>
            public Uid ReferencedSOPClassUID;

            /// <summary>
            /// ReferencedSOPInstanceUID
            /// </summary>
            public Uid ReferencedSOPInstanceUID;

            /// <summary>
            /// ContourGeometricType, currently allways closed planar ROI
            /// </summary>
            public string ContourGeometricType = "CLOSED_PLANAR";

            /// <summary>
            /// Number of Contour Points.
            /// </summary>
            public int NumberOfContourPoints {
                get { 
                    return Points.Count;
                }
            }

            /// <summary>
            /// Default constructor
            /// </summary>
            public RTSTRUCT_ROI()
                : base()
            {
                Unit = DataUnit.Pixel;
            }
        }

        /// <summary>
        /// RTSTRUCT extension for stacked ROI
        /// </summary>
        public class RTSTRCT_ROIStack: ROIStack
        {
            /// <summary>
            /// ROI generation algorithm, tag(3006,0036), default is empty string
            /// </summary>
            public string RoigenerationAlgorithm = "";

            /// <summary>
            /// Referenced frame of referenceUID, tag(3006,0024)
            /// </summary>
            public Uid ReferencedFrameOfReferenceUID;

            /// <summary>
            /// ROI number, tag(3006,0022) and tag(3006,0084)
            /// </summary>
            public int ROINumber;

            /// <summary>
            /// Read-only observation number, tag(3006,00A6), same as ROINumber
            /// </summary>
            /// <see cref="ROINumber"/>
            public int ObservationNumber;

            /// <summary>
            /// Read-only reference number, tag(3006,0084), same as ROINumber
            /// </summary>
            /// <see cref="ROINumber"/>
            public int ReferencedROINumber {
                get { return ROINumber; }
            }

            /// <summary>
            /// ROI RGB display color, tag(3006,0084), [0..255] values. This 
            /// is read-only variable for superclass Color field
            /// </summary>
            /// <see cref="VOI.Color"/>
            public int[] ROIDisplayColor { 
                get {
                    return new int[] { Color[0], Color[1], Color[2] };
                } 
            }

            /// <summary>
            /// Observation number, tag(3006,0082) 
            /// </summary>
            public string ROIInterpreter = "";

            /// <summary>
            /// ROI observation label shown in application, tag(3006,0085)
            /// </summary>
            public string ROIObservationLabel = "";

            /// <summary>
            /// ROI interpreted type, tag(3006,00A4) 
            /// </summary>
            public string RTROIInterpretedType = "";

            /// <summary>
            /// Struct of groupROI
            /// </summary>
            public RTSTRCT_ROIStack()
            {
            }
        }

        /// <summary>
        /// Header of RTSTRUCT ROI file
        /// </summary>
        public class RTROIHeader: HeaderFieldList
        {
            /// <summary>
            /// Creation time and data of file, tags (0008,0012) and (0008,0013)
            /// </summary>
            public System.DateTime InstanceCreationTime = new System.DateTime(0);
            /// <summary>
            /// Creation time and data of file, tags (3006,0008) and (3006,0009)
            /// </summary>
            public System.DateTime StructureSetCreationTime = new System.DateTime(0);
            /// <summary>
            /// File instance UID
            /// </summary>
            public Uid SOPInstanceUID = new Uid("1.2.276.0.20.1.4.4.964242475771.4068.1252498973.921875");
            /// <summary>
            /// Study time and data of file, tags (0008,0030) and (0008,0050)
            /// </summary>
            public System.DateTime StudyTime = new System.DateTime(0);
            /// <summary>
            /// Manufacturer, tag(0008,0070)
            /// </summary>
            public string Manufacturer = "unknown";
            /// <summary>
            /// ReferringPhysiciansName, tag(0008,0090)
            /// </summary>
            public string ReferringPhysiciansName = "unknown";
            /// <summary>
            /// StationName, tag(0008,1010)
            /// </summary>
            public string StationName = "unknown";
            /// <summary>
            /// StudyDescription, tag(0008,1030)
            /// </summary>
            public string StudyDescription = "";
            /// <summary>
            /// SeriesDescription, tag(0008,103e)
            /// </summary>
            public string SeriesDescription = "";
            /// <summary>
            /// ManufacturersModelName, tag(0008,1090)
            /// </summary>
            public string ManufacturersModelName = "";
            /// <summary>
            /// PatientsName, tag(0010,0010)
            /// </summary>
            public string PatientsName = "";
            /// <summary>
            /// PatientID, tag(0010,0020)
            /// </summary>
            public string PatientID = "";
            /// <summary>
            /// PatientsBirthDate, tag(0010,0030)
            /// </summary>
            public System.DateTime PatientsBirthDate = new System.DateTime(0);
            /// <summary>
            /// PatientsSex, tag(0010,0040)
            /// </summary>
            public ImageHeader.Patient_sex_e PatientsSex = ImageHeader.Patient_sex_e._UNKNOWN;
            /// <summary>
            /// PatientsSize, tag(0010,1020)
            /// </summary>
            public float PatientsSize = 0.0f;
            /// <summary>
            /// PatientsWeight, tag(0010,1030)
            /// </summary>
            public float PatientsWeight = 0.0f;
            /// <summary>
            /// SoftwareVersions, tag(0018,1020)
            /// </summary>
            public string SoftwareVersions = "";
            /// <summary>
            /// StudyInstanceUID, tag(0020,000d)
            /// </summary>
            public Uid StudyInstanceUID = new Uid("1.2.276.0.20.1.2.4.964242475771.4068.1252498939.93750");
            /// <summary>
            /// SeriesInstanceUID, tag(0020,000e)
            /// </summary>
            public Uid SeriesInstanceUID = new Uid("1.2.276.0.20.1.3.4.964242475771.4068.1252498973.906250");
            /// <summary>
            /// StudyID
            /// </summary>
            public string StudyID = "";
        }
    }
}
