/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using TPClib.Common;

namespace TPClib.ROI
{
    /// <summary>
    /// Rectangle-shaped Region Of Interest (ROI). The rectangle reference point is at the top left corner. 
    /// The rectangle region grows to the bottom right corner.
    /// </summary>
    public class RectangleROI : TPClib.ROI.ROI, IGeometricalROI
    {
        /// <summary>
        /// Horizontal side vector of rectangle
        /// </summary>
        public double Width = 0.0;
        /// <summary>
        /// Vertical side vector of rectangle
        /// </summary>
        public double Height = 0.0;
        /// <summary>
        /// Direction vector from top left corner to top right corner 
        /// </summary>
        private Point vertical_direction = new Point(1, 0, 0);
        /// <summary>
        /// Direction vector from top left corner to bottom right corner
        /// </summary>
        private Point horizontal_direction = new Point(0, 1, 0);

        /// <summary>
        /// Default constructor
        /// </summary>
        public RectangleROI() {
        
        }

        /// <summary>
        /// Cosntructor with values
        /// </summary>
        /// <param name="name">ROI name</param>
        /// <param name="x">ROI reference point location x-coordinate</param>
        /// <param name="y">ROI reference point location y-coordinate</param>
        /// <param name="z">ROI reference point location z-coordinate</param>
		/// <param name="width">Rectangle width</param>
		/// <param name="height">Rectangle height</param>
		public RectangleROI(String name, double x, double y, double z, double width, double height)
            : base(name, ROI_Type.Rectangle, x, y, z)
        {
            this.Width = width;
            this.Height = height;
        }

        /// <summary>
        /// Creates deep coyp of rectangle ROI
        /// </summary>
        public override object Clone()
        {
            RectangleROI r = new RectangleROI();
            r.Height = this.Height;
            r.Location = new Point(this.Location);
            r.ROIName = this.ROIName;
            r.Unit = this.Unit;
            r.Width = this.Width;
            return r;
        }

        /// <summary>
        /// Fills maskable item with 1's into given size mask image. 
        /// Origo is considered to be located at bottom left low corner of mask 
        /// </summary>
        /// <param name="mask">target mask image</param>
        /// <param name="method">filling method</param>
        public override void Fill(ref TPClib.Image.MaskImage mask, Fill_Method method)
        {
            //resolve four corners
            Point topleft = (Point)Location.Clone();
            Point topright = Location + horizontal_direction * Width;
            Point bottomleft = Location + vertical_direction * Height;
            Point bottomright = Location + vertical_direction * Height + horizontal_direction * Width;
            //resolve points of all sides
            List<Point> top = Image.ImageUtils.maskBresenhamLine2D(ref mask, topleft, topright);
            List<Point> bottom = Image.ImageUtils.maskBresenhamLine2D(ref mask, bottomleft, bottomright);
            List<Point> left = Image.ImageUtils.maskBresenhamLine2D(ref mask, topleft, bottomleft);
            List<Point> right = Image.ImageUtils.maskBresenhamLine2D(ref mask, topright, bottomright);
            List<Point> leftside = new List<Point>(); 
            List<Point> rightside = new List<Point>(); 
            if (horizontal_direction.Y >= 0)
            {
                top.Reverse();
                bottom.Reverse();
                if (vertical_direction.X >= 0)
                {
                    leftside.AddRange(top);
                    leftside.AddRange(left);
                    rightside.AddRange(right);
                    rightside.AddRange(bottom);
                }
                //vertical_direction.x < 0
                else
                {
                    leftside.AddRange(top);
                    leftside.AddRange(left);
                    rightside.AddRange(right);
                    rightside.AddRange(bottom);
                }
            }
            //horizontal_direction.y < 0
            else {
                if (vertical_direction.X >= 0)
                {
                    leftside.AddRange(top);
                    leftside.AddRange(right);
                    rightside.AddRange(left);
                    rightside.AddRange(bottom);
                }
                //vertical_direction.x < 0
                else
                {
                    leftside.AddRange(left);
                    leftside.AddRange(bottom);
                    rightside.AddRange(top);
                    rightside.AddRange(right);
                }
            }           

            //fill rectangle by scanlines along x-axis
            int x_left;
            int x_right;
            int j = 0;
            int pos;
            for (int i = 0; i < left.Count; i++) {
                pos = mask.DimX * mask.DimY * (int)leftside[i].Z + mask.DimX * (int)leftside[i].Y;
                //set starting position
                x_left = pos+(int)left[i].X;
                //find corresponding ending position from right side
                while ((int)rightside[j].Y < leftside[i].Y) j++;
                x_right = pos + (int)rightside[j].Y;
                while (x_left != x_right)
                    mask[x_left++] = 1;
            }
            //fill other direction of border
            if(method == Fill_Method.TwoWayBresenham) {
                Image.ImageUtils.maskBresenhamLine2D(ref mask, topright, topleft);
                Image.ImageUtils.maskBresenhamLine2D(ref mask, bottomright, bottomleft);
                Image.ImageUtils.maskBresenhamLine2D(ref mask, bottomleft, topleft);
                Image.ImageUtils.maskBresenhamLine2D(ref mask, bottomright, topright);            
            }
        }
        /// <summary>
        /// Gets bounding box of circle
        /// </summary>
        /// <returns>bounds of this ROI</returns>
        public override Limits GetBoundingBox()
        {
            return new Limits(new double[] { this.Location.X - Width / 2.0, this.Location.Y - Height / 2.0, this.Location.Z },
                              new double[] { this.Location.X + Width / 2.0, this.Location.Y + Height / 2.0, this.Location.Z });
        }
        /// <summary>
        /// Width of geometrical 2D shape. Note that the orientation may affect actual ROI presentation.
        /// </summary>
        double IGeometricalROI.Width
        {
            get
            {
                return Width;
            }
        }
        /// <summary>
        /// Height of geometrical 2D shape. Note that the orientation may affect actual ROI presentation.
        /// </summary>
        double IGeometricalROI.Height
        {
            get
            {
                return Width;
            }
        }
    }
}
