/********************************************************************************
*                                                                               *
*  TPClib 0.9 Medical imaging library                                           *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
********************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Common;

namespace TPClib.ROI
{
    /// <summary>
    /// Region Of Interest consisting of series of points that are connected in order and lay on the same plane (z-axis index).
    /// </summary>
    public class TraceROI : ROI
    {
        /// <summary>
        /// ROI vertexes in 3D space
        /// </summary>
        public List<Point> Points = new List<Point>();
        /// <summary>
        /// Gets the number of vertexes in this trace ROI.
        /// </summary>
        public int Count
        {
            get { return Points.Count; }
        }
        /// <summary>
        /// Creates Trace ROI
        /// </summary>
        public TraceROI()
            : base(ROI_Type.Trace)
        {
        }
        /// <summary>
        /// Creates Trace ROI
        /// </summary>
        public TraceROI(ROI roi)
            : base(roi.ROIName, ROI_Type.Trace, roi.Location.X, roi.Location.Y, roi.Location.Z)
        {
        }

        /// <summary>
        /// Creates Trace ROI at goven location
        /// </summary>
        /// <param name="x">x-coordinate</param>
        /// <param name="y">y-coordinate</param>
        /// <param name="z">z-coordinate</param>
        public TraceROI(int x, int y, int z)
            : base("", ROI_Type.Trace, x, y, z)
        {
        }

        /// <summary>
        /// Adds one point to TraceROI
        /// </summary>
        /// <param name="x">x coordinate of new point.</param>
        /// <param name="y">y coordinate of new point.</param>
        public void AddPoint(Double x, Double y)
        {
            Points.Add(new Point(x, y, Location.Z));
        }
        /// <summary>
        /// Adds point to TraceROI.
        /// </summary>
        /// <param name="ROIpoint">ROIPoint to add. Note that one x and y values are considered</param>
        public void AddPoint(Point ROIpoint)
        {
            Points.Add(new Point(ROIpoint.X, ROIpoint.Y, ROIpoint.Z));
        }
        /// <summary>
        /// Adds points to TraceROI.
        /// </summary>
        /// <param name="ROIpoints">ROIPoints to add. Note that one x and y values are considered</param>
        public void AddPoint(Point[] ROIpoints )
        {
            foreach (Point rp in ROIpoints)
            {
                Points.Add(new Point(rp.X, rp.Y, rp.Z));
            }
        }

        /// <summary>
        /// Gives new points to TraceROI object.
        /// </summary>
        /// <param name="point_list">List of new points. Note that one x and y values are considered</param>
        public void SetPoints(List<Point> point_list )
        {
            Points = new List<Point>();
            foreach (Point rp in point_list)
            {
                Points.Add(new Point(rp.X, rp.Y, rp.Z));
            }
        }

        /// <summary>
        /// Removes one point from TraceROI
        /// </summary>
        /// <param name="index">Index of point to remove.</param>
        public void RemovePoint(int index)
        {
            if (index < 0 || index >= Points.Count) throw new TPCROIException("RemovePoint: Index out of bounds.");
            Points.RemoveAt(index);
        }
        /// <summary>
        /// Index operator over list of vertexes
        /// </summary>
        /// <param name="i">index of vertex</param>
        /// <returns>vertex at index i</returns>
        public Point this[int i]
        {
            get
            {
                return Points[i];
            }
            set
            {
                Points[i] = (Point)value.Clone();
            }
        }
        /// <summary>
        /// Removes all the points from list.
        /// </summary>
        public void ClearPoints()
        {
            Points.Clear();
        }

		/// <summary>
		/// Fills maskable item with 1's into given sized mask image. 
		/// Origo is considered to be located at bottom left low corner of mask 
		/// (center of 1st voxel).
		/// </summary>
		/// <param name="mask">Mask to fill</param>
		/// <param name="method">Fill method</param>
		public override void Fill(ref TPClib.Image.MaskImage mask, Fill_Method method)
        {
            //ROI borders, that may lay outside mask region
            int[] border = new int[] { 0, 0, mask.DimX, mask.DimY }; 
            if (method == Fill_Method.TwoWayBresenham) {
                List<Point> p = new List<Point>();
                for (int i = 0; i < Points.Count; i++)
                {
                    p.Add(new Point(Math.Floor(Location.X + Points[i].X), Math.Floor(Location.Y + Points[i].Y), Location.Z));
                    if ((int)Math.Floor(p[p.Count - 1].X) < border[0]) border[0] = (int)p[p.Count - 1].X;
                    if ((int)Math.Floor(p[p.Count - 1].Y) < border[1]) border[1] = (int)p[p.Count - 1].Y;
                    if ((int)p[p.Count - 1].X > border[2]) border[2] = (int)p[p.Count - 1].X;
                    if ((int)p[p.Count - 1].Y > border[3]) border[3] = (int)p[p.Count - 1].Y;
                }
                //create 2D mask image that has region containing all vertexes in ROI
                Image.MaskImage maskregion = new TPClib.Image.MaskImage(border[2]-border[0]+2, border[3]-border[1]+2, 1);
                //move vertexes so that they lay relative to mask region
                for (int i = 0; i < Points.Count; i++) {
                    p[i].X -= border[0]-1;
                    p[i].Y -= border[1]-1;
                }
                //mask edge line
                MaskLinesTwoway(ref maskregion, ref p);

                //fill region inside polygon
                BoundaryFillRegionGrow(ref maskregion, (int)Location.Z);

                //copy masked region to target mask 
                for (int y = 0; y < mask.DimY; y++)
                {
                    for (int x = 0; x < mask.DimX; x++)
                    {
                        mask[x, y, (int)Location.Z] = maskregion[(y - (border[1] - 1))*maskregion.DimX + (x - (border[0] - 1))];
                    }
                }
            } else {
                throw new TPCException("Trace ROI filling with method " + method + " is not supported.");
            }
        }
        
        /// <summary>
        /// Masks all lines using two way Bresenham's algorithm
        /// </summary>
        /// <param name="mask">mask</param>
        /// <param name="points">Trace points</param>
        /// <returns>List of steps (one pixel)</returns>
        private static void MaskLinesTwoway(ref TPClib.Image.MaskImage mask, ref List<Point> points)
        {
            if (points.Count == 0) return;
            int i_n = 0;
            for (int i = 0; i < points.Count; i++)
            {
                if (i == points.Count - 1) i_n = 0;
                else i_n = i + 1;
                Image.ImageUtils.maskBresenhamLine2D(ref mask, points[i], points[i_n]);
                Image.ImageUtils.maskBresenhamLine2D(ref mask, points[i_n], points[i]);
            }
        }
        /// <summary>
        /// Returns a String that represents the current Object.
        /// </summary>
        /// <returns>A String that represents the current Object.</returns>
        public override string ToString()
        {
            StringBuilder str = new StringBuilder();
            foreach (Point rp in this.Points)
            {
                str.Append(
                    " (" + rp.X.ToString("f4", new System.Globalization.CultureInfo("en-GB")));
                str.Append(
                    ", " + rp.Y.ToString("f4", new System.Globalization.CultureInfo("en-GB")));
                str.Append(
                    ", " + rp.Z.ToString("f4", new System.Globalization.CultureInfo("en-GB")));
                str.Append(")");
            }

            return str.ToString();
        }
        /// <summary>
        /// Creates deep copy of this object
        /// </summary>
        /// <returns>cloned object</returns>
        public override Object Clone()
        {
            TraceROI roi = new TraceROI();
            roi.Location = new Point(Location);
            roi.Points = new List<Point>(Points);
            roi.ROIName = ROIName;
            roi.Unit = Unit;
            return roi;
        }
        /// <summary>
        /// Gets bounding box of stack of ROIs
        /// </summary>
        /// <returns>bounds of this ROI</returns>
        public override Limits GetBoundingBox()
        {
            Limits bounds = new Limits(new double[] { double.MaxValue, double.MaxValue, double.MaxValue },
                                       new double[] { double.MinValue, double.MinValue, double.MinValue });
            foreach (Point rp in this.Points)
            {
                //update bounding box values
                if (rp.X < bounds.GetLimit(0, Limits.Limit.LOW))
                    bounds.SetLimit(0, Limits.Limit.LOW, rp.X);
                if (rp.Y < bounds.GetLimit(1, Limits.Limit.LOW))
                    bounds.SetLimit(1, Limits.Limit.LOW, rp.Y);
                if (rp.Z < bounds.GetLimit(2, Limits.Limit.LOW))
                    bounds.SetLimit(2, Limits.Limit.LOW, rp.Z);
                if (rp.X > bounds.GetLimit(0, Limits.Limit.HIGH))
                    bounds.SetLimit(0, Limits.Limit.HIGH, rp.X);
                if (rp.Y > bounds.GetLimit(1, Limits.Limit.HIGH))
                    bounds.SetLimit(1, Limits.Limit.HIGH, rp.Y);
                if (rp.Z > bounds.GetLimit(2, Limits.Limit.HIGH))
                    bounds.SetLimit(2, Limits.Limit.HIGH, rp.Z);
            }
            return bounds;
        }
    }
}
