/********************************************************************************
*                                                                               *
*  TPCDicom 0.1 Dicom I/O library                                               *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
*  TPCDicom is based on the openDicom# 0.1.1 library.                           *
*                                                                               *
*  *** openDicom license ***                                                    *
*                                                                               *
*  openDICOM.NET openDICOM# 0.1.1                                               *
*                                                                               *
*  openDICOM# provides a library for DICOM related development on Mono.         *
*  Copyright (C) 2006-2007  Albert Gnandt                                       *
*                                                                               *
*  This library is free software; you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License as published by     *
*  the Free Software Foundation; either version 2.1 of the License, or (at      *
*  your option) any later version.                                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public       *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this library; if not, write to the                                *
*  Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA      *
*  02110-1301 USA                                                               *
*                                                                               *
********************************************************************************/

using System;
using System.IO;
using TPCDicom.Registry;
using TPCDicom.DataStructure;
using TPCDicom.Encoding;

namespace TPCDicom.DataStructure.DataSet
{

    /// <summary>
    ///     This class represents a DICOM tacs element.
    /// </summary>
    public sealed class DataElement: IComparable, IDicomStreamMember
    {
        /// <summary>
        /// Parent object in nested tag hierarchy
        /// </summary>
        public object parent = null;
        /// <summary>
        /// Gets parent object in nested tag hierarchy
        /// </summary>
        public object getParent()
        {
            return parent;
        }

        private Tag tag = null;
        /// <summary>
        ///     Data element tag.
        /// </summary>
        public Tag Tag
        {
            get 
            {
                if (tag != null) 
                    return tag; 
                else
                    throw new DicomException("DataElement.Tag is null.", 
                        (Tag) null);
            }
        }

        private ValueRepresentation vr = null;

		/// <summary>
        ///     Data element value representation.
        /// </summary>
        public ValueRepresentation VR
        {
            get 
            {
                if (vr != null) 
                    return vr; 
                else
                    throw new DicomException("DataElement.VR is null.",
                        this.Tag);
            }
        }

        private ValueLength valueLength = null;
        /// <summary>
        ///     Data element value length.
        /// </summary>
        public ValueLength ValueLength
        {
            get 
            {
                if (valueLength != null) 
                    return valueLength; 
                else
                    throw new DicomException("DataElement.ValueLength is null.",
                        this.Tag);
            }
        }

        private Value value = null;

		/// <summary>
        ///     Data element value.
        /// </summary>
        public Value Value
        {
            get 
            {
                if (value != null) 
                    return value; 
                else
                    throw new DicomException("DataElement.Value is null.",
                        this.Tag);
            }
        }

        /// <summary>
        ///     DICOM stream position of this tacs element instance.
        /// </summary>
        public long StreamPosition
        {
            get { return Tag.StreamPosition; }
        }

        private TransferSyntax transferSyntax = TransferSyntax.Default;

		/// <summary>
        ///     Transfer syntax of this tacs element instance.
        /// </summary>
        public TransferSyntax TransferSyntax
        {
            set
            {
                if (value == null)
                    transferSyntax = TransferSyntax.Default;
                else
                    transferSyntax = value;
            }

            get { return transferSyntax; }
        }
    
        /// <summary>
        ///     Creates a new tacs element instance from specified DICOM
        ///     output stream using the DICOM default transfer syntax.
        /// </summary>
        public DataElement(Stream stream): this(stream, null) {}

        /// <summary>
        ///     Creates a new tacs element instance from specified DICOM
        ///     output stream using specified DICOM transfer syntax.
        /// </summary>
        public DataElement(Stream stream, TransferSyntax transferSyntax)
        {
            TransferSyntax = transferSyntax;
            LoadFrom(stream);
        }

        /// <summary>
        ///     Creates a new tacs element instance from specified DICOM
        ///     output stream using specified DICOM transfer syntax.
        /// </summary>
        public DataElement(Stream stream, TransferSyntax transferSyntax, object obj)
        {
            TransferSyntax = transferSyntax;
            parent = obj;
            LoadFrom(stream);
        }

		/// <summary>
        /// Creates tacs element.
        /// </summary>
        /// <param name="tag">tag number</param>
        /// <param name="vr">value representation</param>
        public DataElement(string tag, string vr): this(tag, vr, null) {}

		/// <summary>
        /// Creates tacs element.
        /// </summary>
        /// <see cref="TransferSyntax"/>
        /// <param name="tag">tag number</param>
        /// <param name="vr">value representation</param>
        /// <param name="transferSyntax">element transfer syntac</param>
        public DataElement(string tag, string vr, TransferSyntax transferSyntax)
        {
            TransferSyntax = transferSyntax;
            this.tag = new Tag(tag, TransferSyntax);
            this.vr = ValueRepresentation.GetBy(vr, Tag);
        }

        /// <summary>
        /// Creates tacs element.
        /// </summary>
        /// <param name="tag">tag number</param>
        /// <param name="value">value at tag</param>
        public DataElement(Tag tag, Value value)
        {
            this.tag = tag;
            this.value = value;
        }

		/// <summary>
		/// Creates data element
		/// </summary>
		/// <param name="t">Tag for this element</param>
		/// <param name="stream">Stream from which to read the rest</param>
		/// <param name="transferSyntax">Element transfer syntax</param>
		/// <param name="obj">Parent object</param>
		public DataElement(Tag t, Stream stream, TransferSyntax transferSyntax, object obj)
		{
			TransferSyntax = transferSyntax;
			parent = obj;
			this.tag = t;
			//sequence
			vr = ValueRepresentation.LoadFrom(stream, Tag);
			valueLength = new ValueLength(stream, VR);
			value = new Value(stream, VR, ValueLength, this);
		}

		/// <summary>
        /// Creates tacs element.
        /// </summary>
        /// <param name="tag">tag number</param>
        /// <param name="vr">value representation</param>
        public DataElement(Tag tag, ValueRepresentation vr)
        {
            this.tag = tag;
            TransferSyntax = Tag.TransferSyntax;
            if (vr == null) vr = ValueRepresentation.GetBy(Tag);
            this.vr = vr;
        }

        /// <summary>
        ///     Re-creates a tacs element instance from specified DICOM
        ///     input stream using <see cref="DataElement.TransferSyntax" />.
        /// </summary>
        /// <param name="stream">input stream</param>
        public void LoadFrom(Stream stream)
        {
            tag = new Tag(stream, TransferSyntax);
            //sequence
            vr = ValueRepresentation.LoadFrom(stream, Tag);
            valueLength = new ValueLength(stream, VR);
            value = new Value(stream, VR, ValueLength, this);
        }

		/// <summary>
        ///     Writes a DICOM stream member instance to a specified
        ///     DICOM output stream.
        /// </summary>      
        public void SaveTo(Stream stream)
        {
            tag.SaveTo(stream);
            value.VR.SaveTo(stream);
            value.SaveTo(stream);     
        }

        /// <summary>
        ///     Implementation of the IComparable interface. So use
        ///     of this class within collections is guaranteed.
        /// </summary>
        public int CompareTo(object obj)
        {
            return Tag.CompareTo(((DataElement) obj).Tag);
        }

		/// <summary>
        /// Returns string representation of this object. 
        /// </summary>
        /// <returns>this object as a string</returns>
        public override string ToString()
        {
            //resolve object type
            if (value == null) return "<null>";

            object[] data_array = value.ToArray();
            if (data_array.Length == 0) return "<no tacs>";
            System.Text.StringBuilder r = new System.Text.StringBuilder();
            for (int i = 0; i < data_array.Length; i++) {
                switch (data_array[i].GetType().Name) {
                    case "PersonName":
                        r.Append(((TPCDicom.Encoding.Type.PersonName)data_array[i]).FullName);
                        break;
                    case "Decimal":
                        if(i > 0) r.Append(" ");
                        r.Append(((Decimal)data_array[i]).ToString().Replace(',', '.'));
                        break;
                    case "String":
                        r.Append(((String)data_array[i]).ToString());
                        break;
                    case "UInt16[]":
                        UInt16[] uint16s = (UInt16[])data_array[i];
                        foreach (UInt16 b in uint16s)
                            r.Append(b.ToString());
                        break;
                    case "UInt16":
                        r.Append(((UInt16)data_array[i]).ToString());
                        break;
                    case "Int16":
                        r.Append(((Int16)data_array[i]).ToString());
                        break;
                    case "UInt32":
                        r.Append(((UInt32)data_array[i]).ToString());
                        break;
                    case "Int32":
                        r.Append(((Int32)data_array[i]).ToString());
                        break;
                    case "UInt64":
                        r.Append(((UInt64)data_array[i]).ToString());
                        break;
                    case "Int64":
                        r.Append(((Int64)data_array[i]).ToString());
                        break;
                    case "Sequence":
                        r.Append(" Sequence");
                        break;
                    case "NestedDataSet":
                        r.Append(" NestedDataSet");
                        break;
                    case "Byte[]":
                        Byte[] bytes = (Byte[])data_array[i];
                        foreach(Byte b in bytes)
                            r.Append(" " + b.ToString());
                        break;
                    case "Uid":
                        //representation resolving skipped for performance reasons
                        if (UidDictionary.Global.GetDictionaryEntry((Uid)data_array[i]) != null)
							r.Append(" " + UidDictionary.Global.GetDictionaryEntry((Uid)data_array[i]).ToString());
                        else
                            r.Append("[" + ((Uid)data_array[i]).ToString() + "] (unrecognized UID)");
                        break;
                    case "DateTime":
                        r.Append(" " + ((System.DateTime)data_array[i]).ToString("HH:mm:ss dd:MM:yyyy"));
                        break;
                    case "TimeSpan":
                        r.Append(" " + ((TimeSpan)data_array[i]).ToString());
                        break;
                    case "Single":
                        r.Append(" " + ((Single)data_array[i]).ToString("0.00000").Replace(',','.'));
                        break;
                    case "Double":
                        r.Append(" " + ((Double)data_array[i]).ToString("0.0000000000").Replace(',','.'));
                        break;
                    case "Age":
                        TPCDicom.Encoding.Type.Age age = (TPCDicom.Encoding.Type.Age)data_array[i];
                        r.Append(" " + age.AgeValue.ToString());
                        switch (age.Context)
                        {
                            case TPCDicom.Encoding.Type.AgeContext.Days:
                                r.Append(" days");
                                break;
                            case TPCDicom.Encoding.Type.AgeContext.Months:
                                r.Append(" months");
                                break;
                            case TPCDicom.Encoding.Type.AgeContext.Weeks:
                                r.Append(" weeks");
                                break;
                            case TPCDicom.Encoding.Type.AgeContext.Years:
                            default:
                                r.Append(" years");
                                break;
                        }
                        break;
                    default:
                        r.Append(" unrecognized type = ["+data_array[i].GetType().Name+"]");
                        break;
                }
            }
            return r.ToString().TrimStart(' ');
        }

		/// <summary>
		/// Load a single dataelement from the stream. In addition, if element has the specified skip tag, do not load the element value
		/// </summary>
		/// <param name="s">Stream to read</param>
		/// <param name="tsyntax">Transfer syntax for this element</param>
		/// <param name="par">Parent object</param>
		/// <param name="skipTag">If element has this tag, skip reading the element value</param>
		/// <returns>next Dataelement from the stream</returns>
		public static DataElement LoadElement(Stream s, TransferSyntax tsyntax, object par, System.Collections.Generic.ICollection<Tag> skipTags)
		{
			DataElement de;

			if (skipTags != null && skipTags.Count > 0)
			{
				Tag t = new Tag(s, tsyntax);
				ValueRepresentation v = ValueRepresentation.LoadFrom(s, t);
				de = new DataElement(t, v);
				de.valueLength = new ValueLength(s, v);
				de.parent = par;

				if (skipTags.Contains(t))
				{
					de.value = new Value(v, de.valueLength);
					s.Seek(de.ValueLength.Value, SeekOrigin.Current);
				}
				else de.value = new Value(s, v, de.ValueLength, de);
			}
			else de = new DataElement(s, tsyntax, par);
			return de;
		}

		/// <summary>
        /// Explicit cast operator.
        /// </summary>
        /// <param name="e">dataelement</param>
        /// <returns>string representation from ToString() method</returns>
        public static implicit operator string(DataElement e) {
            return e.ToString();
        }
    }
}
