/********************************************************************************
*                                                                               *
*  TPCDicom 0.1 Dicom I/O library                                               *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
*  TPCDicom is based on the openDicom# 0.1.1 library.                           *
*                                                                               *
*  *** openDicom license ***                                                    *
*                                                                               *
*  openDICOM.NET openDICOM# 0.1.1                                               *
*                                                                               *
*  openDICOM# provides a library for DICOM related development on Mono.         *
*  Copyright (C) 2006-2007  Albert Gnandt                                       *
*                                                                               *
*  This library is free software; you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License as published by     *
*  the Free Software Foundation; either version 2.1 of the License, or (at      *
*  your option) any later version.                                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public       *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this library; if not, write to the                                *
*  Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA      *
*  02110-1301 USA                                                               *
*                                                                               *
********************************************************************************/

using System;
using System.IO;
using System.Collections;
using TPCDicom.DataStructure;
using TPCDicom.Encoding;


namespace TPCDicom.DataStructure.DataSet
{

    /// <summary>
    ///     This class represents a DICOM tacs set.
    /// </summary>
    /// <remarks>
    ///     This is the basic container class for all other DICOM stream
    ///     releated container classes with unique tacs element support.
    ///     DICOM tacs set is an ascending ordered sequence of tacs elements
    ///     with unique DICOM tags.
    /// </remarks>  
    public class DataSet: Sequence
    {
        // key index pair for access of Sequence as array
        private Hashtable keys = new Hashtable();

        /// <summary>
        ///     Access of a DICOM tacs set instance as array of 
        ///     <see cref="DataElement" />. The index is a DICOM tag.
        /// </summary>
        public DataElement this[Tag tag]
        {
            get 
            { 
                int index = (int) keys[tag.ToString()];
                return base[index]; 
            }
        }

        /// <summary>
        ///     Creates a new empty DICOM tacs set instance.
        /// </summary>
        public DataSet() {}

        /// <summary>
        ///     Creates a new DICOM tacs set instance and fills it from
        ///     specified DICOM output stream using the default transfer
        ///     syntax.
        /// </summary>
        public DataSet(Stream stream): base(stream) {}

        /// <summary>
        ///     Creates a new DICOM tacs set instance and fills it from
        ///     specified DICOM output stream using specified transfer
        ///     syntax.
        /// </summary>
        public DataSet(Stream stream, TransferSyntax transferSyntax, object obj) :
            base(stream, transferSyntax, obj) { }

        /// <summary>
        ///     Creates a new DICOM tacs set instance and fills it from
        ///     specified DICOM output stream using specified transfer
        ///     syntax.
        /// </summary>
        public DataSet(Stream stream, TransferSyntax transferSyntax):
            base(stream, transferSyntax) {}
        /// <summary>
        ///     Creates a new DICOM tacs set instance and fills it from
        ///     specified DICOM output stream using specified transfer
        ///     syntax.
        /// </summary>
        public DataSet(Tag tag, Value value) {
                this.Add(new DataElement(tag, value));
        }

        /// <summary>
        ///     Re-creates a new DICOM tacs set instance and fills it from
        ///     specified DICOM output stream using
        ///     <see cref="Sequence.TransferSyntax" />.
        /// </summary>
		public override void LoadFrom(Stream stream)
		{
			streamPosition = stream.Position;
			TransferSyntax.CharacterRepertoire = CharacterRepertoire.Default;
			bool isTrailingPadding = false;
			while (stream.Position < stream.Length && !isTrailingPadding)
			{
				try
				{
					// DEBUG DataElement element = new DataElement(stream, TransferSyntax);
					DataElement element = DataElement.LoadElement(stream, TransferSyntax, this, Sequence.SkipDataReadTags);
					//END

					if (element.Tag.Equals(CharacterRepertoire.CharacterSetTag))
						TransferSyntax.CharacterRepertoire =
							new CharacterRepertoire((string)element.Value[0]);
					isTrailingPadding = element.Tag.Equals("(0000,0000)");
					if (!isTrailingPadding)
						Add(element);

					if (Sequence.EndConditionTag != null && element.Tag.Equals(Sequence.EndConditionTag))
                    {
                        break;
                    }
				}
				catch (EncodingException encodingException)
				{
					//throw;
					Console.WriteLine("WARNING: failed to read tag(" +
										encodingException.Tag.Group +
										"," + encodingException.Tag.Element +
										"):" + encodingException.Message + " [" +
										encodingException.ParamValue + "]");
				}
				catch (Exception e)
				{
					//throw;
					Console.WriteLine("WARNING: failed to read tag(" + e + ")");
				}
			}
		}

        /// <summary>
        ///     Adds a new DICOM tacs element to this tacs set instance.
        ///     Multiple tacs elements of the same DICOM tag are not allowed
        ///     within a tacs set. Uniqueness is guaranteed by DICOM tags.
        /// </summary>
        public override int Add(DataElement dataElement)
        {
            int index = base.Add(dataElement);
            keys.Add(dataElement.Tag.ToString(), index);
            return index;
        }

        /// <summary>
        ///     Concatenates another DICOM tacs set instance with this tacs set
        ///     instance. Multiple tacs elements of the same DICOM tag are not
        ///     allowed during concatentation. Uniqueness is guaranteed by
        ///     DICOM tags.
        /// </summary>
        public void Add(DataSet dataSet)
        {
            foreach (DataElement element in dataSet)
                Add(element);
        }

        /// <summary>
        ///     Returns all sequences of all levels of a tacs set tree as
        ///     one concatenated zero-level sequence. Multiple tacs elements
        ///     of equal instances are allowed within a DICOM sequence.
        /// </summary>
        public override Sequence GetJointSubsequences()
        {
            return base.GetJointSubsequences();
        }

        /// <summary>
        ///     Determines whether this tacs set instance contains a tacs
        ///     element with specified DICOM tag.
        /// </summary>
        public bool Contains(Tag tag)
        {
            return keys.Contains(tag.ToString());
        }

        /// <summary>
        ///     Clears all DICOM tacs set properties.
        /// </summary>
        public override void Clear()
        {
            keys.Clear();
            base.Clear();
        }

        /// <summary>
        ///     Sorts all tacs elements of a tacs set instance by their DICOM
        ///     tag in ascending order.
        /// </summary>
        public override void Sort()
        {
            base.Sort();
            for (int index = 0; index < Count; index++)
            {
                string tagKey = base[index].Tag.ToString();
                keys[tagKey] = index;
            }
        }
    }

}
