/********************************************************************************
*                                                                               *
*  TPCDicom 0.1 Dicom I/O library                                               *
*  Copyright (C) 2011 Turku PET Centre                                          *
*                                                                               *
*  This library is free software: you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License (LGPL) as           *
*  published by the Free Software Foundation, either version 2.1 of the         *
*  License, or (at your option) any later version.                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public      *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this program.  If not, see <http://www.gnu.org/licenses/>.        *
*                                                                               *
*  TPCDicom is based on the openDicom# 0.1.1 library.                           *
*                                                                               *
*  *** openDicom license ***                                                    *
*                                                                               *
*  openDICOM.NET openDICOM# 0.1.1                                               *
*                                                                               *
*  openDICOM# provides a library for DICOM related development on Mono.         *
*  Copyright (C) 2006-2007  Albert Gnandt                                       *
*                                                                               *
*  This library is free software; you can redistribute it and/or modify it      *
*  under the terms of the GNU Lesser General Public License as published by     *
*  the Free Software Foundation; either version 2.1 of the License, or (at      *
*  your option) any later version.                                              *
*                                                                               *
*  This library is distributed in the hope that it will be useful, but          *
*  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY   *
*  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public       *
*  License for more details.                                                    *
*                                                                               *
*  You should have received a copy of the GNU Lesser General Public License     *
*  along with this library; if not, write to the                                *
*  Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA      *
*  02110-1301 USA                                                               *
*                                                                               *
********************************************************************************/

using System;
using System.IO;
using TPCDicom.DataStructure;
using TPCDicom.Encoding;

namespace TPCDicom.DataStructure.DataSet
{

    /// <summary>
    ///     This class represents a DICOM nested tacs set.
    /// </summary>
    public sealed class NestedDataSet: DataSet
    {
		/// <summary>
		///     DICOM tag (FFFE,E00D).
		/// </summary>
		public static readonly Tag NestedDelimiterTag = new Tag("FFFE", "E00D");

		/// <summary>
        /// Bytes in item list of predefined length.
        /// </summary>
        public new static int bytesInDataset = int.MaxValue;

        /// <summary>
        ///     Creates a new DICOM nested tacs set instance from specified
        ///     DICOM output stream using the default transfer syntax.
        /// </summary>
        public NestedDataSet(Stream stream): base(stream) {}

        /// <summary>
        ///     Creates a new DICOM nested tacs set instance from specified
        ///     DICOM output stream using specified transfer syntax.
        /// </summary>
        public NestedDataSet(Stream stream, TransferSyntax transferSyntax, object obj) :
            base(stream, transferSyntax, obj) { }

        /// <summary>
        ///     Creates a new DICOM nested tacs set instance from specified
        ///     DICOM output stream using specified transfer syntax.
        /// </summary>
        public NestedDataSet(Stream stream, TransferSyntax transferSyntax):
            base(stream, transferSyntax) {}

        /// <summary>
        ///     Re-creates a DICOM nested tacs set instance from specified
        ///     DICOM output stream using <see cref="Sequence.TransferSyntax" />.
        /// </summary>
        public override void LoadFrom(Stream stream)
        {
            streamPosition = stream.Position;
            DataElement element;
            bool isTrailingPadding = false;
            bool bytesInDatasetReached = false;
            do {
                //Console.WriteLine("START n " + this.GetHashCode() + " pos:" + (stream.Position - streamPosition) + " bytes:" + bytesInDataset);
                element = new DataElement(stream, TransferSyntax, this);
                //Console.WriteLine("END   n " + this.GetHashCode() + " pos:" + (stream.Position - streamPosition) + " bytes:" + bytesInDataset);
                bytesInDatasetReached = false;
                //Console.Write(stream.Position + " < ");
                IDicomStreamMember o = (IDicomStreamMember)this;
                while (true)
                {
                    if (o is Sequence)
                    {
                        //Console.Write("(" + (((Sequence)o).StreamPosition + ((Sequence)o).bytesInDataset) + ")");
                        if(stream.Position >= (((Sequence)o).StreamPosition + ((Sequence)o).bytesInDataset))
                            bytesInDatasetReached = true;
                    }
                    else
                    {
                        //Console.Write("( )");
                    }
                    if (!(o.getParent() is IDicomStreamMember))
                        break;
                    o = (IDicomStreamMember)o.getParent();
                }
                //Console.WriteLine();
                isTrailingPadding = element.Tag.Equals("(0000,0000)");
                if (!isTrailingPadding)
                {
                    Add(element);
                }
                else
                {
                    //Console.WriteLine("trailing");
                    break;
                }
			} while (!element.Tag.Equals(NestedDelimiterTag) && !element.Tag.Equals(SequenceDelimiterTag) &&
                     stream.Position < stream.Length && !bytesInDatasetReached);
            bytesInDataset = int.MaxValue;
        }
    }
}
