/*
   
    openDICOM.NET openDICOM# 0.1.1

    openDICOM# provides a library for DICOM related development on Mono.
    Copyright (C) 2006-2007  Albert Gnandt

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA


    $Id: AcrNemaFile.cs 48 2007-03-28 13:49:15Z agnandt $

    This is a modified version of original library for Turku PET Centre 2011
*/
using System;
using System.IO;
using System.Collections;
using openDicom.DataStructure;
using openDicom.Encoding;


namespace openDicom.DataStructure.DataSet
{

	/// <summary>
	///     This class represents a DICOM sequence.
	/// </summary>
	/// <remarks>
	///     This is the basic container class for all other DICOM stream
	///     releated container classes. DICOM sequences of items is an
	///     ordered sequence of tacs element with the same DICOM tag (FFFE,E000).
	/// </remarks>  
	public class Sequence : IDicomStreamMember
	{
		/// <summary>
		///     DICOM tag (FFFE,E0DD).
		/// </summary>
		public static readonly Tag SequenceDelimiterTag = new Tag("FFFE", "E0DD");

		/// <summary>
		/// User defined end condition tag for reading. Reading of file 
		/// ends when this tag is found if it is non-null. Default: null.
		/// </summary>
		public static Tag EndConditionTag = null;

		/// <summary>
		/// List of tags that are not read in full. Reading the value is skipped.
		/// Useful for skipping large data values (e.g. Pixeldata in an image) when reading header info.
		/// Defaults to null.
		/// </summary>
		public static Tag[] SkipDataReadTags = null;

		/// <summary>
		/// List of items in this sequence.
		/// </summary>
		protected ArrayList itemList = new ArrayList();

		/// <summary>
		/// Parent object in nested tag hierarchy.
		/// </summary>
		public object parent = null;

		/// <summary>
		/// Gets parent object in nested tag hierarchy.
		/// </summary>
		public object getParent()
		{
			return parent;
		}

		/// <summary>
		/// Bytes in item list of predefined length. -1 if not set.
		/// </summary>
		public int bytesInDataset = int.MaxValue;

		/// <summary>
		///     Access of this sequence instance as array of
		///     <see cref="DataElement" />.
		/// </summary>
		public DataElement this[int index]
		{
			get { return (DataElement)itemList[index]; }
		}

		/// <summary>
		///     Returns count of DICOM tacs elements.
		/// </summary>        
		public int Count
		{
			get { return itemList.Count; }
		}

		/// <summary>
		///     Returns whether this instance contains DICOM tacs elements.
		/// </summary>        
		public bool IsEmpty
		{
			get { return Count == 0; }
		}

		private TransferSyntax transferSyntax = TransferSyntax.Default;

		/// <summary>
		///     Access corresponding DICOM transfer syntax. If null is assinged,
		///     the DICOM default transfer syntax will be used instead.
		/// </summary>        
		public TransferSyntax TransferSyntax
		{
			set
			{
				if (value == null)
					transferSyntax = TransferSyntax.Default;
				else
					transferSyntax = value;
			}

			get { return transferSyntax; }
		}
		
		/// <summary>
		/// Instance's stream position. (default == -1)
		/// </summary>
		protected long streamPosition = -1;

		/// <summary>
		///     Returns this instance's position within a DICOM tacs stream.
		///     If this instance has not get in contact with a DICOM stream,
		///     no position will be marked and -1 will be returned.
		/// </summary>        
		public long StreamPosition
		{
			get { return streamPosition; }
		}


		/// <summary>
		///     Creates a new empty DICOM sequence instance.
		/// </summary>        
		public Sequence() { }

		/// <summary>
		///     Creates a new DICOM sequence instance and fills it with
		///     DICOM tacs elements from specified DICOM output stream using
		///     the default transfer syntax.
		/// </summary>        
		public Sequence(Stream stream) : this(stream, null) { }

		/// <summary>
		///     Creates a new DICOM sequence instance and fills it with
		///     DICOM tacs elements from specified DICOM output stream using
		///     specified transfer syntax.
		/// </summary>        
		public Sequence(Stream stream, TransferSyntax transferSyntax, object obj)
		{
			TransferSyntax = transferSyntax;
			parent = obj;
			LoadFrom(stream);
		}

		/// <summary>
		///     Creates a new DICOM sequence instance and fills it with
		///     DICOM tacs elements from specified DICOM output stream using
		///     specified transfer syntax.
		/// </summary>        
		public Sequence(Stream stream, TransferSyntax transferSyntax)
		{
			TransferSyntax = transferSyntax;
			LoadFrom(stream);
		}

		/// <summary>
		///     Creates a new DICOM nested tacs set instance from specified
		///     DICOM output stream using specified transfer syntax.
		/// </summary>
		/// <param name="stream">input stream</param>
		/// <param name="transferSyntax">DICOM transfer syntax</param>
		/// <param name="bytes">expected number of bytes in tacs set</param>
		/// <param name="obj">parent object in tag hierarchy</param>
		public Sequence(Stream stream, TransferSyntax transferSyntax, int bytes, object obj)
		{
			TransferSyntax = transferSyntax;
			bytesInDataset = bytes;
			parent = obj;
			LoadFrom(stream);
		}

		/// <summary>
		///     Creates a new DICOM nested tacs set instance from specified
		///     DICOM output stream using specified transfer syntax.
		/// </summary>
		/// <param name="stream">input stream</param>
		/// <param name="transferSyntax">DICOM transfer syntax</param>
		/// <param name="bytes">expected number of bytes in tacs set</param>
		public Sequence(Stream stream, TransferSyntax transferSyntax, int bytes)
		{
			TransferSyntax = transferSyntax;
			bytesInDataset = bytes;
			LoadFrom(stream);
		}

		/// <summary>
		///     Adds a DICOM tacs element instance to this instance. Multiple
		///     tacs elements of equal instances are allowed within a
		///     sequence.
		/// </summary>        
		public virtual int Add(DataElement dataElement)
		{
			return itemList.Add(dataElement);
		}

		/// <summary>
		///     Concatenates another DICOM sequence instance with this sequence
		///     instance. Multiple tacs elements of the same DICOM tag are
		///     allowed during concatentation.
		/// </summary>
		public void Add(Sequence sequence)
		{
			foreach (DataElement element in sequence)
			{
				Add(element);
			}
		}

		/// <summary>
		///     Re-creates a new DICOM sequence instance and fills it with
		///     DICOM tacs elements from specified DICOM input stream using
		///     <see cref="Sequence.TransferSyntax" />.
		/// </summary>
		/// <param name="stream">input stream</param>
		public virtual void LoadFrom(Stream stream)
		{
			streamPosition = stream.Position;
			DataElement element;
			bool isTrailingPadding = false;
			bool bytesInDatasetReached = false;
			do
			{
				if (bytesInDataset == 0) break;
				// DEBUG element = new DataElement(stream, TransferSyntax, this);
				element = DataElement.LoadElement(stream, TransferSyntax, this, SkipDataReadTags);
				// END
				bytesInDatasetReached = false;
				IDicomStreamMember o = (IDicomStreamMember)this;
				//go to root of the stream members
				while (true)
				{
					if (o is Sequence)
					{
						if (stream.Position >= (((Sequence)o).StreamPosition + ((Sequence)o).bytesInDataset))
							bytesInDatasetReached = true;
					}
					if (!(o.getParent() is IDicomStreamMember))
						break;
					o = (IDicomStreamMember)o.getParent();
				}

				isTrailingPadding = element.Tag.Equals("(0000,0000)");
				if (!isTrailingPadding)
				{
					Add(element);
				}
				else
				{
					break;
				}
				//Break if endcondition tag is reached 
				if (Sequence.EndConditionTag != null && element.Tag.Equals(Sequence.EndConditionTag))
					break;
			} while (!element.Tag.Equals(SequenceDelimiterTag) &&
					 stream.Position < stream.Length && !bytesInDatasetReached);
			bytesInDataset = int.MaxValue;
		}

		/// <summary>
		///     Writes a DICOM stream member instance to a specified
		///     DICOM output stream.
		/// </summary>      
		public void SaveTo(Stream stream)
		{
			foreach (DataElement d in itemList)
				d.SaveTo(stream);
		}

		/// <summary>
		/// Converts tree node into sequence recursively
		/// </summary>
		/// <param name="sequence">sequence</param>
		/// <returns>sequence</returns>
		private Sequence TreeNodeToSequence(NestedDataSet sequence)
		{
			Sequence result = new Sequence();
			for (int i = 0; i < sequence.Count; i++)
			//foreach (DataElement element in sequence)
			{
				DataElement element = sequence[i];

				//if (element.Tag.Element.Equals("0000")) continue;
				result.Add(element);
				foreach (object value in element.Value)
				{
					if (value is Sequence)
					{
						result.Add(TreeNodeToSequence((Sequence)value));
						if ((value as Sequence).Count > 0 && (value as Sequence)[0].Tag.Equals("(FFFE,E000)"))
							result.Add(new DataElement("(FFFE,E0DD)", ""));
					}
					else if (value is NestedDataSet)
					{
						result.Add(TreeNodeToSequence((NestedDataSet)value));
						if ((value as Sequence).Count > 0 && (value as Sequence)[0].Tag.Equals("(FFFE,E000)"))
							result.Add(new DataElement("(FFFE,E0DD)", ""));
					}
				}
			}
			return result;
		}

		/// <summary>
		/// Converts tree node into sequence recursively
		/// </summary>
		/// <param name="sequence">sequence</param>
		/// <returns>sequence</returns>
		private Sequence TreeNodeToSequence(Sequence sequence)
		{
			Sequence result = new Sequence();
			for (int i = 0; i < sequence.Count; i++)
			{
				DataElement element = sequence[i];
				result.Add(element);
				foreach (object value in element.Value)
				{
					if (value is Sequence)
					{
						result.Add(TreeNodeToSequence((Sequence)value));
						if ((value as Sequence).Count > 0 && (value as Sequence)[0].Tag.Equals("(FFFE,E000)"))
							result.Add(new DataElement("(FFFE,E0DD)", ""));
					}
					else if (value is NestedDataSet)
					{
						result.Add(TreeNodeToSequence((NestedDataSet)value));
						if ((value as Sequence).Count > 0 && (value as Sequence)[0].Tag.Equals("(FFFE,E000)"))
							result.Add(new DataElement("(FFFE,E0DD)", ""));
					}
				}
			}
			return result;
		}
		/// <summary>
		///     Returns all sequences of all levels of a sequence tree as
		///     one concatenated zero-level sequence. Multiple tacs elements
		///     of equal instances are allowed within a DICOM sequence.
		/// </summary>
		public virtual Sequence GetJointSubsequences()
		{
			Sequence s = TreeNodeToSequence(this);
			return s;
		}

		/// <summary>
		///     Clears all DICOM sequence properties.
		/// </summary>
		public virtual void Clear()
		{
			itemList.Clear();
		}

		/// <summary>
		///     Sorts all tacs elements of a sequence instance using
		///     <see cref="ArrayList.Sort()" />. This method is normally supposed
		///     not to be used, except by derivated classes that
		///     use unique tacs element identifiers like DICOM tacs set.
		/// </summary>
		public virtual void Sort()
		{
			itemList.Sort();
		}

		/// <summary>
		///     Returns all containing tacs elements as array of
		///     <see cref="DataElement" />.
		/// </summary>
		public DataElement[] ToArray()
		{
			System.Collections.Generic.List<DataElement> e = new System.Collections.Generic.List<DataElement>();
			foreach (object o in itemList)
			{
				e.Add((DataElement)o);
			}
			return e.ToArray();
		}

		/// <summary>
		///     Needed by the C# foreach-statement. Makes life easier.
		/// </summary>
		public IEnumerator GetEnumerator()
		{
			return itemList.GetEnumerator();
		}
	}
}
