/******************************************************************************

  Copyright (c) 2007-2013 by Turku PET Centre

  Library:      dftres
  Description:  Utility functions for setting up results structure based on DFT.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/

  Modification history:
  2007-08-09 Vesa Oikonen
    First created.
  2007-08-13 VO
    Uses new unit enums and functions.
  2007-09-13 VO
    Writes datanr in new field with that name instead of datarange. 
  2010-12-12 VO
    Fields titleline and unitline is RES struct are depracated
    and substituted by array fields parname[] and parunit[].
  2012-02-28 VO
    New defines in dft.h are used.
  2013-04-17 VO
    libtpcmisc.h included.
  2013-05-13 VO
    res_allocate_with_dft() initiates fA and E.



******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/dft.h"
#include "include/dftio.h"
#include "include/dftunit.h"
#include "include/dftres.h"
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for regional results based on information in DFT.
\return Returns 0 if successful, otherwise <>0.
 */
int res_allocate_with_dft(
  /** Pointer to initiated RES struct which will be allocated here and filled
   *  with ROI names etc */
  RES *res,
  /** Regional data from where necessary information is read */
  DFT *dft
) {
  int ri;

  // Check the input data
  if(res==NULL || dft==NULL || dft->voiNr<1) return 1;
  // Allocate memory
  if(resSetmem(res, dft->voiNr)!=0) return 2;
  res->voiNr=dft->voiNr;
  // Copy header information
  strcpy(res->studynr, dft->studynr);
  res->Vb=-1.0;
  res->fA=-1.0;
  res->E=-1.0;
  res->time=time(NULL); // Set current time to results
  res->isweight=dft->isweight;
  /* Copy region names, etc */
  for(ri=0; ri<dft->voiNr; ri++) {
    strcpy(res->voi[ri].name, dft->voi[ri].name);
    strcpy(res->voi[ri].voiname, dft->voi[ri].voiname);
    strcpy(res->voi[ri].hemisphere, dft->voi[ri].hemisphere);
    strcpy(res->voi[ri].place, dft->voi[ri].place);
  }
  /* Set data range */
  if(dft->timetype==DFT_TIME_STARTEND)
    sprintf(res->datarange, "%g - %g %s",
      dft->x1[0], dft->x2[dft->frameNr-1], petTunit(dft->timeunit) );
  else
    sprintf(res->datarange, "%g - %g %s",
      dft->x[0], dft->x[dft->frameNr-1], petTunit(dft->timeunit) );
  res->datanr=dft->frameNr;

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy the contents (both header and data) of DFT struct into RES struct.
\return Returns 0 if successful, in case of an error >0, and <0 if warning is
  suggested.
 */
int dftToResult(
  /** Regional data from where necessary information is read */
  DFT *dft,
  /** Pointer to initiated RES struct which will be allocated here */
  RES *res,
  /** Pointer to a string (allocated for at least 64 chars) where error message
      or other execution status will be written; enter NULL, if not needed */     
  char *status
) {
  int ri, fi;

  // Check the input data
  if(status!=NULL) sprintf(status, "program error");
  if(res==NULL || dft==NULL || dft->voiNr<1 || dft->frameNr<1) return 1;
  // Allocate memory and copy most of headers
  if(res_allocate_with_dft(res, dft) != 0) {
    if(status!=NULL) sprintf(status, "cannot setup results data");
    return 2;
  }
  res->parNr=dft->frameNr; if(res->parNr>MAX_RESPARAMS) {
    sprintf(status, "only %d frames can be copied to results", MAX_RESPARAMS);
    res->parNr=MAX_RESPARAMS;
  }
  // Set parameter titles and units
  for(fi=0; fi<res->parNr; fi++) {
    sprintf(res->parname[fi], "%d", fi+1);
    strcpy(res->parunit[fi], dft->unit);
  }
  // Copy regional values
  for(ri=0; ri<dft->voiNr; ri++) for(fi=0; fi<res->parNr; fi++)
    res->voi[ri].parameter[fi]=dft->voi[ri].y[fi];

  if(dft->frameNr>MAX_RESPARAMS) return -1;
  /* Set also deprecated parameter name and unit representations, for now */
  resFixParnames(res);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
