/******************************************************************************

  Copyright (c) 2009-2013 by Turku PET Centre

  Library:      fitres
  Description:  Utility functions for working with FIT struct
                (see also mathfunc.c).

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/

  Modification history:
  2009-07-28 Vesa Oikonen
    First created.
  2010-10-07 VO
    fitToResult(): sample nr (datanr) is copied from FIT to RES, if it is
    equal in all fits.
  2010-12-12 VO
    Fields titleline and unitline is RES struct are depracated
    and substituted by array fields parname[] and parunit[].
  2013-04-17 VO
    libtpcmisc.h included.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/
#include "include/dft.h"
#include "include/mathfunc.h"
#include "include/result.h"
#include "include/fitres.h"
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for regional function fits based on information in DFT.
\return Returns 0 if successful, otherwise <>0.
 */
int fit_allocate_with_dft(
  /** Pointer to initiated FIT struct which will be allocated here and filled
   *  with ROI names etc */
  FIT *fit,
  /** Regional data from where necessary information is read */
  DFT *dft
) {
  int ri;

  // Check the input data
  if(fit==NULL || dft==NULL || dft->voiNr<1) return 1;
  // Allocate memory
  if(fitSetmem(fit, dft->voiNr)!=0) return 2;
  fit->voiNr=dft->voiNr;
  /* Set header contents */
  fit->time=time(NULL); // Set current time to results
  strcpy(fit->unit, dft->unit);
  fit->timeunit=dft->timeunit;
  /* Copy region names, etc */
  for(ri=0; ri<dft->voiNr; ri++) {
    strcpy(fit->voi[ri].name, dft->voi[ri].name);
    strcpy(fit->voi[ri].voiname, dft->voi[ri].voiname);
    strcpy(fit->voi[ri].hemisphere, dft->voi[ri].hemisphere);
    strcpy(fit->voi[ri].place, dft->voi[ri].place);
    fit->voi[ri].dataNr=dft->frameNr;
    if(dft->timetype==DFT_TIME_STARTEND) {
      fit->voi[ri].start=dft->x1[0]; fit->voi[ri].end=dft->x2[dft->frameNr-1];
    } else {
      fit->voi[ri].start=dft->x[0]; fit->voi[ri].end=dft->x[dft->frameNr-1];
    }
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Conversion of FIT contents to RES.
\return Returns 0 when successful.
 */
int fitToResult(
  /** Pointer to FIT structure, contents of which are written to RES struct */
  FIT *fit,
  /** Pointer to initiated RES struct where FIT contents are written;
   *  any previous contenst are removed. */
  RES *res,
  /** Pointer to a string (allocated for at least 64 chars) where error message
   *  or other execution status will be written; enter NULL, if not needed */
  char *status
) {
  int pi, ri, maxParNr, ret;


  /* Check input */
  if(status!=NULL) sprintf(status, "invalid data");
  if(fit==NULL || res==NULL) return 1;
  if(fit->voiNr<1) return 2;

  /* Determine max parameter number in fits */
  for(ri=0, maxParNr=0; ri<fit->voiNr; ri++)
    if(fit->voi[ri].parNr>maxParNr) maxParNr=fit->voi[ri].parNr;
  //printf("maxParNr := %d\n", maxParNr);

  /* Allocate memory for results */
  resEmpty(res); ret=resSetmem(res, fit->voiNr); if(ret) {
    if(status!=NULL) sprintf(status, "cannot allocate memory"); return 11;}
  /* Copy titles & filenames */
  if(strlen(fit->program)>0) sprintf(res->program, "%s (c) 2009", fit->program);
  else strcpy(res->program, "fitToResult (c) 2009");
  strcpy(res->datafile, fit->datafile);
  res->time=fit->time;
  /* Copy region names, etc */
  res->voiNr=fit->voiNr;
  for(ri=0; ri<fit->voiNr; ri++) {
    strcpy(res->voi[ri].name, fit->voi[ri].name);
    strcpy(res->voi[ri].voiname, fit->voi[ri].voiname);
    strcpy(res->voi[ri].hemisphere, fit->voi[ri].hemisphere);
    strcpy(res->voi[ri].place, fit->voi[ri].place);
  }
  /* Copy sample number, if equal in all TACs */
  for(ri=1, ret=0; ri<fit->voiNr; ri++)
    if(fit->voi[ri].dataNr!=fit->voi[0].dataNr) ret++;
  if(ret==0) res->datanr=fit->voi[0].dataNr;
  /* Set parameter names */
  res->parNr=maxParNr+2; // function id and wss too
  strcpy(res->parname[0], "Func");
  for(pi=0; pi<maxParNr; pi++) sprintf(res->parname[pi+1], "p%d", pi+1);
  strcpy(res->parname[pi+1], "WSS");
  /* Copy parameter values */
  for(ri=0; ri<fit->voiNr; ri++) {
    res->voi[ri].parameter[0]=fit->voi[ri].type; // function id
    for(pi=0; pi<maxParNr; pi++) { // function parameters
      if(pi>=fit->voi[ri].parNr) res->voi[ri].parameter[pi+1]=0.0;
      else res->voi[ri].parameter[pi+1]=fit->voi[ri].p[pi];
    }
    res->voi[ri].parameter[pi+1]=fit->voi[ri].wss; // wss
  }
  /* Set also deprecated parameter name and unit representations, for now */
  resFixParnames(res);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
