/******************************************************************************
  Copyright (c) 2012 by Turku PET Centre

  File:        circle.c
  Description: Contains functions for simulating circular image region.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2012-12-12 VO
    First created.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "include/idi.h"
/*****************************************************************************/

/*****************************************************************************/
/** Simulate a mask image of circle with specified radius. The applied method
 *  is only approximate at pixel borders (pixel is divided into 5x5 subpixels).
\return Returns 0 if successful.
 */
int imgCircleMask(
  /** Pointer to allocated static or dynamic image; image must contain pixel
   *  sizes and dimensions; mask values are added to pixel values, thus you may
   *  need to set pixel values to zero before calling this function. */
  IMG *img,
  /** Plane index [0..dimz-1] */
  int zi,
  /** X distance of circle center (mm) from the upper left corner of the image 
   */
  double cx,
  /** Y distance of circle center (mm) from the upper left corner of the image 
   */
  double cy, 
  /** Radius of circle (mm) */
  double r,
  /** Mask value; this value is added to each pixel value that fits inside
   *  the radius; the pixels that are partially inside the radius will get
   *  fraction of mask value. */
  double mv,
  /** Pointer to value where the sum of added mask pixel values is written;
   *  enter NULL if not needed. Use this to validate the results. */
  double *smv, 
  /** Verbose level; set to <=0 to prevent all prints to stdout */
  int verbose
) {
  if(verbose>0) printf("imgCircleMask(img, %d, %g, %g, %g, %g, ...)\n",
     zi, cx, cy, r, mv);

  int n, xi, yi, fi, i, j;
  double dx[5], dy[5], r2, v;

  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(zi<0 || zi>=img->dimz) return(2);
  if(img->sizey<=0.0 || img->sizex<=0.0) return(3);
  if(img->sizey!=img->sizex) return(4);
  if(img->dimt<1) return(5);

  if(smv!=NULL) *smv=0.0;
  /* Compare r^2 to sum of squared distances instead of square root */
  r2=r*r;
  for(yi=0; yi<img->dimy; yi++) {
    dy[0]=(0.1+(double)yi)*img->sizey - cy;
    for(j=1; j<5; j++) dy[j]=dy[j-1]+0.2*img->sizey;
    for(xi=0; xi<img->dimx; xi++) {
      dx[0]=(0.1+(double)xi)*img->sizex - cx;
      for(i=1; i<5; i++) dx[i]=dx[i-1]+0.2*img->sizex;
      for(i=0, n=0; i<5; i++) for(j=0; j<5; j++)
        if((dx[i]*dx[i]+dy[j]*dy[j])<r2) n++;
      if(n==0) continue;
      v=(double)n*mv/25.0;
      for(fi=0; fi<img->dimt; fi++) img->m[zi][yi][xi][fi]+=v;
      if(smv!=NULL) *smv+=v;
    }
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulate a mask image of ring with specified inner and outer radius.
 *  The applied method is only approximate at pixel borders
 *  (pixel is divided into 5x5 subpixels).
\return Returns 0 if successful.
 */
int imgRingMask(
  /** Pointer to allocated static or dynamic image; image must contain pixel
   *  sizes and dimensions; mask values are added to pixel values, thus you may
   *  need to set pixel values to zero before calling this function. */
  IMG *img,
  /** Plane index [0..dimz-1] */
  int zi,
  /** X distance of circle center (mm) from the upper left corner of the image 
   */
  double cx,
  /** Y distance of circle center (mm) from the upper left corner of the image 
   */
  double cy, 
  /** Inner radius of circle (mm) */
  double r1,
  /** Outer radius of circle (mm) */
  double r2,
  /** Mask value; this value is added to each pixel value that fits inside
   *  the radius; the pixels that are partially inside the radius will get
   *  fraction of mask value. */
  double mv,
  /** Pointer to value where the sum of added mask pixel values is written;
   *  enter NULL if not needed. Use this to validate the results. */
  double *smv, 
  /** Verbose level; set to <=0 to prevent all prints to stdout */
  int verbose
) {
  int ret;
  double mv1, mv2, d;

  if(verbose>0) printf("imgRingMask(img, %d, %g, %g, %g, %g, %g, ...)\n",
     zi, cx, cy, r1, r2, mv);
  if(r1>=r2 || r1<0.0) {
    if(verbose>0) fprintf(stderr, "Error: invalid radius.\n");
    return(1);
  }

  /* Simulate circle with the larger radius */
  ret=imgCircleMask(img, zi, cx, cy, r2, mv, &mv1, verbose);
  if(ret!=0) {
    if(verbose>0) fprintf(stderr, "Error: cannot simulate circle 1.\n");
    return(ret);
  }
  /* Subtract circle with the smaller radius */
  ret=imgCircleMask(img, zi, cx, cy, r1, -mv, &mv2, verbose);
  if(ret!=0) {
    if(verbose>0) fprintf(stderr, "Error: cannot simulate circle 2.\n");
    return(ret);
  }
  /* Calculate the sum of mask pixel values */
  d=mv1+mv2; if(smv!=NULL) *smv=d;
  if(d==0.0 && verbose>0)
    fprintf(stderr, "Warning: empty mask image created.\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
