/******************************************************************************
  Copyright (c) 2012,2013 by Turku PET Centre

  File:        heartcor.c
  Description: Contains the functions for correcting spill-over and recovery
               errors in myocardial LV muscle and cavity.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Version history:
  2012-10-22 VO
    First created, code copied from separate program heartcor.
  2013-02-27 VO
    Change in heartRecoverySpilloverCorrectionFactors(): previously, vascular
    volume fraction in tissue was assumed to be 0.1 (Henze et al., 1983),
    but from now on Va is given as argument to this function.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "include/idi.h"
/*****************************************************************************/

/*****************************************************************************/
int heartcor(
  int verbose
) {
  if(verbose>0) printf("heartcor()\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate recovery and spillover correction coefficients based on
    Henze E, Huang S-C, Ratib O, Hoffman E, Phelps ME, Schelbert HR.
    Measurements of regional tissue and blood-pool radiotracer concentrations
    from serial tomographic images of the heart. J Nucl Med. 1983;24:987-996.
\return Returns 0 if successful, or <> 0 in case of an error.
 */
int heartRecoverySpilloverCorrectionFactors(
  /** radius of cavity and circular ROI (mm) */
  double R,
  /** thickness of myocardium (mm) */
  double d,
  /** spatial resolution (mm); s = FWHM/(2*SQRT(2*LN(2))) */
  double s,
  /** Vascular volume fraction in myocardium; Henze et al assumed 0.1 */
  double Vb,
  /** Pointer to resulting correction coefficient FMM, see the article */
  double *FMM,
  /** Pointer to resulting correction coefficient FMB, see the article */
  double *FMB,
  /** Pointer to resulting correction coefficient FBM, see the article */
  double *FBM,
  /** Pointer to resulting correction coefficient FBB, see the article */
  double *FBB
) {
  double hp;

  if(R<=0.0 || d<=0.0 || s<=0.0) return(1);
  if(FMM==NULL || FMB==NULL || FBM==NULL || FBB==NULL) return(2);
  if(Vb<0.0 || Vb>=1.0) return(3);
  hp = exp(-R*R/(2.0*s*s));
  *FMB = hp - exp(-(R+d)*(R+d)/(2.0*s*s));
  *FMM = erf( (d/2.0) / (M_SQRT2*s) );
  *FBB = 1.0 - hp;
  *FBM = Vb + 0.5*(1.0 - *FMM);  //0.6 - 0.5 * *FMM;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
