/*******************************************************************************
  Copyright (c) 2012 by Turku PET Centre
   
  File: libtpcidi.c
   
  Purpose: methods for testing functions in the library libtpcidi,
  and for printing out library information, such as Readme,
  History and build information. 
   
*******************************************************************************/

/******************************************************************************/
/** Library name */
#define LIB_NAME "libtpcidi"
/******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
/******************************************************************************/
#include "imgio.h"
#include "imgproc.h"
#include "libtpcmisc.h"
/******************************************************************************/
#include "include/idi.h"
#include "include/libtpcidiv.h"
/******************************************************************************/
/** Serial numbers for function names. */
#define ALLTESTS 0
#define HEARTCOR1 1
#define IDISIMULATETUBEVOL1 2
#define IMGCIRCLEMASK1 3
#define IMGRINGMASK1 4
/******************************************************************************/

/******************************************************************************/
// Test function declarations:
int test_heartcor1();
int test_idiSimulateTubeVol1();
int test_imgCircleMask1();
int test_imgRingMask1();
/******************************************************************************/

/******************************************************************************/
/** 
    Function for printing the usage information of the test program.
*/
void print_usage();

/******************************************************************************/
// Verbose mode switch:
int VERBOSE = 0;

/******************************************************************************/

/******************************************************************************/
int main (int argc, char *argv[ ])
{
  int c;
  int errflg=0, functionflag=-1, exit_code=0;
  extern char *optarg;
  extern int optind, optopt;
    
  while ((c = getopt(argc, argv, "vVbBrRhHf:F:")) != -1) {
    switch (c) {

    case 'V':
    case 'v':
      // Set verbose flag on:
      VERBOSE = 1;
      break;
    case 'B':
    case 'b':
      // Print out the build information:
      libtpcidi_print_build(stdout);
      return(0);
    case 'r':
      // Print out the Readme message:
      libtpcidi_print_readme(stdout);
      return(0);
    case 'R':
      // Print out the Readme message with Doxygen tags:
      libtpcidi_print_dreadme(stdout);
      return(0);
    case 'h':
      // Print out the History message:
      libtpcidi_print_history(stdout);
      return(0);
    case 'H':
      // Print out the History message with Doxygen tags:
      libtpcidi_print_dhistory(stdout);
      return(0);
    case 'f':
    case 'F':
      // Set function serial number to be tested:
      functionflag = atoi(optarg);
      break;
    case ':': // -f or -z without argument
      fprintf(stderr, "Option -%c requires an argument\n", optopt);
      errflg++;
      break;
    case '?':
      fprintf(stderr, "Unrecognised option: -%c\n", optopt);
      errflg++;
    }
  }// End parsing command line options...

  if(errflg) {print_usage(); return(2);}

  if(functionflag<0) {print_usage(); return(0);}

  libtpcidi_print_build(stdout);

  // Choose function(s) to test:
  switch(functionflag){
    case ALLTESTS:
      exit_code = test_heartcor1();
      if(exit_code) exit(exit_code);
      exit_code = test_idiSimulateTubeVol1();
      if(exit_code) exit(exit_code);
      exit_code = test_imgCircleMask1();
      if(exit_code) exit(exit_code);
      exit_code = test_imgRingMask1();
      if(exit_code) exit(exit_code);
      break;
    case HEARTCOR1:
      exit_code = test_heartcor1();
      break;
    case IDISIMULATETUBEVOL1:
      exit_code = test_idiSimulateTubeVol1();
      break;
    case IMGCIRCLEMASK1:
      exit_code = test_imgCircleMask1();
      break;
    case IMGRINGMASK1:
      exit_code = test_imgRingMask1();
      break;
    default:
      break;
  }
        
  exit(exit_code);
}
/******************************************************************************/

/******************************************************************************/
int test_heartcor1()
{
  int error_code = 0;

  printf("\n=====================================\n");
  printf("\nTesting heartcor #1...\n");
  printf("\n=====================================\n");

  
  if(error_code)
    printf("\n Test FAILED: test_heartcor1 failed with error code: %i\n",
           error_code);
  else
    printf("\n Test SUCCESFULL: test_heartcor1 exited with: %i\n",
           error_code); 

  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_idiSimulateTubeVol1()
{
  char *funcname="idiSimulateTubeVol1";
  int error_code = 0;

  VOL vol;
  int ret;

  printf("\n=====================================\n");
  printf("\nTesting %s #1...\n", funcname);
  printf("\n=====================================\n");

  volInit(&vol);
  ret=volAllocate(&vol, 5, 10, 10);
  if(ret!=0) return(-1);

  /* Error, if pixel sizes are missing */
  vol.sizex=vol.sizey=vol.sizez=0.0;
  ret=idiSimulateTubeVol(&vol, 0, 5.0, 5.0, 2.0, 2.0, 0.0, 10.0);
  if(ret==0) error_code+=1;
  
  vol.sizex=vol.sizey=vol.sizez=1.0;

  /* Must be able to provide correct image even with extreme parameters */
  // all zero
  ret=idiSimulateTubeVol(&vol, 0, 5.0, 5.0, 2.0, 2.0, 0.0, 0.0);
  if(ret!=0) error_code+=10;
  // all the same >0
  ret=idiSimulateTubeVol(&vol, 1, 5.0, 5.0, 2.0, 2.0, 10.0, 10.0);
  if(ret!=0) error_code+=10;
  // vessel covers whole volume
  ret=idiSimulateTubeVol(&vol, 2, 5.0, 5.0, 100.0, 2.0, 0.0, 10.0);
  if(ret!=0) error_code+=10;
  // vessel is outside volume
  ret=idiSimulateTubeVol(&vol, 3, 50.0, 50.0, 2.0, 2.0, 1.0, 10.0);
  if(ret!=0) error_code+=10;
  // FWHM is very high
  ret=idiSimulateTubeVol(&vol, 4, 5.0, 5.0, 2.0, 1.0E+3, 1.0, 10.0);
  if(ret!=0) error_code+=10;
  /* check results */
  if(error_code==0) {
    float avg;
    VOL_RANGE r;
    r.x1=r.y1=1; r.x2=r.y2=vol.dimx;

    r.z1=r.z2=1; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg!=0.0) error_code+=10;

    r.z1=r.z2=2; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg!=10.0) error_code+=10;

    r.z1=r.z2=3; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg!=10.0) error_code+=10;

    r.z1=r.z2=4; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg!=1.0) error_code+=10;

    r.z1=r.z2=5; volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    if(avg<=1.0 || avg>2.0) error_code+=10;
  }

  /* Correct image with reasonable parameters */
  // vessel > bkg
  ret=idiSimulateTubeVol(&vol, 0, 5.0, 5.0, 2.0, 2.0, 10.0, 20.0);
  if(ret!=0) error_code+=100;
  // vessel < bkg
  ret=idiSimulateTubeVol(&vol, 1, 5.0, 5.0, 2.0, 2.0, 20.0, 10.0);
  if(ret!=0) error_code+=100;
  // small FWHM and high radius will give full peak value
  ret=idiSimulateTubeVol(&vol, 2, 5.0, 5.0, 3.0, 1.0, 5.0, 10.0);
  if(ret!=0) error_code+=100;
  // vessel is not at center
  ret=idiSimulateTubeVol(&vol, 3, 1.0, 10.0, 2.0, 2.0, 1.0, 10.0);
  if(ret!=0) error_code+=100;
  // peak value is not the same as given by RC formula (Germano et al)
  // but somewhat higher
  ret=idiSimulateTubeVol(&vol, 4, 5.0, 5.0, 2.0, 3.0, 0.0, 1.0);
  if(ret!=0) error_code+=100;
  /* check results */
  if(error_code==0) {
    float avg, max, min;
    VOL_RANGE r;
    VOL_PIXEL maxp, minp;
    r.x1=r.y1=1; r.x2=r.y2=vol.dimx;

    r.z1=r.z2=1;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    if(max<=10.0 || max>=20.0 || avg<=10) error_code+=100;
    if(maxp.x!=5 || maxp.y!=5) error_code+=100;

    r.z1=r.z2=2;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    if(max>20.0 || avg<=10.0 || avg>=20.0) error_code+=100;
    if(minp.x!=5 || minp.y!=5) error_code+=100;

    r.z1=r.z2=3;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    if(max<9.99999) error_code+=100;

    r.z1=r.z2=4;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    if(maxp.x!=1 || maxp.y!=10) error_code+=100;

    r.z1=r.z2=5;
    volAvg(&vol, &r, &avg); if(VERBOSE) printf("avg := %E\n", avg);
    volMax(&vol, &r, &maxp, &max, &minp, &min);
    if(VERBOSE) printf("max := %E at (%d,%d,%d)\n", max, maxp.x, maxp.y, maxp.z);
    if(VERBOSE) printf("min := %E at (%d,%d,%d)\n", min, minp.x, minp.y, minp.z);
    double rc, rad=2.0, fwhm=3.0, s;
    s=fwhm/2.355; rc=1.0-exp(-rad*rad/(2.0*s*s));
    if(VERBOSE) printf("rc := %g\n", rc);
    if(max<rc || max>=1.0) error_code+=100;
  }

  volEmpty(&vol);

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code); 

  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_imgCircleMask1()
{
  char *funcname="imgCircleMask1";
  int error_code = 0;

  IMG img;
  int ret, i;
  double mv1, mv2, a, d;
  float f;

  printf("\n=====================================\n");
  printf("\nTesting %s #1...\n", funcname);
  printf("\n=====================================\n");


  /* Allocate memory for test image with odd dimensions */
  imgInit(&img); ret=imgAllocate(&img, 2, 5, 5, 1);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot allocate memory for image.\n");
    return(1000+ret);
  }

  /* Simulate circle at the centre of image */
  {
  if(VERBOSE) printf("simulation 1\n");
  img.sizex=img.sizey=img.sizez=20.0;
  float correct[]={
    0,   0,   0,   0,   0,
    0,  52, 100,  52,   0,
    0, 100, 100, 100,   0,
    0,  52, 100,  52,   0,
    0,   0,   0,   0,   0,
    0,   0,   0,   0,   0,
    0,  24,  72,  24,   0,
    0,  72, 100,  72,   0,
    0,  24,  72,  24,   0,
    0,   0,   0,   0,   0
  };
  ret=imgCircleMask(&img, 0, 50.0, 50.0, 30.0, 100.0, &mv1, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 50.0, 50.0, 25.0, 100.0, &mv2, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgCircleMask().\n", ret);
    imgEmpty(&img); return(1100+ret);
  }
  f=100;
  tiffWriteImg(&img, -1, -1, &f, PET_GRAYSCALE, "test_imgCircleMask1a.tif",
               0, 0, 0, NULL);
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) {
    d=fabs(img.pixel[i]-correct[i]);
    if(d>0.0000001) error_code++;
    //printf("%3g, ", img.pixel[i]); if((i+1)%5==0) printf("\n");
    //printf(" %g vs %g\n", img.pixel[i], correct[i]);
  }
  a=100.0*M_PI*1.5*1.5; d=fabs(mv1-a)/a;
  if(VERBOSE) printf("  mv1 := %g vs %g (%g%%)\n", mv1, a, 100.*d);
  if(d>0.05) error_code++;
  a=100.0*M_PI*1.25*1.25; d=fabs(mv2-a)/a;
  if(VERBOSE) printf("  mv2 := %g vs %g (%g%%)\n", mv2, a, 100.*d);
  if(d>0.05) error_code++;

  if(error_code) printf("\nsimulated image 1 is not correct.\n");
  }

  /* Simulate circle NOT at the centre of image */
  if(error_code==0) {

  if(VERBOSE) printf("simulation 2\n");
  img.sizex=img.sizey=img.sizez=20.0;
  float correct[]={
    0,   0,   0,   0,   0,
    0,  36,  36,   0,   0,
   36, 100, 100,  36,   0,
   36, 100, 100,  36,   0,
    0,  36,  36,   0,   0,
    0,   0,  24,  72,  24,
    0,   0,  72, 100,  72,
    0,   0,  24,  72,  24,
    0,   0,   0,   0,   0,
    0,   0,   0,   0,   0
  };
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) img.pixel[i]=0;
  ret=imgCircleMask(&img, 0, 40.0, 60.0, 30.0, 100.0, &mv1, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 70.0, 30.0, 25.0, 100.0, &mv2, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgCircleMask().\n", ret);
    imgEmpty(&img); return(1200+ret);
  }
  f=100;
  tiffWriteImg(&img, -1, -1, &f, PET_GRAYSCALE, "test_imgCircleMask1b.tif",
               0, 0, 0, NULL);
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) {
    d=fabs(img.pixel[i]-correct[i]);
    if(d>0.0000001) error_code++;
    //printf("%3g, ", img.pixel[i]); if((i+1)%5==0) printf("\n");
    //printf(" %g vs %g\n", img.pixel[i], correct[i]);
  }
  a=100.0*M_PI*1.5*1.5; d=fabs(mv1-a)/a;
  if(VERBOSE) printf("  mv1 := %g vs %g (%g%%)\n", mv1, a, 100.*d);
  if(d>0.05) error_code++;
  a=100.0*M_PI*1.25*1.25; d=fabs(mv2-a)/a;
  if(VERBOSE) printf("  mv2 := %g vs %g (%g%%)\n", mv2, a, 100.*d);
  if(d>0.05) error_code++;

  if(error_code) printf("\nsimulated image 2 is not correct.\n");
  }



  /* Allocate memory for large test image with even dimensions */
  imgEmpty(&img); ret=imgAllocate(&img, 2, 100, 100, 1);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot allocate memory for image.\n");
    return(1000+ret);
  }

  /* Simulate circle */
  if(error_code==0) {
  if(VERBOSE) printf("simulation 3\n");
  img.sizex=img.sizey=img.sizez=1.0;
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) img.pixel[i]=0;
  ret=imgCircleMask(&img, 0, 50.0, 50.0, 40.0, 100.0, &mv1, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 70.0, 30.0, 18.3, 100.0, &mv2, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgCircleMask().\n", ret);
    imgEmpty(&img); return(1300+ret);
  }
  f=100;
  tiffWriteImg(&img, -1, -1, &f, PET_GRAYSCALE, "test_imgCircleMask1c.tif",
               0, 0, 0, NULL);
  a=100.0*M_PI*40.0*40.0; d=fabs(mv1-a)/a;
  if(VERBOSE) printf("  mv1 := %g vs %g (%g%%)\n", mv1, a, 100.*d);
  if(d>0.001) error_code++;
  a=100.0*M_PI*18.3*18.3; d=fabs(mv2-a)/a;
  if(VERBOSE) printf("  mv2 := %g vs %g (%g%%)\n", mv2, a, 100.*d);
  if(d>0.001) error_code++;

  if(error_code) printf("\nsimulated image 3 is not correct.\n");
  }


  /* Simulate a ring */
  if(error_code==0) {
  if(VERBOSE) printf("simulation 4\n");
  img.sizex=img.sizey=img.sizez=1.0;
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) img.pixel[i]=0;
  ret=imgCircleMask(&img, 0, 50.0, 50.0, 40.0, 100.0, &mv1, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 0, 50.0, 50.0, 30.0, -100.0, &mv2, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 70.0, 30.0, 18.3, 100.0, &mv2, VERBOSE);
  if(ret==0) ret=imgCircleMask(&img, 1, 70.0, 30.0, 17.3, -100.0, &mv2, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgCircleMask().\n", ret);
    imgEmpty(&img); return(1300+ret);
  }
  f=100;
  tiffWriteImg(&img, -1, -1, &f, PET_GRAYSCALE, "test_imgCircleMask1d.tif",
               0, 0, 0, NULL);
  }


  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code); 

  imgEmpty(&img);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
int test_imgRingMask1()
{
  char *funcname="imgRingMask1";
  int error_code = 0;

  IMG img;
  int ret, i;
  double mv, a, d;
  float f;

  printf("\n=====================================\n");
  printf("\nTesting %s #1...\n", funcname);
  printf("\n=====================================\n");


  /* Allocate memory for test image */
  imgInit(&img); ret=imgAllocate(&img, 1, 200, 200, 1);
  if(ret!=0) {
    fprintf(stderr, "Error: cannot allocate memory for image.\n");
    return(1000+ret);
  }

  /* Simulate ring */
  {
  if(VERBOSE) printf("simulation 1\n");
  img.sizex=img.sizey=img.sizez=1.0;
  ret=imgRingMask(&img, 0, 95.0, 105.0, 75.0, 85.0, 1.0, &mv, VERBOSE);
  if(ret!=0) {
    fprintf(stderr, "Error %d from imgRingMask().\n", ret);
    imgEmpty(&img); return(1100+ret);
  }
  f=1;
  tiffWriteImg(&img, -1, -1, &f, PET_RAINBOW, "test_imgRingMask1a.tif",
               0, 0, 0, NULL);
  for(i=0; i<img.dimz*img.dimx*img.dimy; i++) {
#if(0)
    d=fabs(img.pixel[i]-correct[i]);
    if(d>0.0000001) error_code++;
    printf("%3g, ", img.pixel[i]); if((i+1)%10==0) printf("\n");
#endif
  }

  a=M_PI*(85.0*85.0 - 75.0*75.0); d=fabs(mv-a)/a;
  if(VERBOSE) printf("  mv := %g vs %g (%g%%)\n", mv, a, 100.*d);
  if(d>0.001) error_code++;

  if(error_code) printf("\nsimulated image 1 is not correct.\n");
  }

  if(error_code)
    printf("\n Test FAILED: %s failed with error code: %i\n",
           funcname, error_code);
  else
    printf("\n Test SUCCESFULL: %s exited with: %i\n",
           funcname, error_code); 

  imgEmpty(&img);
  return(error_code);
}
/******************************************************************************/

/******************************************************************************/
void print_usage(){

  libtpcidi_print_build(stdout);
  puts("\n"
  "  Methods for testing functions in the library "LIB_NAME"\n"
  "  and for printing out associated information, such as Readme,\n"
  "  History and build information. \n"
  "\n"
  "  Usage: "LIB_NAME" [-Options]\n"
  "\n"
  "  Options:\n"
  "  -h | H \n"
  "      Print out the "LIB_NAME" History message. Include Doxygen style tags\n"
  "      by using option '-H'.\n"
  "  -r | R \n"
  "      Print out the "LIB_NAME" Readme message. Include Doxygen style tags\n"
  "      by using option '-R'.\n"
  "  -b\n"
  "      Print out the "LIB_NAME" build information.\n"
  "  -f <function serial number>\n"
  "      Runs test procedures for the functions corresponding to given\n"
  "      'function serial number'. Serial numbers are defined in file\n"
  "      "LIB_NAME".c; enter 0 to execute them all.\n"
  "  -v\n"
  "      Run in verbose mode.\n"
  "\n"
  "  E.g.: "LIB_NAME" -h\n"
  "        "LIB_NAME" -v -f 1 \n"
  "\n"
  );
  fflush(stdout);
}
/******************************************************************************/

/******************************************************************************/
